/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelPromoteByTypenameRegistration.h
 *    Registration interface for channel types to enable promotion
 *    from void to typed based on runtime knowledge of the typename.
 *
 *    Keeping this in a separate file from the basic mechanism in
 *    ChannelPromoteByTypename.h helps reduce dependencies (rebuild
 *    time). It might make less sense to keep them separate once
 *    this registration is used more or less ubiquitously. :)
 *    
 *
 * @author Christof Schröter
 * @date   2018/08/26
 */

#ifndef _MIRA_CHANNEL_PROMOTE_BY_TYPENAME_REGISTRATION_H_
#define _MIRA_CHANNEL_PROMOTE_BY_TYPENAME_REGISTRATION_H_

#include <utils/UniqueIdentifier.h>

#include <fw/Channel.h>
#include <fw/ChannelPromoteByTypename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

#define COMMA ,

/**
 * A template class for specialized channel promoters.
 * PromoteTo is the type that this promoter can promote a void chanel to.
 * Using a second template parameter ExtraDiscriminator allows to make
 * different promoter classes registered for the same type but not clashing
 * (at library/manifest load time), e.g. registered in independent
 * source files or libraries.
 * This can make code management much easier - at the cost of allowing
 * definition of many redundant promoters. Runtime effect should be negligible
 * though, as these are just indexed by the ClassFactory and only one of the
 * right PromoteTo type is actually instantiated when the first void channel
 * with the respective typename is published.
 */
template <typename PromoteTo, typename ExtraDiscriminator>
class TypedChannelPromoter : public TypedChannelPromoterBase
{
	MIRA_OBJECT(TypedChannelPromoter<PromoteTo COMMA ExtraDiscriminator>)
public:
	/// Promotes the channel by calling channel_cast<PromoteTo>(channel)
	void promote(ConcreteChannel<void>* channel)
	{
		channel_cast<PromoteTo>(channel);
	}

	/// Returns mira::typeName<PromoteTo>()
	std::string typeName()
	{
		return mira::typeName<PromoteTo>();
	};
};

#undef COMMA

#ifdef MIRA_DISABLE_CHANNELPROMOTERS
	// #define macro empty to entirely disable registration
	#define MIRA_REGISTER_CHANNELTYPE(channeltype)
#else
	#ifndef MIRA_UNIQUE_IDENTIFIER_ENABLED
		#pragma message("UniqueIdentifier not enabled. "                      \
		                "MIRA_REGISTER_CHANNELTYPE disabled.")
		#define MIRA_REGISTER_CHANNELTYPE(channeltype)
	#else
		// semi-unique identifier __project_path_to_file_line__
		#define MIRA_REGISTER_CHANNELTYPE_UNIQUE_NAME                         \
			MIRA_FILE_LINE_IDENTIFIER

		// Define a unique class and use it as extra template parameter
		// for the promoter specialization to register.
		// Using the source file name and line as part of the resulting type name
		// gives the bonus advantage that its origin can easily be found in the code.
		#define MIRA_REGISTER_CHANNELTYPE(channeltype)                        \
		class MIRA_REGISTER_CHANNELTYPE_UNIQUE_NAME{};                        \
		MIRA_VARTEMPLATE_CLASS_REGISTER(                                      \
			(mira), TypedChannelPromoter, 2,                              \
			(channeltype, MIRA_REGISTER_CHANNELTYPE_UNIQUE_NAME),         \
			mira::TypedChannelPromoterBase)
	#endif
#endif

///@endcond

///////////////////////////////////////////////////////////////////////////////

}

#endif
