/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file IntPublisher.C
 *
 * @author Tim Langner
 * @date   2010/09/16
 */

#include <fw/Unit.h>

namespace mira { namespace fw {

///////////////////////////////////////////////////////////////////////////////

/// Publishes int channel
class IntPublisherUnit : public Unit
{
	/// Register at class factory
	MIRA_OBJECT(IntPublisherUnit)
public:

	/// Tell base class to call process in a 500 millisecond interval
	IntPublisherUnit() :
		Unit(Duration::milliseconds(500))
	{
		mCounter = 0;
		mPublish = true;
	}

	/// Reflect method for serialization and services
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		// call base class reflect
		Unit::reflect(r);
		// publish interface ICountSetable
		r.interface("ICountSetable");
		// publish our setCounter message
		r.method("setCounter", &IntPublisherUnit::setCounter, this,
		         "set the internal counter to the specified value, return previous value",
		         "value", "value to set", 5);
		r.method("setCounterAndPublish", &IntPublisherUnit::setCounterAndPublish, this,
		         "set the internal counter to the specified value and publish "
		         "new value immediately, return previous value",
		         "value", "value to set", 5);
		r.method("enablePublish", &IntPublisherUnit::enablePublish, this,
		         "start/stop publishing, return previous state",
		         "publish", "whether to enable publishing", true);
	}

	/// Overwritten from MicroUnit. Gets called once on startup
	virtual void initialize()
	{
		// publish int channel and store it in a member
		mIntChannel = publish<int>("Int");
		// publish our service methods (setCounter)
		publishService(*this);
	}

	/// Overwritten from Unit. Gets called every 500 milliseconds
	virtual void process(const Timer& timer)
	{
		if (!mPublish)
			return;

		// get write proxy object from int channel
		ChannelWrite<int> value = mIntChannel.write();
		// and write our incremented counter to it
		value->value() = mCounter++;
	}

private:

	int setCounter(int value)
	{
		MIRA_LOG(NOTICE) << "IntPublisherUnit: setCounter called: " << value;
		int oldCount = mCounter;
		mCounter = value;
		return oldCount;
	}

	int setCounterAndPublish(int value)
	{
		MIRA_LOG(NOTICE) << "IntPublisherUnit: setCounterAndPublish called: " << value;
		int oldCount = mCounter;
		mCounter = value;
		ChannelWrite<int> w = mIntChannel.write();
		w->value() = mCounter++;
		return oldCount;
	}

	bool enablePublish(bool publish)
	{
		MIRA_LOG(NOTICE) << "IntPublisherUnit: enablePublish called: " << publish;
		bool oldPublish = mPublish;
		mPublish = publish;
		return oldPublish;
	}

protected:
	Channel<int> mIntChannel;
	int mCounter;
	bool mPublish;
};

///////////////////////////////////////////////////////////////////////////////

}}

/// Tell class factory that we are derived from Unit
MIRA_CLASS_SERIALIZATION( mira::fw::IntPublisherUnit, mira::Unit );
