/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file YawPitchRollTest.C
 *    Description.
 *
 * @author Erik Einhorn, Christoph Weinrich
 * @date   2011/10/16
 */


#include <iostream>

#include <boost/test/unit_test.hpp>
#include <boost/test/floating_point_comparison.hpp>

#include <math/YawPitchRoll.h>
#include <platform/Platform.h>
#include <serialization/adapters/boost/tuple.hpp>
#include <serialization/Print.h>

#include <Eigen/Dense>

using namespace mira;
using namespace Eigen;

///////////////////////////////////////////////////////////////////////////////

template <typename T>
inline boost::tuples::tuple<T,T,T> alternativeQuaternionToYawPitchRoll(const Eigen::Quaternion<T>& q)
{
	float w = q.w();
	float x = q.x();
	float y = q.y();
	float z = q.z();
	float ww = w*w;
	float xx = x*x;
	float yy = y*y;
	float zz = z*z;

	float yaw   = atan2(2*(x*y + w*z), ww + xx - yy - zz);
	float pitch = asin(-2*(x*z - w*y));
	float roll  = atan2(2*(y*z + w*x), ww - xx - yy + zz);

	return boost::make_tuple(yaw,pitch,roll);
}


#define BOOST_CHECK_VECTOR3_CLOSE(a,b,tol)    \
	BOOST_CHECK(std::abs(a.x()-b.x()) < tol); \
	BOOST_CHECK(std::abs(a.y()-b.y()) < tol); \
	BOOST_CHECK(std::abs(a.z()-b.z()) < tol);

#define BOOST_CHECK_QUATERNION_CLOSE(a,b,tol) \
	BOOST_CHECK(std::abs(a.w()-b.w()) < tol || std::abs(-a.w()-b.w()) < tol); \
	BOOST_CHECK(std::abs(a.x()-b.x()) < tol || std::abs(-a.x()-b.x()) < tol); \
	BOOST_CHECK(std::abs(a.y()-b.y()) < tol || std::abs(-a.y()-b.y()) < tol); \
	BOOST_CHECK(std::abs(a.z()-b.z()) < tol || std::abs(-a.z()-b.z()) < tol);


BOOST_AUTO_TEST_CASE( YawPitchRollTest )
{
	Quaternionf q = quaternionFromYawPitchRoll<float>(half_pi<float>(),
	                                                  quarter_pi<float>(),
	                                                  quarter_pi<float>());
	std::cout << "q: " << print(q) << std::endl;

	float s = sin(quarter_pi<float>());

	Vector3f x = q*Vector3f::UnitX();
	std::cout << "x: " << print(x) << std::endl;
	BOOST_CHECK_VECTOR3_CLOSE(x, Vector3f(0,s,-s), 0.01f);

	Vector3f y = q*Vector3f::UnitY();
	std::cout << "y: " << print(y) << std::endl;
	BOOST_CHECK_VECTOR3_CLOSE(y, Vector3f(-s,0.5,0.5), 0.01f);

	Vector3f z = q*Vector3f::UnitZ();
	std::cout << "z: " << print(z) << std::endl;
	BOOST_CHECK_VECTOR3_CLOSE(z, Vector3f(s,0.5,0.5), 0.01f);

	auto ypr = alternativeQuaternionToYawPitchRoll(q);
	std::cout << print(ypr) << std::endl;

	std::cout << print(q.toRotationMatrix()) << std::endl;


	std::cout << print(matrixFromYawPitchRoll(half_pi<float>(),
											  quarter_pi<float>(),
											  quarter_pi<float>()))
	          << std::endl;

	float r = two_pi<float>();
	// delta for increasing angles (dividing through a power of 2, makes sure to "catch" all special cases)
	float dr = r / 8.0f;

	for(float yaw=-r;yaw<=r;yaw+=dr)
	for(float pitch=-r;pitch<=r;pitch+=dr)
	for(float roll=-r;roll<=r;roll+=dr)
	{
		Quaternionf q = quaternionFromYawPitchRoll<float>(yaw,pitch,roll);

		auto ypra = quaternionToYawPitchRoll(q);
		Quaternionf q2 = quaternionFromYawPitchRoll<float>(ypra);
		BOOST_CHECK_QUATERNION_CLOSE(q,q2, 0.01f);
		//std::cout << print(q.coeffs()) << std::endl;
		//std::cout << print(q2.coeffs()) << std::endl;

		//auto yprb = alternativeQuaternionToYawPitchRoll(q);
		//std::cout << print(yprb) << std::endl;
		//Quaternionf q3 = quaternionFromYawPitchRoll<float>(yprb);
		//BOOST_CHECK_QUATERNION_CLOSE(q,q3, 0.01f);
		//std::cout << print(q.coeffs()) << std::endl;
		//std::cout << print(q3.coeffs()) << std::endl;

	}

}

BOOST_AUTO_TEST_CASE( QuaternionCovToYawPitchRollCovTest )
{
	Eigen::Matrix<float,6,6> yprCovSet;
	//yprCovSet = Eigen::Matrix<float,6,6>::Identity() * 100.0;

	yprCovSet << 1, 2, 3, 4, 5, 6,
			1, 2, 3, 4, 5, 6,
			1, 2, 3, 4, 5, 6,
			1, 2, 3, 4, 5, 6,
			1, 2, 3, 4, 5, 6,
			1, 2, 3, 4, 5, 6;

	Eigen::Matrix<float,7,7> quaternionCov;

	Eigen::Quaternionf orientation;

	// max value is 70 because 70 / 100.0 = 0.7 and
	// 0.7 ^2 < 0.5
	for(uint32 i = 0; i <= 70; i += 1){
		double y = (((uint32)(i / 100.0 * 1000) % 1000 ) / 1000.0);
		double w = sqrt(0.5 - y*y);
		orientation = Eigen::Quaternionf(w, -w, y, -y);
		std::cout<<"w = "<<w<<", x = "<<-w<<", y = "<<y<<", z = "<<-y<<std::endl;
		std::cout<<"orientation = "<<std::endl<<print(orientation)<<std::endl;

		quaternionCov = quaternionCovFromYawPitchRollCov(yprCovSet, orientation);
		Eigen::Matrix<float,6,6> yprCov = quaternionCovToYawPitchRollCov(quaternionCov, orientation);
		std::cout<<"yprCov = "<<std::endl<<print(yprCov)<<std::endl;
		Eigen::Matrix<float, 6,6> diff = (yprCovSet - yprCov);
		//float diffSum = (Eigen::Matrix<float, 6,6>(yprCovSet - yprCov)).abs().sum();
		float diffSum = diff.array().abs().sum();
		std::cout<<"yprCovDiff.sum() = "<<diffSum<<std::endl;
		BOOST_CHECK_CLOSE(diffSum + 1.0, 1.0, 0.01f);
	}
}

///////////////////////////////////////////////////////////////////////////////

