/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ClassConstructionTest.h
 *    $Try to create objects and check if casting is correct$.
 *
 * @author Ronny Stricker
 * @date   2010/10/25
 */

#include <boost/test/unit_test.hpp>
#include "classes/DiamondClassStructure.h"

using namespace std;

namespace mira {

//////////////////////////////////////////////////////////////////////////////

//#############################################################################
//##  ClassCreation Test
//#############################################################################

#define TEST_CREATION(pCLASS)                                                \
	tObject = ClassFactory::newInstance<pCLASS>( #pCLASS );                  \
	if ( !tObject || tObject->getClass().getIdentifier() != #pCLASS ) {      \
		throw string("Error creating object ")+string(#pCLASS)+string("!");  \
	}                                                                        \
	else {                                                                   \
		delete tObject;                                                      \
	}

BOOST_AUTO_TEST_CASE( ClassCreation )
{
	// Try to create every single class
	Object* tObject = NULL;
	
	TEST_CREATION( mira::A )
	TEST_CREATION( mira::B )
	TEST_CREATION( mira::C )
	TEST_CREATION( mira::D )
	TEST_CREATION( mira::E )
	TEST_CREATION( mira::F )
	TEST_CREATION( mira::G )
	TEST_CREATION( mira::H )
	TEST_CREATION( mira::I )

	// Creation should fail (class X unknown)
	BOOST_CHECK_THROW(
			ClassFactory::newInstance<Object>( "mira::X" ), mira::XInvalidParameter );

	// Should throw BAD_CAST (B is not a base of A)
	BOOST_CHECK_THROW(
			ClassFactory::newInstance<B>( A::CLASS().getIdentifier() ) , mira::XBadCast );

}

//#############################################################################
//##  ClassCreation Cast Test
//#############################################################################

BOOST_AUTO_TEST_CASE( ClassCreationCast )
{
	// obtain class I with pointer to A
	A* tObject = NULL;
	tObject = ClassFactory::newInstance<A>( I::CLASS().getIdentifier() );
	BOOST_CHECK( tObject != NULL );
	delete tObject;

	// obtaining class F with pointer to A should throw
	BOOST_CHECK_THROW(
			ClassFactory::newInstance<A>( F::CLASS().getIdentifier() ),
			XBadCast);

	// obtain class I wihout casting -> should return Object
	tObject = A::CLASS().newInstance( I::CLASS().getIdentifier() );
	BOOST_CHECK( tObject != NULL );
	delete tObject;

	Object* tObject2 = NULL;

	// obtain class A using stored pointer to class object (without casting)
	ClassProxy tClass = ClassFactory::getClassByIdentifier( A::CLASS().getIdentifier() );
	tObject2 = tClass.newInstance();
	BOOST_CHECK( tObject2 != NULL );
	delete tObject2;

	// obtain class A using stored pointer to class object (with casting)
	tObject = tClass.newInstance<A>();
	BOOST_CHECK( tObject != NULL );
	delete tObject;

	tObject = tClass.newInstance<C>( C::CLASS().getIdentifier() );
	BOOST_CHECK( tObject != NULL );
	delete tObject;
}

//////////////////////////////////////////////////////////////////////////////

} // namespace
