/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TypedVoidPtr.h
 *    Type safe handling of typed void pointers.
 *
 * @author Erik Einhorn
 * @date   2011/09/29
 */

#ifndef _MIRA_TYPEDVOIDPTR_H_
#define _MIRA_TYPEDVOIDPTR_H_

#include <error/Exceptions.h>
#include <factory/TypeId.h>
#include <platform/Typename.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Class that allows to maintain type-safety when passing void pointers.
 * Sometimes, it is necessary to pass a void* pointer through an interface,
 * while maintaining type safety. This can be achieved using this class.
 * Assume you want to pass different objects of different types through the
 * following single interface, which must be the same for all types and objects
 * (e.g. when you cannot use templates)
 * \code
 * struct Interface {
 *     virtual void foo(TypedVoidPtr ptr) = 0;
 * };
 * \endcode
 * Now assume, you have different classes that implement this interface for
 * different types, then you can use the TypeVoidPtr class as follows:
 * \code
 * struct IntHandler : public Interface {
 *     virtual void foo(TypedVoidPtr ptr) {
 *         int* p = ptr; // this cast is safe and will do full type checking!
 *         std::cout << "Integer is: " << *p << std::endl;
 *     }
 * };
 *
 * struct StringHandler : public Interface {
 *     virtual void foo(TypedVoidPtr ptr) {
 *         std::string* s = ptr; // this cast is safe and will do full type checking!
 *         std::cout << "String is: " << *s << std::endl;
 *     }
 * };
 *
 * Interface* i1 = new IntHandler;
 * Interface* i2 = new StringHandler;
 *
 * int val = 123;
 * i1->foo(&val);  // foo will internally cast back to int
 *
 * std::string s = "Hello world";
 * i2->foo(&s);  // foo will internally cast back to string
 *
 * i2->foo(&val); // will cause exception, since int cannot be casted into 
 *                // string in StringHandler::foo()
 *
 * \endcode
 */
class TypedVoidPtr
{

public:

	/// Creates nullptr
	TypedVoidPtr() : mType(-1), mPtr(NULL) {}

	/// Creates a typed void pointer from the given pointer
	template <typename T>
	TypedVoidPtr(T* ptr) :
		mType(typeId<T>()), mPtr(ptr) {}

public:

	/// Return if underlying pointer is NULL
	bool isNull() const { return mPtr==NULL; }

public:

	/** 
	 * Safely casts to T*.
	 * @throw XBadCast, if the stored void pointer is not of type T.
	 */
	template <typename T>
	operator T*() {	return cast<T>(); }

	/** 
	 * Safely casts to const T*.
	 * @throw XBadCast, if the stored void pointer is not of type T.
	 */
	template <typename T>
	operator const T*() const { return cast<T>(); }

	/**
	 * Safely casts the object pointer that is stored to T*. 
	 * @throw XBadCast, if the stored void pointer is not of type T.
	 */
	template <typename T>
	T* cast() {
		if(mType!=typeId<T>())
			MIRA_THROW(XBadCast, "Cannot cast to type '" << typeName<T>() << "'");
		// cast is safe due to our above type id check
		return static_cast<T*>(mPtr);
	}

	/**
	 * Safely casts the object pointer that is stored to const T*. 
	 * @throw XBadCast, if the stored void pointer is not of type T.
	 */
	template <typename T>
	const T* cast() const {
		TypedVoidPtr* This = const_cast<TypedVoidPtr*>(this);
		return This->cast<T>();
	}

private:

	/// The unique type id of the stored pointer
	TypeId mType;

	/// The pointer itself
	void* mPtr;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Same as TypedVoidPtr but const.
 */
class TypedVoidConstPtr
{

public:

	/// Creates nullptr
	TypedVoidConstPtr() : mType(-1), mPtr(NULL) {}

	/// Creates a typed void pointer from the given pointer
	template <typename T>
	TypedVoidConstPtr(const T* ptr) :
		mType(typeId<T>()), mPtr(ptr) {}

public:

	/// Return if underlying pointer is NULL
	bool isNull() const { return mPtr==NULL; }

public:

	/**
	 * Safely casts to const T*, 
	 * @throw XBadCast, if the stored void pointer is not of type T.
	 */
	template <typename T>
	operator const T*() const { return cast<T>(); }

	/**
	 * Safely casts the object pointer that is stored to const T*.
	 * @throw XBadCast, if the stored void pointer is not of type T.
	 */
	template <typename T>
	const T* cast() const {
		if(mType!=typeId<T>())
			MIRA_THROW(XBadCast, "Cannot cast to type '" << typeName<T>() << "'");
		// cast is safe due to our above type name check
		return static_cast<const T*>(mPtr);
	}

private:

	/// The unique type id of the stored pointer
	TypeId mType;

	/// The pointer itself
	const void* mPtr;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif
