/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ScopedAccess.h
 *    Grants thread-safe access to an object that should be protected from
 *    concurrent access.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2012/11/06
 */

#ifndef _MIRA_SCOPEDACCESS_H_
#define _MIRA_SCOPEDACCESS_H_

#ifndef Q_MOC_RUN
#include <boost/noncopyable.hpp>
#include <boost/thread/mutex.hpp>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Grants thread-safe access to an object (the Protectee) that should be
 * protected from concurrent access.
 * The protectee can return a ScopedAccess object to the user. The ScopedAccess
 * object contains a reference to the protectee and will delegate all calls
 * to it. Moreover, it will automatically take care of locking and unlocking,
 * i.e. it will release the lock when the ScopedAccess object goes out of
 * scope.
 *
 * The Protectee must meet the following concept:
 * <code>
 * concept Protectee
 * {
 *     // locks the object from concurrent access (blocks, if there is an access already).
 *     void lock();
 *
 *     // unlocks the object
 *     void unlock();
 * }
 * </code>
 * Internally, this behavior usually is implemented using a mutex in the Protectee.
 * For most use cases it would be sufficient to use the ProtecteeMixin class.
 */
template <typename Protectee>
class ScopedAccess : boost::noncopyable
{

public:

	/**
	 * Usually called by the Protectee to return an access object to itself,
	 * passing itself as pointer. The second parameter specifies, if the
	 * object is already locked. If not (the default) it is locked immediately.
	 */
	ScopedAccess(Protectee* p, bool lockedAlready = false) : mObject(p) {
		assert(mObject!=NULL);
		if(!lockedAlready)
			mObject->lock();
	}

	~ScopedAccess() {
		// unlock the object when we go out of scope
		mObject->unlock();
	}

	ScopedAccess(ScopedAccess&& other)
	{
		std::swap(mObject,other.mObject);
	}

	ScopedAccess& operator=(ScopedAccess&& other)
	{
		std::swap(mObject,other.mObject);
		return *this;
	}

public:

	Protectee* operator->() {
		return mObject;
	}

	Protectee& operator*() {
		return *mObject;
	}

protected:
	Protectee* mObject;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Mixin class that can be used to add the Protectee concept used by
 * ScopedAccess to an existing class. A typical use case would be a class with
 * a member that should be accessed by concurrent threads.
 *
 * \code
 * class MyClass
 * {
 * public:
 *     typedef ProtecteeMixin<std::string> ProtectedString;
 *
 *     ScopedAccess<ProtectedString> getString() {
 *         return ScopedAccess<ProtectedString>(&mMyProtectedMember);
 *     }
 *
 * private:
 *     ProtectedString mMyProtectedMember;
 * };
 * \endcode
 */
template <typename T>
class ProtecteeMixin : public T, boost::noncopyable
{
public:

	void lock() {
		mMutex.lock();
	}

	void unlock() {
		mMutex.unlock();
	}

private:

	boost::mutex mMutex;
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
