/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Endian.h
 *    Utilities for byte order conversion between little and big endian.
 *
 * @author Tim Langner
 * @date   2010/06/28
 */

#ifndef _MIRA_ENDIAN_H_
#define _MIRA_ENDIAN_H_

#include <type_traits>
#ifndef Q_MOC_RUN
#include <boost/detail/endian.hpp>
#endif

#include <platform/Types.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Enum that specifies the endianness of the host system.
 */
enum Endian
{
	LITTLE_ENDIAN_ORDER,
	BIG_ENDIAN_ORDER,
	NETWORK_ENDIAN_ORDER = BIG_ENDIAN_ORDER,
#if defined(BOOST_LITTLE_ENDIAN)
	HOST_ENDIAN_ORDER = LITTLE_ENDIAN_ORDER
#elif defined(BOOST_BIG_ENDIAN)
	HOST_ENDIAN_ORDER = BIG_ENDIAN_ORDER
#else
# error "Impossible to determine the endianness of the system"
#endif
};

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL
namespace Private
{

/**
 * This function swaps byte given a char pointer and a size.
 * The default implementation raises an assertion
 */
template<int32 size>
inline void swapBytes(const char* iSrc, char* oDest)
{
	static_assert(size!=size, "swapBytes is not specialized for this size");
	// size!=size forces the static assert to depend on template parameter
	// otherwise the assert will always fail.
}

/**
 * Specialization for 1-byte types.
 */
template<>
inline void swapBytes<1>(const char* iSrc, char* oDest)
{
	*oDest = *iSrc;
}

/**
 * Specialization for 2-byte types.
 */
template<>
inline void swapBytes<2>(const char* iSrc, char* oDest)
{
	// Use bit manipulations instead of accessing individual bytes from memory.
	// Much faster.
	uint16* dest = reinterpret_cast<uint16*>(oDest);
	const uint16* src = reinterpret_cast<const uint16*>(iSrc);
	*dest = (*src >> 8) | (*src << 8);
}

/**
 * Specialization for 4-byte types.
 */
template<>
inline void swapBytes<4>(const char* iSrc, char* oDest)
{
	// Use bit manipulations instead of accessing individual bytes from memory.
	// Much faster.
	uint32* dest = reinterpret_cast<uint32*>(oDest);
	const uint32* src = reinterpret_cast<const uint32*>(iSrc);
	*dest = (*src >> 24) | 
	        ((*src & 0x00ff0000) >> 8) |
	        ((*src & 0x0000ff00) << 8) |
	        (*src << 24);
}

/**
 * Specialization for 8-byte types.
 */
template<>
inline void swapBytes<8>(const char* iSrc, char* oDest)
{
	// Use bit manipulations instead of accessing individual bytes from memory.
	// Much faster.
	uint64* dest = reinterpret_cast<uint64*>(oDest);
	const uint64* src = reinterpret_cast<const uint64*>(iSrc);
	*dest = (((*src & 0xff00000000000000ull) >> 56) |
	        ((*src & 0x00ff000000000000ull) >> 40) |
	        ((*src & 0x0000ff0000000000ull) >> 24) |
	        ((*src & 0x000000ff00000000ull) >> 8) |
	        ((*src & 0x00000000ff000000ull) << 8) |
	        ((*src & 0x0000000000ff0000ull) << 24) |
	        ((*src & 0x000000000000ff00ull) << 40) |
	        ((*src & 0x00000000000000ffull) << 56));
}

/**
 * Union representing a float in ieee754 floating point representation
 */
union IEEE754Float
{
	float f;

	struct
	{
#if defined(BOOST_BIG_ENDIAN)
		uint32 negative:1;
		uint32 exponent:8;
		uint32 mantissa:23;
#elif defined(BOOST_LITTLE_ENDIAN)
		uint32 mantissa :23;
		uint32 exponent :8;
		uint32 negative :1;
#endif
	} ieee;
};

/**
 * Union representing a double in ieee754 floating point representation
 */
union IEEE754Double
{
	double d;

	struct
	{
#if defined(BOOST_BIG_ENDIAN)
		uint32 negative:1;
		uint32 exponent:11;
		uint32 mantissa0:20;
		uint32 mantissa1:32;
#elif defined(BOOST_LITTLE_ENDIAN)
		uint32 mantissa1 :32;
		uint32 mantissa0 :20;
		uint32 exponent :11;
		uint32 negative :1;
#endif
	} ieee;
};


} /// namespace Private
///@endcond

///////////////////////////////////////////////////////////////////////////////


/**
 * Trait that specifies the conversion type of a given data type.
 */
template <typename T>
struct NetworkTypeTrait
{
	typedef T Type;
};

/**
 * Specialization for float.
 */
template <>
struct NetworkTypeTrait<float>
{
	typedef uint32 Type;
};

/**
 * Specialization for double.
 */
template <>
struct NetworkTypeTrait<double>
{
	typedef uint64 Type;
};


/**
 * Converts a value from host byte order to network byte order.
 * @param value The value to convert.
 * @return The value in network byte order
 */
template<typename T>
inline typename NetworkTypeTrait<T>::Type hostToNetwork(const T& value)
{
	static_assert(std::is_arithmetic<T>::value, 
	              "hostToNetwork accepts arithmetic types only");

#if defined(BOOST_BIG_ENDIAN)
		return value;
#else
	T r;
	Private::swapBytes<sizeof(T)>(reinterpret_cast<const char*>(&value), 
	                              reinterpret_cast<char*>(&r));
	return r;
#endif
}

/**
 * Specialization for float
 */
template<>
inline NetworkTypeTrait<float>::Type hostToNetwork<float>(const float& value)
{
#if defined(BOOST_BIG_ENDIAN)
	return *reinterpret_cast<const NetworkTypeTrait<float>::Type*>(&value);
#else
	uint32 r = 0;
	uint8* ptr = reinterpret_cast<uint8*>(&r);

	union Private::IEEE754Float f;
	f.f = value;

	ptr[0] = (f.ieee.negative << 7) | ((f.ieee.exponent >> 1) & 0x7F);
	ptr[1] = ((f.ieee.exponent & 0x01) << 7) | ((f.ieee.mantissa >> 16) & 0x7F);
	ptr[2] = (f.ieee.mantissa >> 8) & 0xFF;
	ptr[3] = (f.ieee.mantissa) & 0xFF;

	return r;
#endif
}

/**
 * Specialization for double
 */
template<>
inline NetworkTypeTrait<double>::Type hostToNetwork<double>(const double& value)
{
#if defined(BOOST_BIG_ENDIAN)
	return *reinterpret_cast<const NetworkTypeTrait<double>::Type*>(&value);
#else
	uint64 r = 0;
	uint8* ptr = reinterpret_cast<uint8*>(&r);

	union Private::IEEE754Double d;
	d.d = value;

	ptr[0] = (d.ieee.negative << 7) + ((d.ieee.exponent >> 4) & 0x7F);
	ptr[1] = (((d.ieee.exponent) << 4) & 0xF0) + ((d.ieee.mantissa0 >> 16) & 0x0F);
	ptr[2] = (d.ieee.mantissa0 >> 8) & 0xFF;
	ptr[3] = (d.ieee.mantissa0) & 0xFF;
	ptr[4] = (d.ieee.mantissa1 >> 24) & 0xFF;
	ptr[5] = (d.ieee.mantissa1 >> 16) & 0xFF;
	ptr[6] = (d.ieee.mantissa1 >> 8) & 0xFF;
	ptr[7] = (d.ieee.mantissa1) & 0xFF;

	return r;
#endif
}

/**
 * Converts a value from network byte order to host byte order.
 * @param value The value to convert.
 * @return The value in host byte order
 */
template<typename T>
inline T networkToHost(const typename NetworkTypeTrait<T>::Type& value)
{
	static_assert(std::is_arithmetic<T>::value, 
	              "networkToHost accepts arithmetic types only");

#if defined(BOOST_BIG_ENDIAN)
	return value;
#else
	T r;
	Private::swapBytes<sizeof(T)>(reinterpret_cast<const char*>(&value), 
	                              reinterpret_cast<char*>(&r));
	return r;
#endif
}

/**
 * Specialization for float
 */
template<>
inline float networkToHost(const NetworkTypeTrait<float>::Type& value)
{
#if defined(BOOST_BIG_ENDIAN)
	return *reinterpret_cast<const float*>(&value);
#else
	const uint8* ptr = reinterpret_cast<const uint8*>(&value);

	union Private::IEEE754Float f;
	f.ieee.negative = ptr[0] >> 7;
	f.ieee.exponent = ((ptr[0] << 1) & 0xFE) + ((ptr[1] >> 7) & 0x01);
	f.ieee.mantissa = (ptr[3] & 0xFF) + ((ptr[2] << 8) & 0xFF00) +
	                  (((ptr[1] & 0x7F) << 16) & 0xFF0000);

	return f.f;
#endif
}

/**
 * Specialization for double
 */
template<>
inline double networkToHost(const NetworkTypeTrait<double>::Type& value)
{
#if defined(BOOST_BIG_ENDIAN)
	return *reinterpret_cast<const double*>(&value);
#else
	const uint8* ptr = reinterpret_cast<const uint8*>(&value);

	union Private::IEEE754Double d;
	d.ieee.negative = ptr[0] >> 7;
	d.ieee.exponent = ((ptr[0] << 4) & 0x7F0) + ((ptr[1] >> 4) & 0x0F);
	d.ieee.mantissa0 = (ptr[3] & 0xFF) + ((ptr[2] << 8) & 0xFF00) +
	(((ptr[1] & 0x0F) << 16) & 0xFF0000);
	d.ieee.mantissa1 = (ptr[7] & 0xFF) + ((ptr[6] << 8) & 0xFF00) +
	((ptr[5] << 16) & 0xFF0000) + ((ptr[4] << 24) & 0xFF000000);

	return d.d;
#endif
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
