/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RandomGenerator.h
 *    Helper singleton to easily generate random generators using the
 *    boost::random distributions and generators.
 *
 * @author Erik Einhorn
 * @date   2012/11/10
 */

#ifndef _MIRA_RANDOMGENERATOR_H_
#define _MIRA_RANDOMGENERATOR_H_

#include <limits>

#ifndef Q_MOC_RUN
#include <boost/random/mersenne_twister.hpp>
#include <boost/random/variate_generator.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/random/uniform_int.hpp>
#endif

#include <platform/Types.h>
#include <utils/Time.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Template class to easily generate random generators using the
 * boost::random distributions and generators.
 *
 * The template takes two template parameters:
 * -# A random distribution, e.g. NormalRandomDistribution or the
 *   <a href="http://www.boost.org/doc/libs/release/doc/html/boost_random/reference.html#boost_random.reference.distributions">
 *   boost random distributions</a>
 * -# The random generator engine, e.g.
 * <a href="http://www.boost.org/doc/libs/release/doc/html/boost_random/reference.html#boost_random.reference.generators">
 *   boost random engines</a>, by default the Mersenne Twister (19937) is used.
 *
 *
 * Example:
 * \code
 * RandomGenerator<boost::uniform_01<>> rnd;
 *
 * for(int i=0; i<100; ++i)
 *     double random_number=rnd();
 *
 * \endcode
 *
 * There are also some predefined random generators:
 * - UniformRandomGenerator
 * - NormalRandomGenerator
 *
 * @ingroup MathModule
 */
template <typename Distribution, typename Engine=boost::mt19937>
class RandomGenerator
{
public:
	typedef typename Distribution::result_type result_type;

public:

	/**
	 * Generates the random generator with default parameters for
	 * the random distribution.
	 * The random generator is seeded with the default seed. Use the
	 * seed() method to set a different seed or to use the system time
	 * to generate a seed.
	 */
	RandomGenerator() :
	    mGenerator(mEngine, Distribution()) {
	}

	/**
	 * Generates the random generator with the parameters provided by
	 * the given distribution.
	 * The random generator is seeded with the default seed. Use the
	 * seed() method to set a different seed or to use the system time
	 * to generate a seed.
	 */
	RandomGenerator(const Distribution& dist) :
	    mGenerator(mEngine, dist) {
	}

	/**
	 * Copy constructor that copies the distribution but initializes the
	 * random generator engine from the scratch, since engines cannot be
	 * copied.
	 * Therfore, the copied generator has the same distribution including
	 * paramters etc, but produces a different random sequence unless it is
	 * called with the same seed.
	 */
	RandomGenerator(const RandomGenerator& other) :
	    mGenerator(mEngine, other.distribution()) {
	}

public:

	/**
	 * Assigns the given distribution (respectively its parameters)
	 * to this random generator.
	 */
	RandomGenerator& operator=(const Distribution& dist) {
		mGenerator.distribution() = dist;
		return *this;
	}


	/**
	 * Assignment operator that copies the distribution but not the
	 * random generator engine, since engines cannot be copied.
	 * Therfore, the copied generator has the same distribution including
	 * paramters etc, but produces a different random sequence unless it is
	 * called with the same seed.
	 */
	RandomGenerator& operator=(const RandomGenerator& other) {
		mGenerator.distribution() = other.distribution();
		return *this;
	}


public:

	/**
	 * Draws a sample from the random distribution.
	 */
	result_type operator()() {
		return mGenerator();
	}

public:

	/**
	 * Seeds the random generator using the current system time. Hence,
	 * after calling this method, the random generator will generate different
	 * sequences of random numbers each time.
	 */
	void seed() {
		seed((uint32)(Time::now()-Time::unixEpoch()).total_nanoseconds());
	}

	/**
	 * Seeds the random generator with the given seed. The random generator
	 * will produce the same sequence of random numbers if the same seed is
	 * used.
	 */
	void seed(uint32 value) {
		mEngine.seed(value);
	}

public:

	/**
	 * Provides direct access to the underlying random distribution.
	 */
	Distribution* operator->() {
		return &mGenerator.distribution();
	}

	/**
	 * Provides direct access to the underlying random distribution.
	 */
	const Distribution* operator->() const {
		return &mGenerator.distribution();
	}

	/**
	 * Provides direct access to the underlying random distribution.
	 */
	Distribution& distribution() {
		return mGenerator.distribution();
	}

	/**
	 * Provides direct access to the underlying random distribution.
	 */
	const Distribution& distribution() const {
		return mGenerator.distribution();
	}

public:

	/**
	 * Provides direct access to the underlying random generator engine.
	 */
	Engine& engine() {
		return mEngine;
	}

	/**
	 * Provides direct access to the underlying random generator engine.
	 */
	const Engine& engine() const {
		return mEngine;
	}

private:

	Engine mEngine;
	// we can remove the variate_generator for boost >= 1.47
	boost::variate_generator<Engine&, Distribution> mGenerator;
};

/**
 * Macro to be used in derived random generators to supply the default interface
 * such copy constructors and assignment operators and typedefs of
 * Base and Distribution
 */
#define MIRA_RANDOM_GENERATOR_COMMON(Derived, TDistribution)              \
	typedef RandomGenerator<TDistribution> Base;                          \
	typedef TDistribution Distribution;                                   \
	Derived(const Distribution& dist) : Base(dist) {}                     \
	Derived(const Derived& other) : Base(other) {}                        \
	Derived& operator=(const Distribution& dist) {                        \
		Base::operator=(dist);                                            \
		return *this;                                                     \
	}                                                                     \
	Derived& operator=(const Derived& other) {                            \
		Base::operator=(other);                                           \
		return *this;                                                     \
	}

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
