/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LogCustomizableFilter.h
 *    A customizable log filter.
 *
 * @author Tim Langner
 * @date   2010/07/08
 */

#ifndef _MIRA_LOGCUSTOMIZABLEFILTER_H_
#define _MIRA_LOGCUSTOMIZABLEFILTER_H_

#include <error/LoggingCore.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * A customizable log filter to match the users needs.
 * e.g.
 * @code
 * LogCustomizableFilter cf;
 * cf << "At "
 *    << LogCustomizableFilter::Severity<LogCustomizableFilter::LESS_EQUAL>(ERROR)
 *    << LogCustomizableFilter::Class<EQUAL>("MyClass");
 * @endcode
 * will only log messages that have a severity of CRITICAL or ERROR
 * and come from the class MyClass.
 * That way sinks can be configured to log only a special part of the log entries.
 * Custom filters can be combined by & and | operators.
 * An unlikely but possible thing to do would be having registered
 * a file sink for each namespace/class and sort the messages
 * from different namespaces/classes into different files.
 *
 * More filters can be added by deriving from LogCustomizableFilter::CustomFilter.
 * The new filter can be chained by adding it via the << operator to the LogCustomizableFilter
 * instance.
 *
 * @ingroup LoggingModule
 */
class LogCustomizableFilter : public LogFilterBase
{
public:
	enum CompareFlags
	{
		LESS = 0,
		LESS_EQUAL,
		EQUAL,
		GREATER_EQUAL,
		GREATER,
		NOT,
	};

	/**
	 * Abstract base class for custom filters.
	 */
	class CustomFilterBase
	{
	public:
		virtual ~CustomFilterBase() {}
		/**
		 * Called by LogCustomizableFilter whenever a new log entry is getting filtered.
		 * Return false to filter out the log entry.
		 * @param record The log entry
		 * @return false if filtered out, true if passed.
		 */
		virtual bool filter(const LogRecord& record) = 0;
	};

	/**
	 * Helper class to get the type info of the derived class.
	 * Derive from this class if you want to add your own custom filter.
	 */
	template <typename Derived>
	class CustomFilter : public CustomFilterBase {};

	/**
	 * Class filter, used to filter log entries with a given class name.
	 */
	template<LogCustomizableFilter::CompareFlags F = LogCustomizableFilter::EQUAL>
	class Class : public LogCustomizableFilter::CustomFilter<LogCustomizableFilter::Class<F> >
	{
	public:
		Class(const std::string& iClassName) :
			className(iClassName)
		{
		}

		virtual bool filter(const LogRecord& record)
		{
			if (F == LogCustomizableFilter::EQUAL)
				return MIRA_LOGGER.parseFunction(record).className == className;
			if (F == LogCustomizableFilter::NOT)
				return MIRA_LOGGER.parseFunction(record).className != className;
			return false;
		}

		std::string className;
	};

	/**
	 * Namespace filter, used to filter log entries with a given namespace.
	 */
	template<LogCustomizableFilter::CompareFlags F = LogCustomizableFilter::EQUAL>
	class Namespace : public LogCustomizableFilter::CustomFilter<LogCustomizableFilter::Namespace<F> >
	{
	public:
		Namespace(const std::string& iNS) :
			ns(iNS)
		{
		}

		virtual bool filter(const LogRecord& record)
		{
			if (F == LogCustomizableFilter::EQUAL)
				return MIRA_LOGGER.parseFunction(record).nameSpace == ns;
			if (F == LogCustomizableFilter::NOT)
				return MIRA_LOGGER.parseFunction(record).nameSpace != ns;
			return false;
		}

		std::string ns;
	};

	/**
	 * Severity filter, used to filter log entries with a given severity level.
	 */
	template<LogCustomizableFilter::CompareFlags F = LogCustomizableFilter::LESS_EQUAL>
	class Severity : public LogCustomizableFilter::CustomFilter<LogCustomizableFilter::Severity<F> >
	{
	public:
		Severity(SeverityLevel iLevel) :
			level(iLevel)
		{
		}

		virtual bool filter(const LogRecord& record)
		{
			if (F == LogCustomizableFilter::LESS)
				return record.level < level;
			if (F == LogCustomizableFilter::LESS_EQUAL)
				return record.level <= level;
			if (F == LogCustomizableFilter::EQUAL)
				return record.level == level;
			if (F == LogCustomizableFilter::GREATER_EQUAL)
				return record.level >= level;
			if (F == LogCustomizableFilter::GREATER)
				return record.level > level;
			if (F == LogCustomizableFilter::NOT)
				return record.level != level;
			return false;
		}

		SeverityLevel level;
	};

	/**
	 * ThreadID filter, used to filter log entries with a given thread id.
	 */
	template<LogCustomizableFilter::CompareFlags F = LogCustomizableFilter::EQUAL>
	class Thread : public LogCustomizableFilter::CustomFilter<LogCustomizableFilter::Thread<F> >
	{
	public:
		Thread(ThreadID iThreadID) :
			threadID(iThreadID)
		{
		}

		virtual bool filter(const LogRecord& record)
		{
			if (F == LogCustomizableFilter::EQUAL)
				return record.threadID == threadID;
			if (F == LogCustomizableFilter::NOT)
				return record.threadID != threadID;
			return false;
		}

		ThreadID threadID;
	};

	/**
	 * Operator class used to combine two custom filters by the & operator
	 */
	template <typename F1, typename F2>
	class AndOperator : public LogCustomizableFilter::CustomFilter<LogCustomizableFilter::AndOperator<F1, F2> >
	{
	public:
		AndOperator(const F1& f1, const F2& f2)
		{
			mF1.reset(new F1(f1));
			mF2.reset(new F2(f2));
		}

		virtual bool filter(const LogRecord& record)
		{
			return mF1->filter(record) && mF2->filter(record);
		}

	private:
		boost::shared_ptr<LogCustomizableFilter::CustomFilterBase> mF1;
		boost::shared_ptr<LogCustomizableFilter::CustomFilterBase> mF2;
	};

	/**
	 * Operator class used to combine two custom filters by the | operator
	 */
	template <typename F1, typename F2>
	class OrOperator : public LogCustomizableFilter::CustomFilter<LogCustomizableFilter::OrOperator<F1, F2> >
	{
	public:
		OrOperator(const F1& f1, const F2& f2)
		{
			mF1.reset(new F1(f1));
			mF2.reset(new F2(f2));
		}

		virtual bool filter(const LogRecord& record)
		{
			return mF1->filter(record) || mF2->filter(record);
		}

	private:
		boost::shared_ptr<LogCustomizableFilter::CustomFilterBase> mF1;
		boost::shared_ptr<LogCustomizableFilter::CustomFilterBase> mF2;
	};

	/**
	 * Add filters by chaining them with this operator.
	 * @param f The filter to add
	 * @return Reference to this.
	 */
	template<typename T>
	LogCustomizableFilter& operator<<(const T& f)
	{
		mFilter.push_back(boost::shared_ptr<
				LogCustomizableFilter::CustomFilterBase>(new T(f)));
		return *this;
	}


	virtual bool filter(const LogRecord& record)
	{
		for (size_t i = 0; i < mFilter.size(); ++i)
			if (!mFilter[i]->filter(record))
				return false;
		return true;
	}

protected:
	std::vector<boost::shared_ptr<LogCustomizableFilter::CustomFilterBase> > mFilter;
};

/**
 * Operator to combine filters by and
 * @ingroup LoggingModule
 */
template <typename Derived1, typename Derived2>
LogCustomizableFilter::AndOperator<Derived1, Derived2> operator&(const LogCustomizableFilter::CustomFilter<Derived1>& f1,
                                                                 const LogCustomizableFilter::CustomFilter<Derived2>& f2)
{
	const Derived1* t1 = (const Derived1*)&f1;
	const Derived2* t2 = (const Derived2*)&f2;
	return LogCustomizableFilter::AndOperator<Derived1, Derived2>(*t1, *t2);
}

/**
 * Operator to combine filters by or
 * @ingroup LoggingModule
 */
template <typename Derived1, typename Derived2>
LogCustomizableFilter::OrOperator<Derived1, Derived2> operator|(const LogCustomizableFilter::CustomFilter<Derived1>& f1,
                                                                const LogCustomizableFilter::CustomFilter<Derived2>& f2)
{
	const Derived1* t1 = (const Derived1*)&f1;
	const Derived2* t2 = (const Derived2*)&f2;
	return LogCustomizableFilter::OrOperator<Derived1, Derived2>(*t1, *t2);
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
