/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageParser.C
 *    Implementation of PackageParser.h.
 *
 * @author Erik Einhorn, Ronny Stricker
 * @date   2011/02/25
 */

#include "core/PackageParser.h"

#include <map>
#include <string>
#include <fstream>

#include <boost/spirit/include/qi.hpp>
#include <boost/algorithm/string/replace.hpp>
#include <boost/algorithm/string/trim.hpp>
#include <boost/algorithm/string/split.hpp>

#include <platform/Process.h>
#include <xml/XMLDom.h>

#include "core/Database.h"
#include "core/Package.h"
#include "core/Dependency.h"
#include "core/Repository.h"

namespace qi = boost::spirit::qi;
namespace ascii = boost::spirit::ascii;

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

string changeLogToHTML(std::string const& log)
{
	string tReturn = log;
	boost::algorithm::replace_all(tReturn, "&", "&amp;");
	boost::algorithm::replace_all(tReturn, "<", "&lt;");
	boost::algorithm::replace_all(tReturn, ">", "&gt;");
	boost::algorithm::replace_all(tReturn, "\n", "<br/>");
	return tReturn;
}

///////////////////////////////////////////////////////////////////////////////

template <typename Iterator>
struct PackageParser : qi::grammar<Iterator, ascii::space_type>
{
	PackageParser(Package& e) : PackageParser::base_type(start), entry(&e)
	{

		using qi::char_;
		using qi::lit;
		using qi::eol;
		using qi::eoi;
		using qi::eps;
		using ascii::no_case;
		using qi::string;
		using qi::lexeme;

		using boost::spirit::qi::phrase_parse;

		parameterText = lexeme[ +(char_ - ')') ];
		parameterText2 = lexeme[ +(char_ - '"') ];

		comment = lexeme['#' >> *(char_ - (eol))];

		commands = no_case[string("MIRA_REQUIRE_PACKAGE")]
		         | no_case[string("MIRA_OPTIONAL_PACKAGE")]
		         | no_case[string("MIRA_RUNTIME_PACKAGE")]
		         | no_case[string("MIRA_PACKAGE_TYPE")]
		         | no_case[string("MIRA_DEVELOPMENT_STATE")]
		         | no_case[string("MIRA_VERSION")]
		         | no_case[string("MIRA_BUILD_ID")]
		         | no_case[string("MIRA_PACKAGE_NAME")];

		quotedcommands = no_case[string("MIRA_DESCRIPTION")]
		               | no_case[string("MIRA_CHANGELOG")]
		               | no_case[string("MIRA_TAGS")]
		               | no_case[string("MIRA_AUTHOR")];

		call = (commands|quotedcommands)[boost::bind(&PackageParser::parseCommand, this, _1)]
		       >> '(' >> *qi::space
		       >> parameterText[boost::bind(&PackageParser::parseCommandParam, this, _1)]
		       >> *qi::space >> ')';

		call2 = quotedcommands[boost::bind(&PackageParser::parseCommand, this, _1)]
		        >> '('
		        >> *(*qi::space >> '"'
		        >> parameterText2[boost::bind(&PackageParser::parseCommandParam, this, _1)]
		        >> '"' >> *qi::space)
		        >> ')';

		rest = +(char_ - (commands|quotedcommands));

		start = *(comment | call2 | call | rest) >> eoi;
	}

	qi::rule<Iterator, ascii::space_type> start;
	qi::rule<Iterator, std::string(), ascii::space_type> comment;
	qi::rule<Iterator, std::string(), ascii::space_type> parameterText;
	qi::rule<Iterator, std::string(), ascii::space_type> parameterText2;
	qi::rule<Iterator, std::string(), ascii::space_type> commands;
	qi::rule<Iterator, std::string(), ascii::space_type> quotedcommands;
	qi::rule<Iterator, std::string(), ascii::space_type> call;
	qi::rule<Iterator, std::string(), ascii::space_type> call2;
	qi::rule<Iterator, std::string(), ascii::space_type> rest;

	void parseCommand(const std::string& str)
	{
//		std::cout << "cmd: " << str << std::endl;
		curCommand = boost::algorithm::trim_copy(str);
		boost::algorithm::to_upper(curCommand);
	}

	std::string extractName( std::string const& str )
	{
		if (str.find(' ') != string::npos )
			return str.substr( 0, str.find(' ') );
		else
			return str;
	}

	Package::Version extractVersion( std::string const& versionStr )
	{
		Package::Version tVersion;

		// try to extract version information
		// since the string is already trimmed, we can search for a space to identify
		// if the version information is given

		// Format: MIRA_REQUIRE_PACKAGE(name [Major[.Minor[.Patch [Build]]]])

		// Check number of arguments of MIRA_REQUIRE_PACKAGE
		std::vector<std::string> parts;
		boost::split(parts, versionStr, boost::is_from_range(' ', ' '), boost::token_compress_on);
		if (parts.size() > 3) {
			MIRA_LOG(WARNING) << "Unexpected number of arguments for MIRA_REQUIRE_PACKAGE.";
			return tVersion;
		}

		// Parse required version
		if (parts.size() > 1) {
			std::vector<std::string> tVersionParts;
			boost::split(tVersionParts, parts[1], boost::is_from_range('.', '.'));
			if ((tVersionParts.size() < 1 ) || (tVersionParts.size() > 3)) {
				std::string tName = extractName( versionStr );
				MIRA_LOG(WARNING) << "Cannot obtain version information for required package " << tName;
			}
			else {
				uint32 vMajor = 0, vMinor = 0, vPatch = 0;
				vMajor = fromString<uint32>(tVersionParts[0]);
				if (tVersionParts.size() >= 2)
					vMinor = fromString<uint32>(tVersionParts[1]);
				if (tVersionParts.size() >= 3)
					vPatch = fromString<uint32>(tVersionParts[2]);
				tVersion = Package::Version(vMajor, vMinor, vPatch);
			}
		}

		// Extract required BuildId
		if (parts.size() > 2)
			tVersion.mBuildId = fromString<uint32>(parts[2]);

		return tVersion;
	}

	void parseCommandParam(std::string& str)
	{
		boost::algorithm::trim(str);
		if(curCommand=="MIRA_REQUIRE_PACKAGE")
		{
			std::string tName = extractName( str );
			Package::Version tVersion = extractVersion( str );

			entry->mDependencies.push_back( new Dependency( tName, tVersion,
				Dependency::NONE ) );
		}
		else if(curCommand=="MIRA_OPTIONAL_PACKAGE")
		{
			std::string tName = extractName( str );
			Package::Version tVersion = extractVersion( str );

			entry->mDependencies.push_back( new Dependency( tName, tVersion,
				Dependency::FACULTATIVE ) );
		}
		else if(curCommand=="MIRA_RUNTIME_PACKAGE")
		{
			std::string tName = extractName( str );
			Package::Version tVersion = extractVersion( str );

			entry->mDependencies.push_back( new Dependency( tName, tVersion,
					Dependency::RUNTIME) );
		}
		else if(curCommand=="MIRA_VERSION")
		{
			std::vector<std::string> params;
			boost::split(params, str, boost::is_space(),boost::token_compress_on);
			if ( params.size() == 3 ) {
				// If MIRA_BUILD_ID was already there, we have to save the BuildID
				uint32 buildId = entry->mVersion.mBuildId;

				// Parse the version string
				entry->mVersion = Package::Version( fromString<uint32>(params[0]),
						fromString<uint32>(params[1]),
						fromString<uint32>(params[2]) );

				// Restore the BuildID
				entry->mVersion.mBuildId = buildId;
			}
		}
		else if(curCommand=="MIRA_BUILD_ID")
		{
			entry->mVersion.mBuildId = fromString<uint32>(str);
		}
		else if(curCommand=="MIRA_DESCRIPTION") {
			entry->mDescription += str;
		}
		else if(curCommand=="MIRA_CHANGELOG") {
			if ( !entry->mChangeLog.empty() )
				entry->mChangeLog += string("<br/>") + changeLogToHTML(str);
			else
				entry->mChangeLog = changeLogToHTML(str);
		}
		else if(curCommand=="MIRA_TAGS") {
			boost::algorithm::split( entry->mTags, str, boost::is_any_of(",;"),
					boost::token_compress_on );
			vector<string>::iterator it = entry->mTags.begin();
			while ( it != entry->mTags.end() ) {
				if ( it->empty() ) {
					entry->mTags.erase( it );
					it = entry->mTags.begin();
				}
				else {
					boost::trim( *it );
					boost::to_lower( *it );
				}
				it++;
			}
		}
		else if(curCommand=="MIRA_AUTHOR") {
			entry->mAuthor = str;
		}
		else if (curCommand=="MIRA_PACKAGE_TYPE")
		{
			if ( str == "SOURCE")
				entry->mType = Package::SOURCE;
			else if ( str=="BINARY")
				entry->mType = Package::BINARY;
			else if ( str=="ETC")
				entry->mType = Package::ETC;
		}
		else if (curCommand=="MIRA_DEVELOPMENT_STATE")
		{
			if ( str == "EXPERIMENTAL")
				entry->mDevelState = Package::EXPERIMENTAL;
			else if ( str=="STABLE")
				entry->mDevelState = Package::STABLE;
			else if ( str=="RELEASE")
				entry->mDevelState = Package::RELEASE;
			else if ( str=="DEPRECATED")
				entry->mDevelState = Package::DEPRECATED;
		}
		else if (curCommand=="MIRA_PACKAGE_NAME")
		{
			entry->mName = str;
		}
	}

	std::string curCommentKeyword;
	std::string curCommand;

	Package* entry;
};

///////////////////////////////////////////////////////////////////////////////

/*
void test()
{
    std::ifstream in( "/local/erik/wrk/MIRA/domainstmp/adapters/BlackboardAdapter/BlackboardAdapter.package", std::ifstream::in );

    in.unsetf(std::ios::skipws); //  Turn of white space skipping on the stream
    std::string fileContent;
    fileContent = std::string(std::istream_iterator<char>(in),std::istream_iterator<char>());

    PackageEntry e;
	PackageParser<std::string::const_iterator> parser(e);
	std::string::const_iterator iter = fileContent.begin();
	std::string::const_iterator end = fileContent.end();
	bool r = qi::phrase_parse(iter,end, parser, ascii::space);
	if(r) {
		//db.packages.insert(e);
	} else {
		std::cout << "Failed to parse package file " << std::endl;
	}
}
*/

template<class InputIterator, class T>
InputIterator findName ( InputIterator first, InputIterator last, const T& value )
{
	for ( ;first!=last; ++first)
		if ( (*first)->mName==value ) break;
	return first;
}

Package* indexWebPackage( Url const& relUrl, Repository* repo, string* iFileContent)
{
	assert(Path(relUrl).extension() == ".package" || ".extpackage");

	MIRA_LOG(NOTICE) << "indexing: " << relUrl;

	// obtain filename
	std::string filename = getFilename( relUrl );
	// extract name of package by removing the file extension
	std::string name = filename.substr(0, filename.size()
			- Path(relUrl).extension().string().length() );

	// if the file content is not already given, we have to load it
	string fileContent;
	if ( !iFileContent )
		fileContent = repo->catFile( relUrl );
	else
		fileContent = *iFileContent;
	fileContent = QString::fromStdString(fileContent).toStdString();

	Package* source = new Package( name );

	// parse the file content using boost spirit
	try {
		PackageParser<std::string::const_iterator> parser(*source);
		std::string::const_iterator iter = fileContent.begin();
		std::string::const_iterator end = fileContent.end();
		bool r = qi::phrase_parse(iter,end, parser, ascii::space);
		if(r) {
			source->mRepos.insert( repo->url );
			Url tSubPath( relUrl );
			source->mFileName = getFilename( tSubPath );
			source->mRepoSubPath = getParentUrl( tSubPath );
			if ( Path(relUrl).extension() == ".extpackage" )
				source->mType = Package::ETC;
		} else {
			// package file cannot be parsed properly
			delete source;
			source = NULL;
			MIRA_LOG(NOTICE) << "Failed to parse package file: " << repo->url << relUrl;
		}
	} catch(Exception& ex) {
		MIRA_LOG(WARNING) << "Failed to parse package file: " << repo->url << relUrl << ": " << ex.what();
	}
	return source;
}

Package* indexLocalPackage(const Path& path, Repository* repository)
{
	assert( path.extension() == ".package" || ".extpackage" );

	// obtain filename
	std::string filename = path.filename().string();
	// extract name of package by removing the file extension
	std::string name = filename.substr(0, filename.size() -
			path.extension().string().length() );
	// generate the associated changelog filename
	std::string changelog = (path.parent_path()/name).string()+".changelog";

	// load the local file
	ifstream file(path.string());
	std::string fileContent( (std::istreambuf_iterator<char>(file) ),
	                       (std::istreambuf_iterator<char>()    ) );

	// try to read and append separate changelog
	ifstream log(changelog);
	std::string logContent( (std::istreambuf_iterator<char>(log) ),
						   (std::istreambuf_iterator<char>()    ) );

	Package* source = new Package( name );
	Path tRootPath( path );
	// obtain the path where the package is located
	source->mLocalPath = tRootPath.remove_filename().string();

	// obtain the repository root and the relative path for packages
//	Repository* tRepository = db.getRepositoryForLocalFile( path.string() );
	if ( repository ) {
		source->mRepos.insert( repository->url );
		source->mCurrentRepo = repository->url;
		Path tSubPath( repository->getRelativePathFor( path.string() ) );
		source->mFileName = path.filename().string();
		source->mRepoSubPath = tSubPath.generic_string();
		// handle special case. we don't want to lost a single /
		if ( source->mRepoSubPath != "/" )
			source->mRepoSubPath = getParentUrl( source->mRepoSubPath );
	}

	// parse the package using boost spirit
	try {
		PackageParser<std::string::const_iterator> parser(*source);
		std::string::const_iterator iter = fileContent.begin();
		std::string::const_iterator end = fileContent.end();
		bool r = qi::phrase_parse(iter,end, parser, ascii::space);

		if(r) {
			if ( path.extension() == ".extpackage" )
				source->mType = Package::ETC;
			if (logContent.size())
				source->mChangeLog = changeLogToHTML(logContent);
		} else {
			// something went wrong during parsing
			delete source;
			source = NULL;
			MIRA_LOG(WARNING) << "Failed to parse package file: " << path;
		}
	} catch(Exception& ex) {
		MIRA_LOG(WARNING) << "Failed to parse package file: " << path << ": " << ex.what();
	}
	return source;
}

///////////////////////////////////////////////////////////////////////////////

} // end namespace
