/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FileRepository.C
 *    Description.
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2012/10/07
 */
#include "core/FileRepository.h"

#include <QBuffer>
#include <QDataStream>
#include <QTextStream>

#include <quazip/quazip.h>
#include <quazip/JlCompress.h>

#include <serialization/Serialization.h>

#include "core/Tools.h"
#include "core/Package.h"

#ifdef MIRA_LINUX
# include <sys/stat.h>
#endif

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

FileRepository::FileRepository( QObject* parent) : Repository()
{
	name = getClass().getMetaInfo("RepoType");
	url += name;
}

///////////////////////////////////////////////////////////////////////////////

string FileRepository::catFile( Url const& relUrl )
{
	QByteArray byteArray ;
	getFile( url / relUrl, byteArray );
	return QString(byteArray).toStdString();
}

bool FileRepository::isResponsibleFor( Path const& localPath )
{
	Path tLogFile = findLogFile( localPath );
	QFile logFile( QString::fromStdString( tLogFile.string() ) );
	if ( logFile.exists() ) {
		if ( isGeneric() )
			return true;

		logFile.open( QIODevice::ReadOnly );
		QString repo = logFile.readLine();
		QString subPath = logFile.readLine();
		logFile.close();

		repo = repo.section('=',1,-1).trimmed();
		subPath = subPath.section('=',1,-1).trimmed();

		string packagePath = (Path( repo.toStdString() ) / Path( subPath.toStdString() )).string();

#ifdef MIRA_LINUX
		return ( packagePath.size() > url.size() &&
				 packagePath.substr( 0, url.size()+1 ) == url +"/" );
#else
		return ( packagePath.size() > url.size() &&
				 packagePath.substr( 0, url.size()+1 ) == url +"\\" );
#endif

	}
	return false;
}

mira::Path FileRepository::getRelativePathFor( Path const& localPath )
{
	Path tLogFile = findLogFile( localPath );
	QFile logFile( QString::fromStdString( tLogFile.string() ) );
	if ( logFile.exists() ) {
		logFile.open( QIODevice::ReadOnly );
		QString repo = logFile.readLine();
		QString subPath = logFile.readLine();
		logFile.close();

		subPath = subPath.section('=',1,-1).trimmed();

		return Path( subPath.toStdString() ) / localPath.filename();
	}
	return "";
}

void FileRepository::install( Package const& package, Path const& destPath )
{
	Url relUrl = package.mRepoSubPath / package.mFileName;

	MIRA_LOG( NOTICE ) << "install (protocol=" << protocol << ")";
	MIRA_LOG( NOTICE ) << "source: " << relUrl;
	MIRA_LOG( NOTICE ) << "destination: " << destPath;

	// create the new directory and CMakeLists.txt if necessary
	prepareCheckoutDirectory( destPath, package.mType & Package::SOURCE );

	///////////////////////////////////////////////////////////////////////
	// we need a log file in addition to the package file to be able to
	// delete the package later on
	///////////////////////////////////////////////////////////////////////

	QFile logFile( QString::fromStdString(
			( destPath / getFilename( relUrl )).string() + ".log" ) );
	if ( logFile.exists() ) {
		MIRA_THROW( XLogical, "Logfile " + getFilename( relUrl )+".log" +
		            " already exists!");
	}
	if ( !logFile.open( QIODevice::WriteOnly ) )
		MIRA_THROW( XLogical, "Cannot create file " + getFilename( relUrl ) );

	QTextStream logWriter( &logFile );
	Url relRepoUrl = getParentUrl( relUrl );
	//relRepoUrl.remove_filename();
	logWriter << "Repo=" << QString::fromStdString( url )<<"\n";
	logWriter << "RelRepoUrl=" << QString::fromStdString( relRepoUrl ) << "\n";

	///////////////////////////////////////////////////////////////////////
	// we can load and extract the zip file right now
	///////////////////////////////////////////////////////////////////////

	Url fullPath;
	if ( relUrl.find('/') != string::npos )
		fullPath = fullPath / Url( relUrl.substr(0,relUrl.rfind('/') ) );

	fullPath = fullPath / "";

	MIRA_LOG(NOTICE) << "get zipfile list for repo suburl " << fullPath;
	vector<Url> tZipFiles = getZipFileList( fullPath );

	foreach( Url tZipFile, tZipFiles ) {

		QByteArray byteArray ;
		getFile( url / tZipFile, byteArray );
		QBuffer buffer( &byteArray ) ;

		QuaZip testZip;
		testZip.setIoDevice( &buffer );
		if ( !testZip.open(QuaZip::mdUnzip) ) {
			MIRA_THROW( XIO, "Unable to open zip file \""+tZipFile+"\"" );
		}

		QDir dir( QString::fromStdString( destPath.string() ) );

		QString relativeMIRAPath;

		bool isWithinMIRAPath = false;
		if ( !PathProvider::getAssociatedMiraPath( destPath ).empty() ) {
			isWithinMIRAPath = true;
			relativeMIRAPath = dir.relativeFilePath(
					QString::fromStdString(
					PathProvider::getAssociatedMiraPath( destPath ).string() ) );

			// HACK: When both paths are the same, Qt4 returns a "" but Qt5 returns "."
			if (relativeMIRAPath == ".")
				relativeMIRAPath = "";
		}

		QuaZipFile file(&testZip);
		for(bool more=testZip.goToFirstFile(); more; more=testZip.goToNextFile()) {
			file.open(QIODevice::ReadOnly);

			QString fileName = file.getActualFileName();
			if ( fileName.contains("MIRA_") ) {
				// replace placeholder with the corresponding mira path
				if ( !isWithinMIRAPath )
					MIRA_THROW( XLogical, string( "Cannot find associated MIRA_PATH for path " ) + destPath.string() );
				fileName.replace("MIRA_", relativeMIRAPath );
			}

			bool isSymLink = false;
#ifdef MIRA_LINUX
			// On Linux/Unix we have to ensure, that the file permissions
			// are restored based on the attributes in the zip file. Other-
			// wise e.g., installed executables will not be executable.
			// For some more information read here:
			//    http://unix.stackexchange.com/questions/14705/the-zip-formats-external-file-attribute
			QuaZipFileInfo fileInfo;
			file.getFileInfo(&fileInfo);
			uint32 fileAttr = (fileInfo.externalAttr >> 16);
			QFile::Permissions filePerm;
			if (fileAttr & S_IROTH)
				filePerm |= QFile::ReadOther;
			if (fileAttr & S_IWOTH)
				filePerm |= QFile::WriteOther;
			if (fileAttr & S_IXOTH)
				filePerm |= QFile::ExeOther;
			if (fileAttr & S_IRGRP)
				filePerm |= QFile::ReadGroup;
			if (fileAttr & S_IWGRP)
				filePerm |= QFile::WriteGroup;
			if (fileAttr & S_IXGRP)
				filePerm |= QFile::ExeGroup;
			if (fileAttr & S_IRUSR)
				filePerm |= QFile::ReadUser;
			if (fileAttr & S_IWUSR)
				filePerm |= QFile::WriteUser;
			if (fileAttr & S_IXUSR)
				filePerm |= QFile::ExeUser;

			isSymLink = ((fileAttr & S_IFLNK) == S_IFLNK);
#endif

			if ( !fileName.isEmpty() ) {
				int lastSlashIdx = fileName.lastIndexOf('/');

				// check if the entry contains a path name
				if (lastSlashIdx != -1) {
					// create a new directory
					QString pathName = fileName.left(lastSlashIdx);
					if ( !dir.exists( pathName ) ) {
						dir.mkpath( pathName );
						logWriter << pathName << "/\n";
					}
				}

				// if the entry was not a "dir-only" entry, we create a file
				if (lastSlashIdx < fileName.length()-1){
					// create a new file
					char* buffer = new char[10000];
					QDataStream reader( &file );

					QFile outFile( QString::fromStdString( destPath.string() )
						+ "/" + fileName );
					if ( outFile.exists() ) {
						if ( !prompt->showYesNoErrorMessage("File " + fileName.toStdString() +
						                                    " already exists. Overwrite?"))
						{
							// revert everything we did until here
							MIRA_LOG(NOTICE) << "Revert installation of package " << package.mName;
							logFile.close();
							uninstall(package);
							// stop here
							MIRA_THROW( XIO, "File " + fileName.toStdString() + " already exists!");
						}
					}

					if ( !outFile.open( QIODevice::WriteOnly ) ) {
						// revert everything we did until here
						MIRA_LOG(NOTICE) << "Revert installation of package " << package.mName;
						logFile.close();
						uninstall(package);
						// stop here
						MIRA_THROW( XIO, "Cannot create file " + fileName.toStdString() +"!");
					}
					QDataStream writer( &outFile );

					uint bytesRead = 10000;

					while ( bytesRead == 10000 ) {
						bytesRead = reader.readRawData( buffer, bytesRead );
						writer.writeRawData( buffer, bytesRead );
					}

#ifdef MIRA_LINUX
					outFile.setPermissions(filePerm);
#endif
					outFile.close();

#ifdef MIRA_LINUX
					// If this entry is only a symlink, the created file
					// will only contain the name of the link destination.
					// Therefore, we have to read it and create the symlink.
					if (isSymLink) {
						outFile.open(QIODevice::ReadOnly | QIODevice::Text);
						QTextStream outFileReader(&outFile);
						QString linkDest = outFileReader.readLine();
						outFile.close();

						Path symLinkPath = normalizePath(destPath / fileName.toStdString());
						Path linkDestPath = normalizePath(Path(linkDest.toStdString()));

						boost::filesystem::remove(symLinkPath);
						boost::filesystem::create_symlink(linkDestPath, symLinkPath);
					}
#endif

					delete buffer;
					logWriter << fileName << "\n";
				}
			}

			file.close();
		}

		testZip.close();
	}
	logFile.close();

	// finally check if a CMakeLists.txt file exists in the root dir
	// (do not add to the log file since other packages may use it)
	if ( package.mType & Package::SOURCE ) {
		Path cm = destPath/"CMakeLists.txt";
		if (!boost::filesystem::exists(cm))
			generateCMakeListsTxt(cm);
	}
}

void FileRepository::uninstall( Package const& package )
{
	Path tLogFileSearch = package.mLocalPath / package.mFileName;
	Path tLogFile = findLogFile(tLogFileSearch);

	QFile logFile( QString::fromStdString( tLogFile.string() ) );
	if ( logFile.exists() ) {
		if ( !logFile.open( QIODevice::ReadOnly ) )
			MIRA_THROW( XIO, "Cannot open log file for package \"" + package.mName + "\"!" );
		QString repo = logFile.readLine();
		QString subPath = logFile.readLine();

		Path tLocalPath = tLogFile;
		tLocalPath.remove_filename();

		QDir tDir( QString::fromStdString( tLocalPath.string() ) );
		vector<QString> tItemList;

		while( !logFile.atEnd() ) {
			QString entry = logFile.readLine();
			tItemList.push_back( entry.trimmed() );
		}
		logFile.close();

		vector<QString>::reverse_iterator it = tItemList.rbegin();
		for(; it != tItemList.rend(); it++ ) {
			if ( (*it).size() > 0 ) {
				if ( (*it)[(*it).size()-1] == '/' ) {
					// remove directory
					MIRA_LOG( NOTICE ) << "remove directory " << it->toStdString();
					if ( !tDir.rmdir( (*it) ) ) {
						/*
						if ( !prompt->showYesNoErrorMessage("cannot remove directory \""
								+ it->toStdString() + "\", during uninstall of package \""
								+ package.mName + "\"!\nDo you want to proceed?") )
							MIRA_THROW( XIO, "Cannot remove directory " + it->toStdString() + "!" );
						*/
						MIRA_LOG(WARNING) << "Can't remove directory " << it->toStdString();
					}
				}
				else {
					// remove file
					MIRA_LOG( NOTICE ) << "remove file " << it->toStdString();
					if (!tDir.remove( (*it) )) {
						/*
						if ( !prompt->showYesNoErrorMessage("cannot remove file \""
								+ it->toStdString() + "\", during uninstall of package \""
								+ package.mName + "\"!\nDo you want to proceed?") )
							MIRA_THROW( XIO, "Cannot remove file " + it->toStdString() + "!" );
						*/
						MIRA_LOG(WARNING) << "Can't remove file " << it->toStdString();
					}
				}
			}
		}

		// now we should remove all generated files of this package
		if ( package.mType == Package::SOURCE )
			removeGeneratedLibraryFiles( package );

		// finally, remove the log file
		if ( !tDir.remove( QString::fromStdString( tLogFile.filename().string() ) ) )
			MIRA_THROW( XIO, "Cannot remove log file " + tLogFile.filename().string() + "!" );
	}
	else {
		MIRA_THROW( XIO, "Cannot find log file " << tLogFileSearch <<
		            " for package \"" << package.mName << "\"!" );
	}
}

Path FileRepository::findLogFile( Path const& packagePath )
{
	// get associated MIRA path
	Path miraPath = PathProvider::getAssociatedMiraPath(packagePath);

	// Try to locate the log file in the current directory on one of the
	// parent directories.
	Path pkgLogPath = packagePath.string()+".log";
	do {
		if (boost::filesystem::exists(pkgLogPath))
			return pkgLogPath;
		// Abort search when we reached the associated MIRA path.
		if (Path(pkgLogPath).remove_filename() == miraPath)
			break;
		pkgLogPath = pkgLogPath.parent_path().parent_path() / pkgLogPath.filename();
	} while(true);

	return Path();
}

vector<Url> FileRepository::getZipFileList(const Url& subdir)
{
	vector<Url> emptyVec;
	return emptyVec;
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::FileRepository, mira::Repository);
