/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DriveView.C
 *    Implementation of DriveView.h.
 *
 * @author Tim Langner
 * @date   2011/04/26
 */

#include <gui/views/DriveView.h>

#include <serialization/PropertySerializer.h>
#include <transform/Velocity.h>

#include <QVBoxLayout>
#include <QMenu>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

DriveView::DriveView() :
	velocityIncrement(0.1f),
	watchdogTimeout(Duration::seconds(5)),
	driveService("IDrive"),
	mControl(NULL),
	ui(NULL)
{
	motorstopChannel.set("/robot/Bumper", NULL);
}


DriveView::~DriveView()
{
	delete mControl;
}

QWidget* DriveView::createPartControl()
{
	ui = new UI(this);
	motorstopChannel.setDataChangedCallback(&UI::onMotorstop, ui);
	return ui;
}

void DriveView::driveServiceChanged()
{
	if (ui)
		ui->updateService();
}

Object* DriveView::getAdapter(const Class& adapter)
{
	if(adapter == PropertyViewPage::CLASS())
	{
		if(mControl==NULL)
		{
			PropertySerializer s;
			DriveView* This = this;
			PropertyNode* p = s.reflectProperties(getClass().getName(), This);
			mControl = new PropertyViewPage(boost::shared_ptr<PropertyNode>(p));
			mControl->setWindowTitle(this->windowTitle());
		}
		return mControl;
	}

	return ViewPart::getAdapter(adapter);
}

///////////////////////////////////////////////////////////////////////////////

DriveView::UI::UI(DriveView* parent) :
	QWidget(parent),
	authority("/", "DriveView", Authority::ANONYMOUS | Authority::INTERNAL),
	lastCommand(Time::invalid()), mParent(parent)
{
	Ui::DriveViewWidget::setupUi(this);
	setFocusPolicy(Qt::StrongFocus);
	commandTimer = new QTimer(this);
	commandTimer->setInterval(100);
	commandTimer->setSingleShot(false);

	serviceTimer = new QTimer(this);
	serviceTimer->setInterval(5000);
	serviceTimer->setSingleShot(false);

	forward->setIcon(QIcon(":/icons/ArrowUp.png"));
	backward->setIcon(QIcon(":/icons/ArrowDown.png"));
	left->setIcon(QIcon(":/icons/ArrowLeft.png"));
	right->setIcon(QIcon(":/icons/ArrowRight.png"));
	stop->setIcon(QIcon(":/icons/Cancel.png"));

	connect(forward, SIGNAL(clicked()), this, SLOT(onForward()));
	connect(backward, SIGNAL(clicked()), this, SLOT(onBackward()));
	connect(left, SIGNAL(clicked()), this, SLOT(onLeft()));
	connect(right, SIGNAL(clicked()), this, SLOT(onRight()));
	connect(stop, SIGNAL(clicked()), this, SLOT(onStop()));
	connect(bumper, SIGNAL(clicked()), this, SLOT(onResetBumper()));
	connect(odometry, SIGNAL(clicked()), this, SLOT(onResetOdometry()));
	connect(mBtEnable, SIGNAL(clicked()), this, SLOT(onEnableMotors()));
	connect(mBtDisable, SIGNAL(clicked()), this, SLOT(onDisableMotors()));
	connect(commandTimer, SIGNAL(timeout()), this, SLOT(tick()));
	connect(serviceTimer, SIGNAL(timeout()), this, SLOT(updateService()));
	connect(mBtGrabKeyboard, SIGNAL(toggled(bool)), this, SLOT(onGrabKeyboard(bool)));

	updateService();
	transSpeed = 0;
	rotSpeed = 0;
	commandTimer->start();
	serviceTimer->start();
}

void DriveView::UI::tick()
{
	try {
		mParent->motorstopChannel.update();
	}
	catch(XRuntime&) {}

	if (!lastCommand.isValid()) // do we have a command velocity?
		return;

	try {
		if ((Time::now() - lastCommand) > mParent->watchdogTimeout)
		{
			onStop();
			return;
		}
		authority.callService<void>(service, "setVelocity",
		                            Velocity2(transSpeed, 0.0f, rotSpeed));
	} catch(...) {}
}

void DriveView::UI::enableDriveButtons(bool enable)
{
	forward->setEnabled(enable);
	backward->setEnabled(enable);
	left->setEnabled(enable);
	right->setEnabled(enable);
	stop->setEnabled(enable);
}

void DriveView::UI::enableMotorButtons(bool enable)
{
	bumper->setEnabled(enable);
	odometry->setEnabled(enable);
	mBtEnable->setEnabled(enable);
	mBtDisable->setEnabled(enable);
}

void DriveView::UI::updateService()
{
	// parent gives us a service so use it
	if (!mParent->driveService.empty())
		service = mParent->driveService;

	// if service is not set in the DriveView's ServiceProperty
	// (and we don't have one yet), try to find a required interface
	// (prefer IMotorController, but also accept IDrive)
	if (service.empty())
	{
		auto l = authority.queryServicesForInterface("IMotorController");
		if (l.empty())
			l = authority.queryServicesForInterface("IDrive");
		if (l.empty())
		{
			enableDriveButtons(false);
			enableMotorButtons(false);
			return;
		}
		service = *l.begin();
	}

	if (authority.waitForService(service, Duration::seconds(1)))
	{
		enableDriveButtons(authority.implementsInterface(service, "IDrive"));
		enableMotorButtons(authority.implementsInterface(service, "IMotorController"));
		return;
	}

	enableDriveButtons(false);
	enableMotorButtons(false);
	service.clear();
}

void DriveView::UI::onForward()
{
	if (rotSpeed == 0)
		transSpeed += mParent->velocityIncrement;
	rotSpeed = 0;
	lastCommand = Time::now();
	tick();
}

void DriveView::UI::onBackward()
{
	if (rotSpeed == 0)
		transSpeed -= mParent->velocityIncrement;
	rotSpeed = 0;
	lastCommand = Time::now();
	tick();
}

void DriveView::UI::onLeft()
{
	rotSpeed += mParent->velocityIncrement;
	lastCommand = Time::now();
	tick();
}

void DriveView::UI::onRight()
{
	rotSpeed -= mParent->velocityIncrement;
	lastCommand = Time::now();
	tick();
}

void DriveView::UI::onStop()
{
	rotSpeed = 0;
	transSpeed = 0;
	try {
		authority.callService<void>(service, "setVelocity",
		                            Velocity2(transSpeed, 0.0f, rotSpeed));
		lastCommand = Time::invalid();
	}
	catch(...)
	{}
}

void DriveView::UI::onResetBumper()
{
	rotSpeed = 0;
	transSpeed = 0;
	try {
		authority.callService<void>(service, "setVelocity",
		                            Velocity2(transSpeed, 0.0f, rotSpeed));
		lastCommand = Time::invalid();
		authority.callService<void>(service, "resetMotorStop");
	}
	catch(...)
	{}
}

void DriveView::UI::onResetOdometry()
{
	try {
		authority.callService<void>(service, "resetOdometry");
	}
	catch(...)
	{}
}

void DriveView::UI::onEnableMotors()
{
	try {
			authority.callService<void>(service, "enableMotors", true);
		}
	catch(...)
	{}
}

void DriveView::UI::onDisableMotors()
{
	try {
			authority.callService<void>(service, "enableMotors", false);
		}
	catch(...)
	{}
}

void DriveView::UI::onGrabKeyboard(bool activate)
{
	if(activate)
		this->grabKeyboard();
	else
		this->releaseKeyboard();
}

void DriveView::UI::keyPressEvent(QKeyEvent *e)
{
	if (e->key() == Qt::Key_Right)
	{
		onRight();
		return;
	}

	if (e->key() == Qt::Key_Left)
	{
		onLeft();
		return;
	}

	if (e->key() == Qt::Key_Up)
	{
		onForward();
		return;
	}

	if (e->key() == Qt::Key_Down)
	{
		onBackward();
		return;
	}

	if (e->key() == Qt::Key_Space)
	{
		onStop();
		return;
	}


	if (e->key() == Qt::Key_Escape)
	{
		mBtGrabKeyboard->setChecked(false);
		return;
	}


	QWidget::keyPressEvent(e);
}

void DriveView::UI::onMotorstop(ChannelRead<bool> status)
{
	if (status->value()) {
		// It seems impossible to change a QPushButton's background color easily and consistently.
		// Palette is ignored by qt5ct style (but not e.g. by Windows style), while the stylesheet
		// way is over-complicated and rendered slightly differently across styles (mostly with
		// ugly result). I once liked Qt...
		//bumper->setStyleSheet("background-color:red; border:none; padding:6px");

		// So we stick to highlighting the text. Not as eye-catching, but it does as expected.
		bumper->setStyleSheet("color:red; font-weight:bold; font-size:18px");
	} else {
		bumper->setStyleSheet("");
	}
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::robot::DriveView, mira::ViewPart);
