/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OdometryCorrectionDifferentialDrive.h
 *    Allows the correction of systematical odometry errors of differential
 *    drives.
 *
 * @author Erik Einhorn
 * @date   2012/12/23
 */

#ifndef _MIRA_ODOMETRYCORRECTIONDIFFERENTIALDRIVE_H_
#define _MIRA_ODOMETRYCORRECTIONDIFFERENTIALDRIVE_H_

#include <serialization/DefaultInitializer.h>

#include <robot/Odometry.h>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * Allows the correction of systematical odometry errors of differential drives.
 * This includes inaccuracies in the wheel base (the distance between both
 * wheels), slightly different sizes of the left and right wheel and
 * inaccuracies in the average size of both wheels.
 */
class OdometryCorrectionDifferentialDrive
{
public:
	OdometryCorrectionDifferentialDrive()
	{
		MIRA_INITIALIZE_THIS;
	}

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("cb", cb, "Correction factor for wheel base (distance between both wheels). "
		                     "Compensates deviations of the wheel base due to construction. "
		                     "(default: 1.0 which does no correction)", 1.0f);

		r.property("cr",cr, "Correction factor for wheel diameter ratio between left and right wheel. "
		                    "This compensates small differences between the wheel diameter of the "
		                    "left and right wheel. (default: 1.0 which does no correction)", 1.0f);

		r.property("cd",cd, "Correction factor for the average wheel diameter. "
		                    "This can be regarded as an overall scaling factor for the "
		                    "left and right wheel. (default: 1.0 which does no correction)", 1.0f);
	}


public:

	/**
	 * Corrects an odometry reading consisting of a translation and rotation.
	 * For the specified translation and rotation the corrected values will
	 * be returned in oTrans and oRot.
	 */
	void correctTransRot(float trans, float rot, float& oTrans, float& oRot);

	/**
	 * Returns the updated odometry lastCorrected by adding the corrected
	 * distance between the previous reading p0 and the current reading p1.
	 *
	 * @param lastCorrected The corrected odometry of the last call
	 * @param p0 The previous odometry
	 * @param p1 The current odometry
	 * @return The current corrected odometry
	 */
	Odometry2 correct(const Odometry2& lastCorrected, const Odometry2& p0, const Odometry2& p1);

public:
	// Parameters

	/**
	 * Correction factor for wheel base (distance between both wheels).
	 * Compensates deviations of the wheel base due to construction.
	 */
	float cb;

	/**
	 * Correction factor for wheel diameter ratio between left and right wheel.
	 * This compensates small differences between the wheel diameter of the
	 * left and right wheel.
	 */
	float cr;

	/**
	 * Correction factor for the average wheel diameter.
	 * This can be regarded as an overall scaling factor for the
	 * left and right wheel.
	 */
	float cd;
};

///////////////////////////////////////////////////////////////////////////////

inline void OdometryCorrectionDifferentialDrive::correctTransRot(float trans, float rot, float& oTrans, float& oRot)
{
	// precompute correction factors for left and right wheel
	// TODO: this could be precomputed if we noticed any performance troubles
	float clw = 2.0f/(cr+1.0f)  * cd;
	float crw = 2.0f/(1.0f/cr+1.0f)  * cd;

	// convert to left/right wheel speed
	float t = rot*0.5f;
	float vl = trans-t;
	float vr = trans+t;

	// correct ratio of left/right wheel diameter
	vl *= clw;
	vr *= crw;

	// back to translation/rotation
	oTrans = (vr + vl)*0.5f;
	oRot   = (vr - vl)/cb; // correct wheel base here
}

inline Odometry2 OdometryCorrectionDifferentialDrive::correct(const Odometry2& lastCorrected,
                                                          const Odometry2& p0,
                                                          const Odometry2& p1)
{
	// get relative movement since last update
	Eigen::Vector2f dt = p1.pose.t -  p0.pose.t;
#ifdef MIRA_LINUX
	float trans = copysignf(dt.norm(), p1.velocity.x());
#endif
#ifdef MIRA_WINDOWS
	float trans = _copysign(dt.norm(), p1.velocity.x());
#endif
	float turn = smallestAngleDifference(p1.pose.r.angle(), p0.pose.r.angle());
	correctTransRot(trans,turn, trans,turn);
	float sinphi0=std::sin(lastCorrected.pose.phi());
	float cosphi0=std::cos(lastCorrected.pose.phi());
	return Odometry2(Pose2(lastCorrected.pose.x()   + cosphi0*trans,
	                       lastCorrected.pose.y()   + sinphi0*trans,
	                       lastCorrected.pose.phi() + turn),
	                 p1.velocity);
}


///////////////////////////////////////////////////////////////////////////////

} } // namespace

#endif
