/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GenericRobotModel.h
 *    A generic robot model that supports multi wheel robots.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2011/10/21
 */


#include <robot/RobotModel.h>

#ifndef _MIRA_GENERICROBOTMODEL_H_
#define _MIRA_GENERICROBOTMODEL_H_

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

/**
 * A generic robot model that allows to specify a wheel geometry for multi
 * wheel robots with fixed or non-fixed castor wheels.
 * For a special optimized implementation for robots with a differential drive
 * please use DifferentialRobotModel.
 */
class MIRA_ROBOT_DATATYPES_EXPORT GenericRobotModel : public RobotModel
{
	MIRA_OBJECT(mira::robot::GenericRobotModel)
public:

	template<typename Reflector>
	void reflect(Reflector& r)
	{
		RobotModel::reflect(r);
		r.member("WheelGeometry", wheelGeometry, "The position of each wheel");
	}

public:

	/**
	 * Computes the robots overall velocity from the readings of its wheels.
	 * Each row represents the state of a single wheel.
	 * First column contains the velocity of that wheel.
	 * Second column contains the orientation of that wheel.
	 * For castor wheels orientation will change while driving. For fixed wheels
	 * orientation will most likely be fixed.
	 */
	Velocity2 computeRobotVelocity(const Eigen::Matrix<float,
	                               Eigen::Dynamic,2>& castorWheelState);

	/// Implementation of RobotModel
	virtual Pose2 localKinematics(const Velocity2& v, float dt) const;

	/// Implementation of RobotModel
	virtual float predictStandstillDistance(const Velocity2& v) const;
	virtual float predictStandstillRotation(const Velocity2& v) const;

	/**
	 * Generates a trajectory by accelerating/decelerating starting at velocity v
	 * for lookAheadTime time up to the velocity targetV.
	 * The resulting trajectory will contain samples+1 trajectory samples starting
	 * at position 0,0 with velocity v.
	 * It uses the acceleration limits given in the robot model.
	 */
	virtual PoseVelocityTrajectory generateTrajectory(Velocity2 v, const Velocity2& targetV,
	                                                  float lookAheadTime, int samples);

public:

	/// The position of each wheel relative to the robots base
	std::vector<Pose2> wheelGeometry;
};

/// Typedef for GenericRobotModel pointer
typedef boost::shared_ptr<GenericRobotModel> GenericRobotModelPtr;

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
