/*
 * Copyright (C) by
 *   MetraLabs GmbH (MLAB), GERMANY
 *  and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Redistribution and modification of this code is strictly prohibited.
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PinholeCameraIntrinsic
 *    intrinsic parameters of a pinhole camera
 *
 * @author Erik Einhorn
 * @date   2010/11/26
 */

#ifndef _MIRA_CAMERA_PINHOLECAMERAINTRINSIC_H_
#define _MIRA_CAMERA_PINHOLECAMERAINTRINSIC_H_

#include <vector>

#include <math/Eigen.h>
#include <geometry/Size.h>

#include <cameraparameters/CameraParametersExports.h>
#include <cameraparameters/DistortionParameters.h>

namespace mira { namespace camera {

//////////////////////////////////////////////////////////////////////////////

/**
 * Describes the intrinsic parameters of a pinhole camera (focal length, pincushion
 * distortion parameters, etc).
 * For each physical camera just one instance of this class must be used since
 * the intrinsic camera parameters remain constant.
 *
 * The focal length and the optical center parameters must be specified
 * relative to the image extension and NOT in pixel. E.g. if the focal
 * length fx was estimated using an 640x480 image during the calibration
 * then (fx/640) must be passed as normalized focal length parameter.
 */
class MIRA_CAMERAPARAMETERS_EXPORT PinholeCameraIntrinsicNormalized : public DistortionParameters
{
public:

	/// Constructor.
	PinholeCameraIntrinsicNormalized();

	/**
	 * Sets all intrinsic parameters, such as the focal length Fx and Fy,
	 * the optical center Cx, Cy and the distortion parameters K1, K2, P1, P2.
	 *
	 * The focal length and the optical center parameters must be specified
	 * relative to the image extension and NOT in pixel. E.g. if the focal
	 * length fx was estimated using an 640x480 image during the calibration
	 * then (fx/640) must be passed as normalized focal length parameter.
	 */
	PinholeCameraIntrinsicNormalized(float iFx, float iFy, float iCx, float iCy,
	                                 float iK1, float iK2, float iP1, float iP2);

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Fx",  fx,  "norm. coordinate of the focal point");
		r.property("Fy",  fy , "norm. coordinate of the focal point");
		r.property("Cx",  cx,  "norm. coordinate of the optical center");
		r.property("Cy",  cy,  "norm. coordinate of the optical center");
		DistortionParameters::reflect(r);
	}

public:

	/**
	 * Returns the camera calibration matrix of the rectified images
	 * with the given size
	 */
	Eigen::Matrix3f getK(const Size2i& imageSize) const;

	/**
	 * Returns the inverse camera calibration matrix of the rectified
	 * images with the given size
	 */
	Eigen::Matrix3f getK_inv(const Size2i& imageSize) const;

public:

	bool operator==(const PinholeCameraIntrinsicNormalized& other) const {
		return ((fx == other.fx) && (fy == other.fy) && (cx == other.cx) && (cy == other.cy) && ((DistortionParameters)*this == (DistortionParameters)other));
	}

public:

	// the intrinsic parameters
	float fx; ///<norm. coordinate of the focal point in x
	float fy; ///<norm. coordinate of the focal point in y
	float cx; ///<norm. coordinate of the optical center in x
	float cy; ///<norm. coordinate of the optical center in y
};

///////////////////////////////////////////////////////////////////////////////

class MIRA_CAMERAPARAMETERS_EXPORT PinholeCameraIntrinsic : public DistortionParameters
{
public:

	/// Constructor.
	PinholeCameraIntrinsic();

	/**
	 * Sets all intrinsic parameters, such as the focal length Fx and Fy,
	 * the optical center Cx, Cy and the distortion parameters K1, K2, P1, P2.
	 */
	PinholeCameraIntrinsic(float iFx, float iFy, float iCx, float iCy,
	                       float iK1, float iK2, float iP1, float iP2);



	PinholeCameraIntrinsic(const PinholeCameraIntrinsicNormalized& normalizedIntrinsic,
	                       const Size2i& imageSize);

public:

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("Fx",  fx,  "coordinate of the focal point");
		r.property("Fy",  fy , "coordinate of the focal point");
		r.property("Cx",  cx,  "coordinate of the optical center");
		r.property("Cy",  cy,  "coordinate of the optical center");
		DistortionParameters::reflect(r);
	}

public:

	/**
	 * Returns the camera calibration matrix of the rectified images
	 * with the given size
	 */
	Eigen::Matrix3f getK() const;

	/**
	 * Returns the inverse camera calibration matrix of the rectified
	 * images with the given size
	 */
	Eigen::Matrix3f getK_inv() const;

public:

	bool operator==(const PinholeCameraIntrinsic& other) const {
		return ((fx == other.fx) && (fy == other.fy) && (cx == other.cx) && (cy == other.cy) && ((DistortionParameters)*this == (DistortionParameters)other));
	}

public:
	// the intrinsic parameters
	float fx; ///<coordinate of the focal point in x [pixel]
	float fy; ///<coordinate of the focal point in y [pixel]
	float cx; ///< coordinate of the optical center in x [pixel]
	float cy; ///< coordinate of the optical center in y [pixel]
};

///////////////////////////////////////////////////////////////////////////////

}} // namespace

#endif
