#!/bin/bash

###############################################################################
#
# A bash script, which creates a new release of a package. The scripts does
# the following steps:
#   - put a release date to MIRA_CHANGELOG in the package file
#   - ask the user for verifying the modified package file
#   - commit the package file
#   - create a tag for the new release
#   - add a mountdir.xml for the new tag
#
# Usage:  releasePackage.sh PathToPackage PackageName
#
#     PathToPackage:  The path to package directory
#     PackageName:    The name of the package (without .package extension!)
#
# Author: Christian Martin
#
###############################################################################

confirm()
{
	echo -n "$1 ? (n) "
	read ans
	case "$ans" in
	y|Y|yes|YES|Yes) return 0 ;;
	*) return 1 ;;
	esac
}

###############################################################################
# Extract and check the command line arguments

if [ $# != 2 ]; then
	echo "Usage: $0 PathToPackage PackageName"
	exit
fi

SRCDIR=$1
PKGNAME=$2

if [ ! -e $SRCDIR ]; then
	echo "The directory $SRCDIR does not exist. Abort!"
	exit
fi

if [ ! -f $SRCDIR/$PKGNAME.package ]; then
	echo "The file $SRCDIR/$PKGNAME.package does not exist. Abort!"
	exit
fi

###############################################################################

MSG_PREFIX="Auto-commit by releasePackage.sh: "

# Perform a SVN update
echo "Update SVN for directory $SRCDIR..."
svn update $SRCDIR

# Extract the version number from the package file
PKG_VER=`cat $SRCDIR/$PKGNAME.package | grep MIRA_VERSION | \
	sed -r "s/MIRA_VERSION\([ \t]*([0-9]*)[ \t]*([0-9]*)[ \t]*([0-9]*)\)/\1.\2.\3/g"`

###############################################################################
# First, extract all relevant information

SVN_URL=`svn --xml info $SRCDIR | xml_grep --text_only 'url'`
SVN_ROOT=`svn --xml info $SRCDIR | xml_grep --text_only 'root'`

# remove $SVN_ROOT from the beginning of $SVN_URL
SVN_REL=${SVN_URL#${SVN_ROOT}/}

# get the sub-dir with the repository and check, if it starts with "branches"
PKG_REL_PATH=`dirname ${SVN_REL}`
if [[ ! $PKG_REL_PATH == branches/* ]] ; then
	echo "ERROR: Don't know how to handle sub-dir '$PKG_REL_PATH'."
	exit
fi

# remove "branches" from beginning of PKG_REL_PATH
MOUNT_DIR=${PKG_REL_PATH#branches/}

# get the directory name of the package
PKG_DIR_NAME=`basename ${SVN_REL}`

# create name of the tag
TAG_DIR_NAME=$SVN_ROOT/tags/$MOUNT_DIR/$PKG_DIR_NAME-$PKG_VER/$PKG_DIR_NAME
MOUNT_DIR_DEST=$SVN_ROOT/tags/$MOUNT_DIR/$PKG_DIR_NAME-$PKG_VER

###############################################################################
# Show the information to the user and let the user confirm them

echo ""
echo "Please verify the following information:"
echo "  Name of package  : $PKGNAME"
echo "  Version          : $PKG_VER"
#echo "  SVN              : $SVN_URL"
echo "  SVN root URL     : $SVN_ROOT"
#echo "  SVN rel          : $SVN_REL"
echo "  Path to package  : $PKG_REL_PATH"
echo "  Package dir name : $PKG_DIR_NAME"
echo ""
echo "  MountDir         : $MOUNT_DIR/$PKG_DIR_NAME"
echo "  TagDir           : $TAG_DIR_NAME"
echo ""

if ! confirm "Are this information correct" ; then
	echo "Abort"
	exit
fi

###############################################################################
# Modify the changelog file for the new release

# Put the release date into the package file
DATESTR=`date +%Y-%m-%d`
cat $SRCDIR/$PKGNAME.changelog | \
	sed "0,/Version $PKG_VER.*/s//Version $PKG_VER (Release date: $DATESTR)/" \
	> $SRCDIR/$PKGNAME.changelog.new

# Lets verify the changelog file by the user
echo ""
read -p "Please check the updated changelog file (Enter)..."
less $SRCDIR/$PKGNAME.changelog.new
echo ""
if ! confirm "Is the changelog file correct" ; then
	rm $SRCDIR/$PKGNAME.changelog.new
	echo "Abort"
	exit
fi

# Lets verify the package file by the user
echo ""
read -p "Please check the package file (Enter)..."
less $SRCDIR/$PKGNAME.package
echo ""
if ! confirm "Is the package file correct" ; then
	echo "Abort"
	exit
fi

# Update the changelog file
mv $SRCDIR/$PKGNAME.changelog.new $SRCDIR/$PKGNAME.changelog

###############################################################################
# Create and add the mountdir.xml

if [ ! -f $SRCDIR/mountdir.xml ]; then

	echo "<root>" > $SRCDIR/mountdir.xml
	echo "  <path>$MOUNT_DIR/$PKG_DIR_NAME</path>" >> $SRCDIR/mountdir.xml
	echo "</root>" >> $SRCDIR/mountdir.xml

	echo ""
	echo "Adding mountdir.xml..."
	svn add $SRCDIR/mountdir.xml

else
	echo ""
	echo "Skipped mountdir.xml, since it already exists."
fi

###############################################################################
# Show "svn stat" and let the user confirm the status

echo ""
echo "Please verify the current SVN status:"
echo ""
svn stat $SRCDIR
echo ""
if ! confirm "Continue" ; then
	echo "Abort"
	exit
fi

###############################################################################
# Commit the package file to the SVN

echo ""
echo "Commiting new release..."
svn commit -m "$MSG_PREFIX Released $PKGNAME-$PKG_VER" $SRCDIR

###############################################################################
# Tag the new version

echo ""
echo "Creating tag for new release..."
svn copy -m "$MSG_PREFIX Created tag for $PKGNAME-$PKG_VER" -r HEAD --parents $SVN_URL $TAG_DIR_NAME

