/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization3DContainer.C
 *    Implementation of Visualization3DContainer.h.
 *
 * @author Erik Einhorn
 * @date   2012/09/01
 */

#include <visualization/Visualization3DContainer.h>
#include <widgets/PropertyEditor.h>

#include <serialization/PropertySerializer.h>

#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Visualization3DContainer::Visualization3DContainer(QWidget* parent) :
	OgreWidgetEx(parent), mCameraAcquired(false), mEditor(NULL)
{
	mFixedFrame = "/GlobalFrame";
	mCameraFrame = "/GlobalFrame";
}

Visualization3DContainer::~Visualization3DContainer()
{
	foreach(auto& p, mVisualizations)
		delete p.first;
}

void Visualization3DContainer::initOgre()
{
	OgreWidgetEx::initOgre();
	setupVisualizations();
}

void Visualization3DContainer::setupVisualizations()
{
	assert(getSceneManager()!=NULL);

	// initialize all exisiting visualizations
	foreach(auto& p, mVisualizations)
		p.first->init(this);


	ClassProxy c = ClassFactory::instance().getClassByIdentifier("mira::CameraOrbitTool3D");
	mActiveTool.reset(c.newInstance<VisualizationTool3D>());
	mActiveTool->init(this);
	mActiveTool->activate();


	mLastUpdateTime = Time::now();
	startTimer(50);
}

Visualization3D* Visualization3DContainer::addVisualization(const std::string& visualizationClassName)
{
	ClassProxy c = ClassFactory::instance().getClassByIdentifier(visualizationClassName);
	Visualization3D* vis = c.newInstance<Visualization3D>();

	std::string name = c.getMetaInfo("Name");
	if(name.empty())
		name = c.getIdentifier(); // use class name, if no meta name was specified;
	vis->setName(name);
	addVisualization(vis);

	return vis;
}

void Visualization3DContainer::addVisualization(Visualization3D* vis)
{
	if(getSceneManager()!=NULL)
		vis->init(this);
	// otherwise it is initialized in setupVisualizations()

	PropertySerializer s;
	PropertyNode* p = s.reflectProperties(vis->getName(), vis);

	mVisualizations[vis] = boost::shared_ptr<PropertyNode>(p);

	if(mEditor) {
		mEditor->addProperty(p, QVariant::fromValue((void*)vis));
		mEditor->setColor(p, PropertyEditor::lightGreen);
	}

}

std::string Visualization3DContainer::getPropertyString(Visualization3D* vis, const std::string& property)
{
	auto it = mVisualizations.find(vis);
	if(it==mVisualizations.end())
		MIRA_THROW(XInvalidParameter, "Visualization is not part of this container");

	boost::shared_ptr<PropertyNode> p = it->second;
	PropertyNode* c = p->findChildNode(property);
	if(c==NULL)
		MIRA_THROW(XInvalidParameter, "The property '" << property << "' does not exist in the visualization");

	return c->getAsString();
}

void Visualization3DContainer::setPropertyString(Visualization3D* vis, const std::string& property, const std::string& value)
{
	auto it = mVisualizations.find(vis);
	if(it==mVisualizations.end())
		MIRA_THROW(XInvalidParameter, "Visualization is not part of this container");

	boost::shared_ptr<PropertyNode> p = it->second;
	PropertyNode* c = p->findChildNode(property);
	if(c==NULL)
		MIRA_THROW(XInvalidParameter, "The property '" << property << "' does not exist in the visualization");

	c->setFromString(value);
}

PropertyEditor* Visualization3DContainer::getPropertyEditor(QWidget* parent)
{
	if(mEditor!=NULL)
		return mEditor;

	// create the editor
	mEditor = new PropertyEditor(parent);

	foreach(auto& p, mVisualizations)
	{
		mEditor->addProperty(p.second.get(), QVariant::fromValue((void*)p.first));
		mEditor->setColor(p.second.get(), PropertyEditor::lightGreen);
	}

	return mEditor;
}

Ogre::SceneManager* Visualization3DContainer::getSceneManager()
{
	return mSceneManager;
}

void Visualization3DContainer::registerInteractionListener(InteractionListener3D* listener)
{
	boost::mutex::scoped_lock lock(mListenerMutex);
	mListeners.insert(listener);
}

void Visualization3DContainer::removeInteractionListener(InteractionListener3D* listener)
{
	boost::mutex::scoped_lock lock(mListenerMutex);
	mListeners.erase(listener);
}

std::set<InteractionListener3D*> Visualization3DContainer::getInteractionListeners()
{
	boost::mutex::scoped_lock lock(mListenerMutex);
	return mListeners;
}

const Visualization3DContainer::Camera& Visualization3DContainer::getCamera() const
{
	// waaah
	mCameraSettings.position = OgreWidgetEx::mCameraSettings.position;
	mCameraSettings.yaw = OgreWidgetEx::mCameraSettings.yaw;
	mCameraSettings.pitch = OgreWidgetEx::mCameraSettings.pitch;
	mCameraSettings.roll = OgreWidgetEx::mCameraSettings.roll;
	mCameraSettings.distance = OgreWidgetEx::mCameraSettings.distance;
	return mCameraSettings;
}

void Visualization3DContainer::setCamera(const Camera& camera)
{
	if (mCameraAcquired)
		return;

	OgreWidgetEx::setCamera(camera.position, camera.yaw, camera.pitch, camera.roll, camera.distance);
	this->updateGL();
}

Ogre::Camera* Visualization3DContainer::acquireCamera(boost::function<void()> lostCameraCallback)
{
	assert(mCamera!=NULL);

	if(mCameraAcquired && mLostCameraCallback)
		mLostCameraCallback();

	mLostCameraCallback = lostCameraCallback;
	mCameraAcquired = true;

	return mCamera;
}

void Visualization3DContainer::releaseCamera()
{
	mCameraAcquired = false;
	mLostCameraCallback.clear();
}

const std::string& Visualization3DContainer::getFixedFrame() const
{
	return mFixedFrame;
}

const std::string& Visualization3DContainer::getCameraFrame() const
{
	return mCameraFrame;
}

void Visualization3DContainer::setFixedFrame(const std::string& frame)
{
	mFixedFrame = frame;
}

void Visualization3DContainer::setCameraFrame(const std::string& frame)
{
	mCameraFrame = frame;
}

///////////////////////////////////////////////////////////////////////////////

void Visualization3DContainer::mousePressEvent(QMouseEvent* e) {
	if(mActiveTool!=NULL)
		mActiveTool->onMousePressed(e);
}

void Visualization3DContainer::mouseReleaseEvent(QMouseEvent* e) {
	if(mActiveTool!=NULL)
		mActiveTool->onMouseReleased(e);
}

void Visualization3DContainer::mouseMoveEvent(QMouseEvent* e)
{
	if(mActiveTool!=NULL)
		mActiveTool->onMouseMoved(e);
}

void Visualization3DContainer::wheelEvent(QWheelEvent * e)
{
	if(mActiveTool!=NULL)
		mActiveTool->onMouseWheel(e);
}

void Visualization3DContainer::timerEvent(QTimerEvent* e)
{
	Time time = Time::now();
	Duration dt = time - mLastUpdateTime;
	mLastUpdateTime = time;

	foreach(auto& p, mVisualizations)
		p.first->processUpdate(dt);

	if(mActiveTool)
		mActiveTool->processUpdate(dt);

	this->updateGL();
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

