/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Visualization2DContainer.C
 *    Implementation of Visualization2DContainer.h.
 *
 * @author Christian Reuther
 * @date   2014/09/08
 */

#include <visualization/Visualization2DContainer.h>

#include <widgets/PropertyEditor.h>
#include <serialization/PropertySerializer.h>
#include <error/Exceptions.h>

#include <QVBoxLayout>
#include <QDebug>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

Visualization2DContainer::Visualization2DContainer(QWidget* parent) : QGraphicsView(parent),
		mScene(NULL), mEditor(NULL)
{
	mBackgroundColor = Qt::white;
	mFixedFrame = "/GlobalFrame";
	mCameraFrame = "/GlobalFrame";

	mScene = new QGraphicsScene();
	setBackgroundColor(mBackgroundColor);

	setScene(mScene);
	setViewportUpdateMode(QGraphicsView::FullViewportUpdate);
	setOptimizationFlag(QGraphicsView::DontSavePainterState, true);
	setAlignment(Qt::AlignCenter);
	setAcceptDrops(false);
	setDragMode(QGraphicsView::NoDrag);
	setInteractive(true);
	setFrameStyle(QGraphicsView::NoFrame);
	setMouseTracking(true); // Necessary to receive mouseMoveEvents (at all)
	setHorizontalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setVerticalScrollBarPolicy(Qt::ScrollBarAlwaysOff);
	setTransformationAnchor(QGraphicsView::NoAnchor);
	setSceneRect(-1.0e10,-1.0e10,2.0*1e10,2.0*1.0e10);

	setupVisualizations();
}

Visualization2DContainer::~Visualization2DContainer() {
	foreach(auto& p, mVisualizations)
		delete p.first;
}

void Visualization2DContainer::setupVisualizations() {
	// initialize all existing visualizations
	foreach(auto& p, mVisualizations)
		p.first->init(this);

	ClassProxy c = ClassFactory::instance().getClassByIdentifier("mira::CameraOrbitTool2D");
	mActiveTool.reset(c.newInstance<VisualizationTool2D>());
	mActiveTool->init(this);
	mActiveTool->activate();

	mLastUpdateTime = Time::now();
	startTimer(50);
}

Visualization2D* Visualization2DContainer::addVisualization(const std::string& visualizationClassName) {
	ClassProxy c = ClassFactory::instance().getClassByIdentifier(visualizationClassName);
	Visualization2D* vis = c.newInstance<Visualization2D>();

	std::string name = c.getMetaInfo("Name");
	if(name.empty())
		name = c.getIdentifier(); // use class name, if no meta name was specified;

	vis->setName(name);
	addVisualization(vis);

	return vis;
}

void Visualization2DContainer::addVisualization(Visualization2D* vis) {
	if(getSceneManager())
		vis->init(this);
	// otherwise it is initialized in setupVisualizations()

	PropertySerializer s;
	PropertyNode* p = s.reflectProperties(vis->getName(), vis);

	mVisualizations[vis] = boost::shared_ptr<PropertyNode>(p);

	if(mEditor) {
		mEditor->addProperty(p, QVariant::fromValue((void*)vis));
		mEditor->setColor(p, PropertyEditor::lightGreen);
	}
}

void Visualization2DContainer::removeVisualization(Visualization2D* vis)
{
	if(!vis || !mVisualizations.count(vis))
		return;
	mVisualizations.erase(vis);
}

std::string Visualization2DContainer::getPropertyString(Visualization2D* vis, const std::string& property) {
	auto it = mVisualizations.find(vis);
	if(it == mVisualizations.end())
		MIRA_THROW(XInvalidParameter, "Visualization is not part of this container");

	boost::shared_ptr<PropertyNode> p = it->second;
	PropertyNode* c = p->findChildNode(property);
	if(!c)
		MIRA_THROW(XInvalidParameter, "The property '" << property << "' does not exist in the visualization");

	return c->getAsString();
}

void Visualization2DContainer::setPropertyString(Visualization2D* vis, const std::string& property, const std::string& value) {
	auto it = mVisualizations.find(vis);
	if(it == mVisualizations.end())
		MIRA_THROW(XInvalidParameter, "Visualization is not part of this container");

	boost::shared_ptr<PropertyNode> p = it->second;
	PropertyNode* c = p->findChildNode(property);
	if(!c)
		MIRA_THROW(XInvalidParameter, "The property '" << property << "' does not exist in the visualization");

	c->setFromString(value);
}

PropertyEditor* Visualization2DContainer::getPropertyEditor(QWidget* parent) {
	if(mEditor)
		return mEditor;

	// create the editor
	mEditor = new PropertyEditor(parent);

	foreach(auto& p, mVisualizations) {
		mEditor->addProperty(p.second.get(), QVariant::fromValue((void*)p.first));
		mEditor->setColor(p.second.get(), PropertyEditor::lightGreen);
	}

	return mEditor;
}

////////////////////////////////////////////////////////////////////////////////

QGraphicsScene* Visualization2DContainer::getSceneManager() {
	return mScene;
}

QGraphicsView* Visualization2DContainer::getViewManager() {
	return this;
}

const Visualization2DContainer::Camera& Visualization2DContainer::getCamera() const {
	return mCameraSettings;
}

void Visualization2DContainer::setCamera(const Camera& camera) {
	mCameraSettings = camera;
	updateCamera();
}

float Visualization2DContainer::getToolBaseZValue() const {
	return 100.0f; // Arbitrary but should do
}

const std::string& Visualization2DContainer::getFixedFrame() const {
	return mFixedFrame;
}

const std::string& Visualization2DContainer::getCameraFrame() const {
	return mCameraFrame;
}

void Visualization2DContainer::setFixedFrame(const std::string& frame) {
	mFixedFrame = frame;
}

void Visualization2DContainer::setCameraFrame(const std::string& frame) {
	mCameraFrame = frame;
}

void Visualization2DContainer::setBackgroundColor(const QColor& color) {
	mScene->setBackgroundBrush(QBrush(color));
	mBackgroundColor = color;
}

void Visualization2DContainer::fitRectInView(const QRectF& rect)
{
	if(rect.isNull())
		return;

	// This code is copied directly from QGraphicsView::fitInView. The thing is that we can't
	// use the method because it sets the transform manually (by calling QGraphicsView::scale)
	// and this, combined with the possible rotation of the view, makes it impossible for us
	// to determine the proper camera settings afterwards...I think.
	const QRectF unity = matrix().mapRect(QRectF(0, 0, 1, 1));
	scale(1 / unity.width(), 1 / unity.height());

	const QRectF viewRect = viewport()->rect().adjusted(2, 2, -2, -2);
	const QRectF sceneRect = matrix().mapRect(rect);
	const qreal scaleFactor = qMin(viewRect.width() / sceneRect.width(), viewRect.height() / sceneRect.height()); // KeepAspectRatio
	scale(scaleFactor, scaleFactor);
	centerOn(rect.center());

	// Set the scale of our camera so that it matches the scale of the QGraphicsView / its transform,
	// otherwise the cyclic updateCamera will undo everything in here
	Camera cam = getCamera();
	cam.scale = scaleFactor;
	setCamera(cam);
}

////////////////////////////////////////////////////////////////////////////////

void Visualization2DContainer::updateCamera() {
	QTransform t;

	// By inverting the scale of the y axis, we flip the view vertically.
	// This allows us to have the same coordinate system as the 3D view.
	// Only caveat is that some calculations need to be adjusted to account
	// for this, e.g. in the 2D view tools.
	t.scale(mCameraSettings.scale, -mCameraSettings.scale);
	t.rotate(mCameraSettings.orientation.value(), Qt::ZAxis);
	t.translate(mCameraSettings.position(0), mCameraSettings.position(1));

	setTransform(t);
}

///////////////////////////////////////////////////////////////////////////////

void Visualization2DContainer::mousePressEvent(QMouseEvent* e) {
	if(mActiveTool!=NULL)
		mActiveTool->onMousePressed(e);
}

void Visualization2DContainer::mouseReleaseEvent(QMouseEvent* e) {
	if(mActiveTool!=NULL)
		mActiveTool->onMouseReleased(e);
}

void Visualization2DContainer::mouseMoveEvent(QMouseEvent* e) {
	if(mActiveTool!=NULL)
		mActiveTool->onMouseMoved(e);
}

void Visualization2DContainer::wheelEvent(QWheelEvent * e) {
	if(mActiveTool!=NULL)
		mActiveTool->onMouseWheel(e);
}

void Visualization2DContainer::timerEvent(QTimerEvent* e) {
	Time time = Time::now();
	Duration dt = time - mLastUpdateTime;
	mLastUpdateTime = time;

	foreach(auto& p, mVisualizations)
		p.first->processUpdate(dt);

	if(mActiveTool)
		mActiveTool->processUpdate(dt);

	updateCamera();
	processUpdate(dt);
}

///////////////////////////////////////////////////////////////////////////////

}
