/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeshFactory.C
 *    Singleton that allows to load arbitrary meshes.
 *
 * @author Erik Einhorn
 * @date   2011/12/16
 */

#include <boost/algorithm/string.hpp>

#include <error/LoggingCore.h>

#include <utils/ToString.h>
#include <utils/Path.h>

#include <visualization/3d/MeshFactory.h>
#include <visualization/3d/MeshLoader.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

std::vector<MeshFactory::MeshFormatInfo> MeshFactory::getSupportedFormats()
{
	std::vector<MeshFormatInfo> formats;

	typedef std::map<std::string, ClassProxy> ClassMap;
	ClassMap meshLoaders = MeshLoader::CLASS().getDerivedClasses();
	foreach(ClassMap::value_type& i, meshLoaders)
	{
		if(i.second.isAbstract())
			continue;
		std::string supportedExtensions = i.second.getMetaInfo("Extension");
		std::vector<std::string> extensions;
		boost::split(extensions, supportedExtensions, boost::is_from_range(';',';'));
		if (extensions.empty())
			continue;

		foreach(const std::string extension, extensions)
		{
			std::vector<std::string> components;
			boost::split(components, extension, boost::is_from_range(',',','));
			if(components.size()!=2) {
				MIRA_LOG(WARNING) << "Invalid format for extension. "
				                     "Should be 'Description,.ext'";
				continue;
			}

			formats.push_back(MeshFormatInfo(components[0],components[1]));
		}
	}

	return formats;
}

Ogre::MeshPtr MeshFactory::loadMesh(const std::string& filename)
{
	typedef std::map<std::string, ClassProxy> ClassMap;
	ClassMap meshLoaders = MeshLoader::CLASS().getDerivedClasses();
	ClassProxy* loader = NULL;
	foreach(ClassMap::value_type& i, meshLoaders)
	{
		if(i.second.isAbstract())
			continue;
		std::string supportedExtensions = i.second.getMetaInfo("Extension");
		std::vector<std::string> extensions;
		boost::split(extensions, supportedExtensions, boost::is_from_range(';',';'));
		if (extensions.empty())
			continue;

		foreach(const std::string extension, extensions)
		{
			std::vector<std::string> components;
			boost::split(components, extension, boost::is_from_range(',',','));
			if(components.size()!=2) {
				MIRA_LOG(WARNING) << "Invalid format for extension: '" << extension
				                  << "'. Should be: 'Description','.ext'";
				continue;
			}

			if(filename.size()<components[1].size())
				continue;
			// extract the last chars as extension
			// (same length as the extension we are comparing with)
			std::string ext = filename.substr(filename.size()-components[1].size(),
			                                  components[1].size());
			// and check for equality (case insenstive)
			if (boost::iequals(components[1],ext))
			{
				loader = &i.second;
				break;
			}
		}
		if (loader != NULL)
			break;
	}
	if (loader == NULL)
		MIRA_THROW(XInvalidConfig, "Mesh file format is not supported - "
		           "No loader found for mesh '" << filename << "'")

	MeshLoader* l = loader->newInstance<MeshLoader>();
	Ogre::MeshPtr mesh = l->loadMesh(filename);
	delete l;

	return mesh;
}

///////////////////////////////////////////////////////////////////////////////

}
