/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PropertyEditor.h
 *    Declaration of PropertyEditor.
 *
 * @author Erik Einhorn
 * @date   2010/12/12
 */

#ifndef _MIRA_PROPERTYEDITOR_H_
#define _MIRA_PROPERTYEDITOR_H_

#include <QVariant>
#include <QModelIndex>
#include <QIcon>
#include <QColor>
#include <QWidget>
#include <QLineEdit>

#include <serialization/PropertyManager.h>
#include <utils/EnumToFlags.h>

#include <widgets/GuiWidgetsExports.h>
#include <widgets/TreeViewFilter.h>

class QToolButton;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class PropertyItemEditor;

class MIRA_GUI_WIDGETS_EXPORT PropertyEditor : public QWidget
{
	Q_OBJECT

public:
	PropertyEditor(QWidget *parent = 0);
	virtual ~PropertyEditor();

public:

	class Delegate;
	typedef boost::shared_ptr<Delegate> DelegatePtr;

public:

	/// Returns the current selected property node, or NULL if nothing is selected.
	PropertyNode* currentProperty();

public slots:

	/**
	 * Removes all property nodes that were set and therefore clears the whole
	 * editor content (without deleting the nodes).
	 */
	void clear();

	/// Adds the specified property node as top level node to the editor.
	void addProperty(PropertyNode* property, QVariant user = QVariant());

	/// Remove the specified property node from the editor.
	void removeProperty(PropertyNode* property);

	/// Moves the property node up in the editor's property node list.
	void moveUpProperty(PropertyNode* property);

	/// Moves the property node down in the editor's property node list.
	void moveDownProperty(PropertyNode* property);

	QVariant getUser(PropertyNode* property);

public:

	/**
	 * If hiding of the single root node is enabled, a single top level node
	 * will not be shown explicitly. Its children will be shown only.
	 */
	void setHideSingleRootNode(bool hide = true);

	/// Returns the set mode, whether a single root node should be hidden or not.
	bool getHideSingleRootNode() const;

public:

	void installFilterShortCut(QWidget* widget, const QString& shortcut = "Ctrl+F");

	bool getAutoHideFilterBar() const;

public slots:

	void setAutoHideFilterBar(bool hide = true);
	void showFilter();
	void hideFilter();

public:

	/// Sets the background color for the specified property
	void setColor(const PropertyNode* property, const QColor& color);

	QColor getColor(const PropertyNode* property) const;

public:

	std::string getText(const PropertyNode* property) const;
	QIcon getIcon(const PropertyNode* property) const;
	QColor getBackgroundColor(const PropertyNode* property) const;
	QWidget* createEditor(PropertyNode* property, QWidget* parent);
	std::string getName(const PropertyNode* property);

protected:

	void timerEvent(QTimerEvent*);

private slots:

	void onDestroyedEditorDelegate(QObject*);

public:

	const PropertyNode* getCurrentEditedProperty() const { return mCurrentEditedProperty; }

public:

	const static QColor lightBlue;
	const static QColor lightRed;
	const static QColor lightGreen;
	const static QColor lightYellow;
	const static QColor lightPurple;

private:

	class DelegateRegistry;
	class Model;
	class TreeView;
	class ItemDelegate;

	//friend class DelegateRegistry;
	friend class Model;
	friend class TreeView;
	friend class ItemDelegate;
	friend class PropertyItemEditor;

	TreeView* mTreeView;
	Model* mModel;
	ItemDelegate* mItemDelegate;

	TreeViewFilter* mFilter;

	std::map<const PropertyNode*, QColor> mColorMap;

	int mUpdateTimerId;
	bool mHideSingleRootNode;

	PropertyNode* mCurrentEditedProperty;
};

///////////////////////////////////////////////////////////////////////////////

/**
 * Provides display and editing facilities for property items in the editor.
 * A PropertyEditor::Delegate is used to provide custom display features
 * and editor widgets. Using the delegate mechanism allows the development
 * of display and editing facilities independently from the
 * PropertyEditor.
 * PropertyEditor::Delegate classes can be implemented in different
 * libraries. The PropertyEditor automatically asks the ClassFactory for
 * loaded Delegates and instantiates them.
 */
class MIRA_GUI_WIDGETS_EXPORT PropertyEditor::Delegate : public Object
{
	MIRA_OBJECT(Delegate)
public:

	struct SupportedType;
	typedef std::list<SupportedType> SupportedTypes;

	struct SupportedType
	{
		enum Flags {
			/// set if the Delegate provides at least a text for a label
			TEXT   = 0x01,

			/// set if the Delegate provides an icon
			ICON   = 0x02,

			/// set if the Delegate provides an background color
			COLOR  = 0x04,

			/// set if the Delegate can provide an editor for the data
			EDITOR = 0x08,

			/// set if the Delegate can provide a dialog for editing the data
			DIALOG = 0x10,

			/// set if the Delegate provides a (dynamic) name
			NAME = 0x20,
		};
		MIRA_ENUM_TO_FLAGS_INCLASS(Flags);


		SupportedType(const Typename& iType, Flags iFlags) :
			type(iType), flags(iFlags) {}

		Typename type; // the typename of the type

		Flags flags;

		operator SupportedTypes()
		{
			SupportedTypes list;
			list.push_back(*this);
			return list;
		}

		friend SupportedTypes operator+(SupportedTypes list, const SupportedType& type)	{
			list.push_back(type);
			return list;
		}
	};

	template <typename Type>
	static SupportedType makeSupportedType(SupportedType::Flags flags) {
		return SupportedType(typeName<Type>(), flags);
	}

	virtual SupportedTypes supportedTypes() const = 0;

	/**
	 * Returns text for a text label.
	 */
	virtual std::string getText(const PropertyNode* property) { return std::string(); }

	virtual QIcon getIcon(const PropertyNode* property) { return QIcon(); }

	virtual QColor getBackgroundColor(const PropertyNode* property) { return QColor(); }

	virtual QWidget* createEditor(PropertyNode* property, QWidget* parent) { return NULL; }

	virtual bool execDialog(PropertyNode* property, QWidget* parent) { return false; }

	virtual std::string getName(const PropertyNode* property) { return property->name(); }
};

///////////////////////////////////////////////////////////////////////////////

class MIRA_GUI_WIDGETS_EXPORT PropertyItemEditor : public QWidget
{
	Q_OBJECT
public:

	PropertyItemEditor(PropertyEditor* editor, QWidget* parent,
	                   PropertyNode* property,
	                   PropertyEditor::DelegatePtr editorDelegate,
	                   PropertyEditor::DelegatePtr dialogDelegate);

	bool eventFilter(QObject *obj, QEvent *e);

private slots:

	void buttonClicked();

private:

	PropertyEditor* mEditor;
	PropertyNode* mProperty;
	QToolButton*  mButton;
	PropertyEditor::DelegatePtr mDialogDelegate;

};

///////////////////////////////////////////////////////////////////////////////

}

#endif
