/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapePlayerUnit.C
 *    Unit that acts as an interface to a tape player
 *
 * @author Tim Langner
 * @date   2013/01/29
 */

#include <fw/Unit.h>
#include <fw/TapePlayer.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Unit that exposes access to a TapePlayer via RPC.
 * It allows automatic play back of a tape via config file.
 */
class TapePlayerUnit : public Unit
{
	MIRA_OBJECT(TapePlayerUnit)
public:
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		Unit::reflect(r);
		r.member("AutoPlayFile", mAutoFile, "File(s) for automatic play back at startup",
		         std::vector<std::string>());
		r.member("PrePlay Calls", mPrePlayCalls,
		         "Service methods to be called before starting playback of a tape",
		         std::vector<std::string>());
		r.member("PostPlay Calls", mPostPlayCalls,
		         "Service methods to be called after finishing playback of a tape",
		         std::vector<std::string>());

		r.property("NamespacePrefix", mNamespacePrefix, "The prefix for tape channels", "");
		r.property("UseOriginalTimestamp", mUseOriginalTimestamp,
		           "Use the original recording time as base for play back", false);
		r.property("TimeScale", mTimeScale, setter<float>(&TapePlayerUnit::setTimeScale, this),
		           "Time scale for play back", 1.0f);
		r.property("Loop", mLoop, "Is the tape played back in loop mode", false);
		r.property("LoopCount", mLoopCount, "How often is the tape played back in loop mode", 1);
		r.method("load", &TapePlayerUnit::load, this, "Load a tape",
		         "filename", "name of tape file", "/home/user/1.tape");
		r.method("play", &TapePlayerUnit::play, this, "Play the loaded tape(s)");
		r.method("step", &TapePlayerUnit::step, this, "Play back next message");
		r.method("stepToTime", &TapePlayerUnit::stepToTime, this, "Play back all messages with timestamp < time",
		         "time", "time (offset) to play to", Duration::milliseconds(1200));
		r.method("stepToChannel", &TapePlayerUnit::stepToChannel, this,
		         "Play back all messages until one with channelID is reached",
		         "channelID", "channel ID to play to", "/robot/Mileage");
		r.method("jump", &TapePlayerUnit::jump, this, "Skip all messages and start at given time",
		         "time", "time (offset) to jump to", Duration::milliseconds(10000));
		r.method("pause", &TapePlayerUnit::pause, this, "Pause play back");
		r.method("stop", &TapePlayerUnit::stop, this, "Stop play back");

		r.roproperty("Playing", getter(&TapePlayerUnit::isPlaying, this), "Is the unit currently playing a tape?");
		r.roproperty("Paused", getter(&TapePlayerUnit::isPaused, this), "Is the unit currently paused?");
		r.roproperty("PlayingFile", mPlayingFile, "File currently playing");
	}

	TapePlayerUnit();

	virtual void initialize();

	virtual void process(const Timer& timer);

	void load(const std::string& filename);
	void play();
	void step();
	void stepToTime(Duration time);
	void stepToChannel(const std::string& channelID);
	void jump(Duration time);
	virtual void pause();
	virtual void stop();

	void setTimeScale(float scale);
	void setLoop(bool loop);

	bool isPlaying() const { return mPlayer && mPlayer->isPlaying(); }
	bool isPaused() const { return mPlayer && mPlayer->isPaused(); }

protected:

	void playNext();
	void callServices(const std::vector<std::string>& services);

	boost::shared_ptr<TapePlayer> mPlayer;
	TapeVisitor mVisitor;
	std::map<std::string, boost::shared_ptr<Tape>> mTapes;
	std::string mNamespacePrefix;
	bool mUseOriginalTimestamp;
	float mTimeScale;
	bool mLoop;
	uint32 mLoopCount;
	Time mLastTime;
	std::vector<std::string> mAutoFile;
	uint mAutoFileIndex;
	std::string mPlayingFile;

	std::vector<std::string> mPrePlayCalls;
	std::vector<std::string> mPostPlayCalls;
};

///////////////////////////////////////////////////////////////////////////////

TapePlayerUnit::TapePlayerUnit() : Unit(Duration::milliseconds(1000))
{
}

void TapePlayerUnit::initialize()
{
	publishService(*this);
	mLastTime = Time::now();

	mAutoFileIndex = 0;
	playNext();
}

void TapePlayerUnit::process(const Timer& timer)
{
	if (isPlaying() || mPlayingFile.empty())
		return;

	callServices(mPostPlayCalls);
	mPlayingFile.clear();
	playNext();
}

void TapePlayerUnit::load(const std::string& filename)
{
	stop();
	mVisitor = TapeVisitor();
	mTapes[filename].reset(new Tape());
	mTapes[filename]->open(filename, Tape::READ);
	mVisitor.visit(mTapes[filename].get());
	mPlayingFile = filename;
	Time n = Time::now();
	if (mLastTime < n)
		mLastTime = n;
	mPlayer.reset(new TapePlayer());
	mPlayer->load(&mVisitor, mUseOriginalTimestamp?mVisitor.getStartTime():mLastTime, mNamespacePrefix);
	mPlayer->setTimeScaler(mTimeScale);
	mPlayer->setLoop(mLoop, mLoopCount);
}

void TapePlayerUnit::play()
{
	if (mPlayer)
		mPlayer->play();
}

void TapePlayerUnit::step()
{
	if (mPlayer)
		mPlayer->step();
}

void TapePlayerUnit::stepToTime(Duration time)
{
	if (mPlayer)
		mPlayer->stepTo(time);
}

void TapePlayerUnit::stepToChannel(const std::string& channelID)
{
	if (mPlayer)
		mPlayer->stepTo(channelID);
}

void TapePlayerUnit::jump(Duration time)
{
	if (mPlayer)
		mPlayer->jumpTo(time);
}

void TapePlayerUnit::pause()
{
	if (mPlayer)
		mPlayer->pause();
}

void TapePlayerUnit::stop()
{
	if (!mPlayer)
		return;
	mPlayer->stop();
	mLastTime = mPlayer->getRelativeStartTime() +
	            mPlayer->getCurrentMessageTimeOffset()+
	            Duration::microseconds(1);
	mPlayer.reset();
	mTapes.clear();
}

void TapePlayerUnit::setTimeScale(float scale)
{
	mTimeScale = scale;
	if (mPlayer)
		mPlayer->setTimeScaler(scale);
}

void TapePlayerUnit::playNext()
{
	if (mAutoFileIndex < mAutoFile.size())
	{
		load(mAutoFile[mAutoFileIndex++]);
		callServices(mPrePlayCalls);
		play();
	}
}

void TapePlayerUnit::callServices(const std::vector<std::string>& services)
{
	foreach(const auto& s, services) {
		size_t p = s.find('.');
		callServiceJSON(s.substr(0, p), s.substr(p+1), "").get();
	}
}
///////////////////////////////////////////////////////////////////////////////

} // namespace

MIRA_CLASS_SERIALIZATION(mira::TapePlayerUnit, mira::MicroUnit);
