/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapeRecorder.C
 *    Implementation of TapeRecorder.h.
 *
 * @author Tim Langner
 * @date   2012/03/09
 */

#include <fw/TapeRecorder.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

TapeRecorder::TapeRecorder() :
	mAuthority(Authority::ANONYMOUS | Authority::INTERNAL)
{
	mAuthority.checkin("/", "TapeRecorder");
	mMaxMessageBlockSize = 64*1024*1024; // default size is 64MB
	mIsRecording = false;
}

TapeRecorder::~TapeRecorder()
{
	stop();
	mAuthority.checkout();
}

void TapeRecorder::addChannel(const std::string& channelID,
                              const std::list<BinarySerializerCodecPtr>& codecs,
                              bool compress, Duration interval, bool avoidDataLoss)
{
	std::string resolvedChannelID = mAuthority.resolveName(channelID);
	if (mChannels.count(resolvedChannelID) > 0)
	{
		MIRA_LOG(WARNING) << "Already recording channel '" << resolvedChannelID << "'. Skipping!";
		return;
	}
	MIRA_LOG(DEBUG) << "Now recording channel '" << resolvedChannelID << "'";
	ChannelInfo info;
	info.compress = compress;
	info.interval = interval;
	info.expected = Time::unixEpoch();
	info.codecs = codecs;
	info.avoidDataLoss = avoidDataLoss;
	mChannels[resolvedChannelID] = info;
	if (avoidDataLoss)
		mAuthority.subscribeInterval<void>(resolvedChannelID, boost::bind(&TapeRecorder::intervalValueChanged, this, _1, false),Duration::seconds(1));
	else
		mAuthority.subscribe<void>(resolvedChannelID, boost::bind(&TapeRecorder::valueChanged, this, _1, false));

}

void TapeRecorder::addChannel(const std::string& channelID,
                              const RecordedChannelInfo& info)
{
	addChannel(channelID, info.codecs, info.compress, info.interval, info.avoidDataLoss);
}

void TapeRecorder::record(const Path& file)
{
	stop();
	mFirstCallbackMessage = false;
	mTape.setMaxMessageBlockSize(mMaxMessageBlockSize);
	Path fileName = file;
	if (fileName.extension().empty())
		fileName.replace_extension(".tape");
	mTape.open(fileName, Tape::WRITE);
	mTape.waitForAlteredStartTime();

	mStorageThreadTerminationRequest = false;
	mStorageThreadStarted = false;
	mStorageThread=boost::thread(&TapeRecorder::storageThread, this);

	while (!mStorageThreadStarted)
		MIRA_SLEEP(10)

	// Initially read the latest data from all channels to have at least
	// one entry in the tape. Used for channels like static transforms,
	// that get not updated and therefore will never change their value
	// so that the callback gets never called.
	ChannelMap::iterator i=mChannels.begin();
	for(; i!=mChannels.end(); ++i)
	{
		i->second.stats.reset();
		i->second.expected = Time::unixEpoch(); // reset the initial expected time
		try
		{
			valueChanged(mAuthority.getChannel<void>(i->first).read(), true);
		}
		catch(Exception&)
		{
			// ignore if there is no data
		}
	}
	mFirstCallbackMessage = true;
	mIsRecording = true;
	mAuthority.start();
}

void TapeRecorder::record(const TapeRecorderConfig& config)
{
	stop();
	foreach(auto& channel, config.channels)
		addChannel(channel.first, channel.second);
	setMaxMessageBlockSize(config.maxMessageBlockSize);
	record(config.filename);
}

void TapeRecorder::stop()
{
	mAuthority.stop();
	mStorageThreadTerminationRequest = true;
	mStorageThread.join();

	// write remaining messages
	while(!mStorageQueue.empty()) {
		storeMessage(mStorageQueue.front());
		mStorageQueue.pop_front();
	}

	mIsRecording = false;
	mTape.close();
}

void gatherMetaData(ChannelRead<void> value, TapeRecorder::ChannelInfo& info)
{
	info.meta = value.getTypeMeta();
	if (info.meta)
	{
		// try to get a subset of meta descriptions that is required to describe the type
		// and all its members. If it fails do not add meta information.
		try
		{
			info.metaDB = MIRA_FW.getMetaDatabase()->getDependentTypesDB(info.meta);
		}
		catch(Exception& ex)
		{
			MIRA_LOG(WARNING) << ex.message();
			info.meta = TypeMetaPtr();
		}
	}
}

void TapeRecorder::valueChanged(ChannelRead<void> value, bool forceStorage)
{
	if (mFirstCallbackMessage)
	{
		mTape.alterStartTime(value->timestamp);
		mFirstCallbackMessage = false;
	}
	ChannelInfo& info = mChannels[value.getChannelID()];

	// handle interval stuff
	Time current = value->timestamp;
	if (current < info.expected && !forceStorage)
		return;

	if(info.interval.totalMilliseconds()>0) {
		if (info.expected<current)
			info.expected += Duration::nanoseconds(info.interval.totalNanoseconds() *
			                                       ( ((current - info.expected+info.interval).totalNanoseconds()-1) /
			                                         info.interval.totalNanoseconds()));
	}

	if (info.recordCounter == 0)
		gatherMetaData(value, info);
	info.recordCounter++;


	// serialize the data
	Buffer<uint8> buf = value.readSerializedValue(info.codecs);

	// insert pending message into storage queue
	boost::mutex::scoped_lock lock(mStorageMutex);
	mStorageQueue.push_back(MessageToStore());
	//std::cout << "CurSize: " << mStorageQueue.size() << std::endl;
	MessageToStore& msg = mStorageQueue.back();

	msg.channelID = value.getChannelID();
	msg.typeName = value.getTypename();
	msg.header = (StampedHeader)(*value);
	msg.enqueueTime = Time::now();
	msg.buf = std::move(buf);
	msg.info = &info;

	mStorageThreadCondition.notify_all();
}

void TapeRecorder::intervalValueChanged(ChannelReadInterval<void> values, bool forceStorage)
{
	for(auto value = values.begin(); value !=values.end(); ++value)
		valueChanged(value, forceStorage);
}

void TapeRecorder::storageThread()
{
	mStorageThreadStarted = true;
	while(!mStorageThreadTerminationRequest)
	{
		boost::mutex::scoped_lock lock(mStorageMutex);
		if(!mStorageThreadCondition.timed_wait(lock, Duration::milliseconds(500)))
			continue;

		lock.unlock();
		while(true)
		{
			boost::mutex::scoped_lock mlock(mStorageMutex);
			if(mStorageQueue.empty())
				break;

			//std::cout << "Size: " << mStorageQueue.size() << std::endl;
			MessageToStore& msg = mStorageQueue.front();
			mlock.unlock();

			storeMessage(msg);

			mlock.lock();
			mStorageQueue.pop_front();
		}

	}
}

void TapeRecorder::storeMessage(MessageToStore& msg)
{

	std::size_t bytes = mTape.write(msg.channelID, msg.typeName,
	                                msg.header, msg.buf,
	                                msg.info->compress, msg.info->meta,
	                                msg.info->metaDB);
	msg.info->stats.update(msg.enqueueTime, bytes);
}


const PerformanceStatistics& TapeRecorder::getStats(const std::string& name)
{
	static PerformanceStatistics sEmpty;
	auto it=mChannels.find(name);
	if(it==mChannels.end())
		return sEmpty;

	return it->second.stats;
}

///////////////////////////////////////////////////////////////////////////////

}
