/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DispatcherThread.C
 *    Implementation of DispatcherThread.h
 *
 * @author Erik Einhorn
 * @date   2010/11/21
 */

#include <fw/DispatcherThread.h>

#include <error/Exceptions.h>
#include <error/Logging.h>
#include <utils/Foreach.h>
#include <utils/ToString.h>
#include <thread/ThreadMonitor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////


bool DispatcherThread::insertRunnable(IRunnablePtr runnable, bool singleton, Time time)
{
	boost::mutex::scoped_lock lock(mConditionMutex);
	auto r = mPendingRunnables.insert(runnable);
	if(singleton && !r.second)
		return false;

	mQueue.push(QueueItem(runnable,time));

	mCondition.notify_one();

	return true;
}

void DispatcherThread::removeRunnable(IRunnablePtr runnable)
{
	boost::mutex::scoped_lock lock(mConditionMutex);
	foreach(QueueItem& i, mQueue)
	{
		if(i.runnable==runnable) {
			i.toBeRemoved = true;
			i.runnable.reset();
		}
	}
	mPendingRunnables.erase(runnable);
}

bool DispatcherThread::processSpin(const Duration& maxWait)
{

	boost::system_time now = boost::get_system_time();
	// Workaround for boost timed_wait which cannot handle infinity or
	// eternity in maxTime, hence we set maxTime far into the future.
	boost::system_time maxTime = maxWait.is_pos_infinity() ?
		now+Duration::hours(65535) : now + maxWait;

	do
	{
		boost::mutex::scoped_lock lock(mConditionMutex);

		if(mQueue.empty()) {
			if(!mCondition.timed_wait(lock, maxTime))
				return true;
		}

		if(mQueue.empty())
			return true;


		// get next runnable
		QueueItem item = mQueue.top();

		// remove inactive items
		if(item.toBeRemoved) {
			mQueue.pop();
			continue;
		}

		boost::system_time invocationTime = item.time;
		boost::system_time waitTime = std::min(maxTime, invocationTime);

		// while waiting for a signal
		if(now < waitTime) { // current time is before waitTime, so we need to wait
			mCondition.timed_wait(lock, waitTime);
		}

		if(isInterruptionRequested())
			return false; // terminate thread if we were interrupted

		now = boost::get_system_time();
		item = mQueue.top();
		invocationTime = item.time;

		if(now>=invocationTime)
		{
			// invoke the runnable
			mQueue.pop();
			mPendingRunnables.erase(item.runnable);

			// unlock the mutex to allow adding and removing runnables while
			// invoking the timer callback
			lock.unlock();

			if(!item.toBeRemoved) {
				try {
					item.runnable->run(this);
				}
				catch(boost::thread_interrupted&) {
					throw;
				}
				catch(XUnrecoverableFailure&) {
					mUnrecoverableFailure = true;
				}
				catch(std::exception& ex)
				{
					MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception:\n";
				}
				catch(...) {
					MIRA_LOG(ERROR) << "Unknown unhandled exception while "
						"processing timer in DispatcherThread '" <<
						mName << "'";
				}
				return true;
			}
		}

	} while(boost::get_system_time()<maxTime);

	return true;
}





///////////////////////////////////////////////////////////////////////////////

DispatcherThread::Timer::Timer(TimerCallback callback,
                               Duration period, Duration tolerance) :
	mCallback(callback),
	mPeriod(period),
	mTolerance(tolerance),
	mOneShot(false),
	mActive(true)
{
	assert(mPeriod.isValid());
	currentExpected = Time::now();
}

DispatcherThread::Timer::Timer(TimerCallback callback, Time invokationTime) :
	currentExpected(invokationTime),
	mCallback(callback),
	mOneShot(true),
	mActive(true)
{
	assert(currentExpected.isValid());
}

void DispatcherThread::Timer::start()
{
	mActive = true;
}

void DispatcherThread::Timer::stop()
{
	mActive = false;
}

void DispatcherThread::Timer::run(DispatcherThread* dispatcher)
{
	// this ensures, that the timer object is not destoyed as long as we run in here
	TimerPtr thisTimer = boost::dynamic_pointer_cast<Timer>(this->shared_from_this());

	Time now = Time::now();
	last    = current;
	current = now;

	if(mActive) {
		// call the timer and catch all exceptions
		try {
			mCallback(*this);
			Time after = Time::now();
			lastDuration = after-now;
		}
		catch(boost::thread_interrupted&) {
			throw;
		}
		catch(XUnrecoverableFailure&) {
			throw;
		}
		catch(std::exception& ex)
		{
			MIRA_LOG_EXCEPTION(ERROR, ex) << "Exception:\n";
		}
		catch(...) {
			MIRA_LOG(ERROR) << "Unknown unhandled exception while processing timer";
		}
	}

	dispatcher->postProcessTimer(thisTimer,now);
}

void DispatcherThread::postProcessTimer(TimerPtr timer, Time now)
{
	boost::mutex::scoped_lock lock(mTimerMutex);
	// remove timer if one shot flag is set
	if(timer->mOneShot)
		mTimers.erase(timer);

	if(mTimers.count(timer) > 0) {
		// special treatment for timers with period = 0
		// since they should run as fast as possible they would
		// shadow other timers since their invocation time will
		// always stay the same - so just set their next time
		// to now
		if (timer->mPeriod == Duration::seconds(0))
			timer->currentExpected = now;
		else
		{
			// if we have already passed the next timers invocation time...
			Duration dt = timer->current - timer->currentExpected;
			if (dt > timer->getTolerance())
			{
				// ...set the expected time to now to avoid
				// accumulation of the time the timer is behind
				timer->currentExpected = now;
			}
		}
		// update next invocation time and reschedule the timer
		timer->updateInvocationTime();
		insertRunnable(timer, false, timer->getNextInvocationTime());
	}
}

///////////////////////////////////////////////////////////////////////////////

DispatcherThread::DispatcherThread(const std::string& name) :
	mUnrecoverableFailure(false),
	mIsRunning(false),
	mInterruptionRequested(false),
	mPendingSignal(false)
{
	setName(name);
}

DispatcherThread::~DispatcherThread()
{
	if(isRunning()) {
		MIRA_LOG(ERROR) << "Destructor of DispatcherThread called, "
		                   "while thread is still running. Call stop() first.";
	}
}

void DispatcherThread::setName(const std::string& name)
{
	mName = name;
	if (mName.empty() )
		mName = "Dispatcher " + toString(this);
}

void DispatcherThread::addImmediateHandler(IRunnablePtr runnable)
{
	boost::mutex::scoped_lock handlerlock(mImmediateHandlerMutex);
	mImmediateHandlers.push_back(runnable);
	// signal
	boost::mutex::scoped_lock lock(mConditionMutex);
	mCondition.notify_one();
}

void DispatcherThread::addFinalizeHandler(IRunnablePtr runnable)
{
	boost::mutex::scoped_lock lock(mFinalizeHandlerMutex);
	mFinalizeHandlers.push_back(runnable);
}


TimerPtr DispatcherThread::createTimer(Duration period, Duration periodTolerance,
                                       TimerCallback callback, bool oneshot)
{
	boost::mutex::scoped_lock lock(mTimerMutex);

	TimerPtr timer;

	if(oneshot)
		timer.reset(new Timer(callback, Time::now()+period));
	else
		timer.reset(new Timer(callback, period, periodTolerance));

	mTimers.insert(timer);
	lock.unlock();
	insertRunnable(timer,false, timer->getNextInvocationTime());
	return timer;
}

TimerPtr DispatcherThread::createTimer(Duration period, TimerCallback callback,
                                       bool oneshot)
{
	return createTimer(period, Duration::invalid(), callback, oneshot);
}

TimerPtr DispatcherThread::createTimer(Time time, TimerCallback callback)
{
	boost::mutex::scoped_lock lock(mTimerMutex);
	TimerPtr timer(new Timer(callback, time));
	mTimers.insert(timer);
	lock.unlock();
	insertRunnable(timer,false,timer->getNextInvocationTime());
	return timer;
}

void DispatcherThread::removeTimer(TimerPtr timer)
{
	if(!timer)
		return;
	boost::mutex::scoped_lock lock(mTimerMutex);
	timer->stop();
	removeRunnable(timer);
	mTimers.erase(timer);
}

void DispatcherThread::start(bool startThread)
{
	if(mUnrecoverableFailure)
		MIRA_THROW(XLogical, "Cannot start dispatcher since a previous call "
		           "to an immediate handler has failed and we are in an "
		           "unrecoverable error state");

	if(mIsRunning)
		return;

	if(startThread) {
		mThread = boost::thread(boost::bind(&DispatcherThread::run, this));
		mThreadId = mThread.get_id();
	} else
		mThreadId = boost::thread::id();

	mIsRunning = true;
}

void DispatcherThread::stop()
{
	if(!mIsRunning)
		return;

	mInterruptionRequested = true;
	if(mThread.get_id() != boost::this_thread::get_id() &&
		mThread.get_id() != boost::thread::id()) {
		mThread.interrupt();
		mThread.join();
		mInterruptionRequested = false; // here we can reset the flag,since we finally terminated
	}

	mThreadId = boost::thread::id(); // set invalid
	mIsRunning = false;
}

bool DispatcherThread::hasUnrecoverableFailure() const
{
	return mUnrecoverableFailure;
}

bool DispatcherThread::isRunning() const
{
	return mIsRunning;
}

bool DispatcherThread::spin(const Duration& maxWait)
{
	if(mThreadId==boost::thread::id())
		mThreadId = boost::this_thread::get_id();
	else {
		if(mThreadId != boost::this_thread::get_id()) {
			MIRA_LOG(WARNING) << "Thread id has changed since last call of spin()";
			mThreadId = boost::this_thread::get_id();
		}
	}

	if(!mImmediateHandlers.empty()) {
		if (!processImmediateHandlers())
			return false;
		return true;
	}

	if(processSpin(maxWait))
		return true;

	processFinalizeHandlers();

	return false;
}

bool DispatcherThread::processImmediateHandlers()
{
	boost::mutex::scoped_lock handlerLock(mImmediateHandlerMutex);

	while(!mImmediateHandlers.empty())
	{
		IRunnablePtr h = mImmediateHandlers.front();
		mImmediateHandlers.pop_front();
		try {
			h->run(this);
		}
		catch(boost::thread_interrupted&) {
			return false;
		}
		catch(XUnrecoverableFailure&) {
			mUnrecoverableFailure = true;
			return false;
		}
		catch(...) {
			MIRA_LOG(ERROR) << "Unknown unhandled exception while processing "
				"immediate handler in DispatcherThread '" << mName << "'";
			mUnrecoverableFailure = true;
			return false;
		}
	}
	return true;
}

bool DispatcherThread::processFinalizeHandlers()
{
	boost::mutex::scoped_lock handlerLock(mFinalizeHandlerMutex);

	while(!mFinalizeHandlers.empty())
	{
		IRunnablePtr h = mFinalizeHandlers.front();
		mFinalizeHandlers.pop_front();
		try {
			h->run(this);
		}
		catch(boost::thread_interrupted&) {
			return false;
		}
		catch(XUnrecoverableFailure&) {
			mUnrecoverableFailure = true;
			return false;
		}
		catch(...) {
			MIRA_LOG(ERROR) << "Unknown unhandled exception while processing "
				"finalize handler in DispatcherThread '" << mName << "'";
		}
	}
	return true;
}


bool DispatcherThread::isInterruptionRequested()
{
	return mInterruptionRequested || boost::this_thread::interruption_requested();
}

void DispatcherThread::runThread()
{
	// Running phase / main loop
	try {
		while(!isInterruptionRequested())
		{
			if(!mImmediateHandlers.empty()) {
				if (!processImmediateHandlers())
					return;
			}
			if(!processSpin())
				break;
		}
	} catch(boost::thread_interrupted&) {
		// our thread was interrupted, so end it gently now
		MIRA_LOG(DEBUG) << "Got thread_interruption, exiting Dispatcher thread now";
	}

	processFinalizeHandlers();
}

void DispatcherThread::run()
{
	mIsRunning = true;
	ThreadMonitor::instance().addThisThread(mName);

	runThread(); // this performs the actual action

	// remove thread from monitor (only, if ThreadMonitor singleton was not destroyed already)
	if(!ThreadMonitor::isDestroyed())
		ThreadMonitor::instance().removeThisThread();
	mIsRunning = false;
}

///////////////////////////////////////////////////////////////////////////////

}
