/*
 * Copyright (C) 2018 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ChannelPromoteByTypename.h
 *    Infrastructure for promotion of channels from void to
 *    typed based on runtime knowledge of the typename.
 *
 * @author Christof Schröter
 * @date   2018/08/26
 */

#ifndef _MIRA_CHANNEL_PROMOTE_BY_TYPENAME_H_
#define _MIRA_CHANNEL_PROMOTE_BY_TYPENAME_H_

#include <fw/Channel.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

///@cond INTERNAL

/**
 * Base class for channel promoters. Sub classes shall implement the
 * promote() method for specific target types.
 */
class TypedChannelPromoterBase : public Object
{
	MIRA_OBJECT(TypedChannelPromoterBase)

public:
	/// promote a channel to the specific type that this promoter can handle
	virtual void promote(ConcreteChannel<void>* channel) = 0;

	/// Return the typename that this promoter can handle
	virtual std::string typeName() = 0;
};

typedef boost::shared_ptr<TypedChannelPromoterBase> TypedChannelPromoterBasePtr;

///////////////////////////////////////////////////////////////////////////////

class ChannelPromoteByTypename : public LazySingleton<ChannelPromoteByTypename>
{

public:

	/**
	 * Look for a registered promoter for channelType and use it
	 * to promote the channel to typed.
	 */
	bool promoteChannel(ConcreteChannel<void>* channel, const Typename& channelType);

	/** Overload for non-void channel does nothing
	 * (only void channels can be promoted).
	 */
	template <typename PromoteFrom>
	bool promoteChannel(ConcreteChannel<PromoteFrom>* channel,
	                    const Typename& channelType)
	{
		return false;
	}

	/**
	 * Look for a registered promoter for type PromoteTo and warn
	 * if none is registered. 
	 */
	template <typename PromoteTo>
	bool checkForPromoter(ConcreteChannel<PromoteTo>* channel)
	{
		return checkForPromoter(typeName<PromoteTo>(), channel->getID());
	}

	/**
	 * Overload for void channel does nothing
	 * (promotion to void channel is never required).
	 */
	bool checkForPromoter(ConcreteChannel<void>* channel)
	{
		return false;
	}

private:

	bool checkForPromoter(const Typename& channelType, const std::string& channelID);
	void lookForNewPromoters();
	std::string getSourceHint();

private:

	/// descriptions of all currently known TypedChannelPromoterBase subclasses
	std::set<ClassProxy> mKnownPromoterClasses;

	/// index of known Promoters' descriptions by typename
	std::multimap<Typename, ClassProxy> mKnownPromoters; 

	/// actually instantiated (used) Promoter objects, stored for later re-use
	std::map<Typename, TypedChannelPromoterBasePtr> mInstantiatedPromoters;
};

///@endcond

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
