/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Authority.h
 *    Authority class that is used to access the functionality of the framework.
 *
 * @author Erik Einhorn, Tim Langner
 * @date   2010/10/04
 */

#ifndef _MIRA_AUTHORITY_H_
#define _MIRA_AUTHORITY_H_

#ifndef _MIRA_FRAMEWORK_H_
#  error "Authority.h must be included via the Framework.h. You must not include it directly. Use #include <fw/Framework.h> instead"
#endif

#include <utils/EnumToFlags.h>
#include <rpc/RPCFuture.h>
#include <rpc/RPCMacros.h>

#include <math/NearestNeighborInterpolator.h>

#include <serialization/adapters/std/set>
#include <serialization/PropertyTree.h>

#include <fw/AbstractAuthority.h>
#include <fw/AuthorityRuntime.h>
#include <fw/Channel.h>
#include <fw/RPCHandler.h>
#include <fw/ServiceInterfaceHandler.h>

namespace mira {

//////////////////////////////////////////////////////////////////////////////

// forward declarations
struct TransformDesc;
class AuthorityManager;
class FrameworkTransformerNode;

///////////////////////////////////////////////////////////////////////////////

/// Prefix for hidden Authority names
#define MIRA_HIDDEN_AUTHORITY_PREFIX "#"

/**
 * @ingroup FWModule
 * Authorities act as a facade to the framework. 
 * To gain access to the frameworks functionality you must first log in 
 * by creating an Authority object.
 * \code
 * Authority authority("/namespace", "name");
 * \endcode
 */
class MIRA_FRAMEWORK_EXPORT Authority : public AbstractAuthority, public DiagnosticsModule
{
public:

	/**
	 * Flags for creating an authority.
	 */
	enum Flags
	{
		/// Normally authorities must have a unique name.
		NORMAL = 0x0000,

		/**
		 * However sometimes more than one instance is created of an authority
		 * where the name does not matter and could be generated automatically.
		 * This flag provides automatically id generation based on uuids.
		 */
		ANONYMOUS = 0x0001,

		/**
		 * For Authorities that are not shown by default to the user.
		 * Will get the MIRA_HIDDEN_AUTHORITY_PREFIX as name prefix.
		 */
		HIDDEN = 0x0002,

		/**
		 * Authorities with this flag set are invisible subscribers and publishers
		 * for channels e.g. they will not count as official publishers 
		 * for channels which is used for remote framework connections.
		 * When a remote framework connects it will receive a list of all local
		 * published channels excluding the ones from invisible publishers.
		 */
		INVISIBLE_PUBLISHER_SUBSCRIBER = 0x0004,

		/**
		 * Authorities with this flag set are automatically HIDDEN and
		 * are invisible to remote frameworks (internal to the local framework).
		 */
		INTERNAL = 0x000A,

		/**
		 * Instructs the authority to use a separate and independent thread
		 * for handling subscriber callbacks. 
		 * By default the authority uses the same thread for handling
		 *  - subscriber callbacks, 
		 *  - RPC calls
		 *  - cyclic processes 
		 * in order to relieve the user from the need to synchronize his/her
		 * threads. You can use a separate subscriber thread if you know that
		 * you will block inside of your cyclic process method or your RPC
		 * methods but need to react quickly and independently on subscriber
		 * callbacks.
		 */
		INDEPENDENT_SUBSCRIBER_THREAD = 0x0010,

		/**
		 * Instructs the authority to use a separate and independent thread
		 * for handling RPC requests. 
		 * By default the authority uses the same thread for handling
		 *  - subscriber callbacks
		 *  - RPC calls
		 *  - cyclic processes 
		 * in order to relieve the user from the need to synchronize his/her
		 * threads. You can use a separate RPC thread if you know that you
		 * will block inside of your subscriber callbacks or your cyclic process
		 * method but don't want to affect the RPC calls to your authority.
		 */
		INDEPENDENT_RPC_THREAD = 0x0020,

		/**
		 * Does not start a thread for processing the main dispatcher, instead
		 * the user needs to call the spin() method periodically
		 */
		NO_MAIN_THREAD = 0x0040

	};
	MIRA_ENUM_TO_FLAGS_INCLASS(Flags)

public:

	/**
	 * Creates an invalid (not yet checked in) authority by optionally providing
	 * a combination of flags.
	 * @param[in] flags A combination of Flags
	 */
	Authority(Flags flags = NORMAL);

	/**
	 * Creates, checks in and starts an authority object.
	 * This constructor is provided for convenience.
	 * This constructor simplifies creation of an authority where otherwise
	 * the user would need to type the following equivalent code:
	 * Authority a(flags);
	 * a.checkin(fullname);
	 * a.start();
	 * @param[in] fullname The fully qualified name of the authority, including
	 *                     the namespace.
	 * @param[in] flags A combination of Flags
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified name is invalid.
	 */
	Authority(const ResourceName& fullname, Flags flags = NORMAL);

	/**
	 * Creates, checks in and starts an authority object.
	 * This constructor is provided for convenience. It is essentially the same
	 * as the above constructor, but allows to specify the namespace and
	 * name separately.
	 * This constructor simplifies creation of an authority where otherwise
	 * the user would need to type the following equivalent code:
	 * Authority a(flags);
	 * a.checkin(ns, name);
	 * a.start();
	 * @param[in] ns The namespace the authority will reside in
	 * @param[in] name The name of the authority
	 * @param[in] flags A combination of Flags
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified namespace/name is invalid.
	 */
	Authority(const ResourceName& ns, const std::string& name, Flags flags = NORMAL);

	/**
	 * Creates and checks in an authority as a sub-authority of the given parent.
	 * The newly created authority uses the parents thread dispatcher
	 * and registers itself as a diagnostic module on the parents status manager.
	 * When the parent authority is started all its child authorities will be started.
	 * One needs to call start() after constructing the sub authority if the parent
	 * authority is already started.
	 * @param[in] iParent The parent authority
	 * @param[in] name The name of the authority
	 * @param[in] flags A combination of Flags
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified namespace/name is invalid.
	 */
	Authority(Authority& iParent, const std::string& name, Flags flags = ANONYMOUS);

	virtual ~Authority();

	/// reflect method
	template<typename Reflector>
	void reflect(Reflector& r)
	{
		r.roproperty("Status", &getStatusManager(), "The status of this unit");
		r.property("HeartbeatInterval",
		           getter<Duration>(&DiagnosticsModule::getHeartbeatInterval,
		                            (DiagnosticsModule*)this),
		           setter<Duration>(&DiagnosticsModule::setHeartbeatInterval,
		                            (DiagnosticsModule*)this),
		           "The interval of the watchdog",
		           DiagnosticsModule::getHeartbeatInterval());
	}

public:
	/**
	 * @name Session Management and Runtime Control.
	 * These methods allow to change the running state of the Authority.
	 * Moreover, some methods allow to configure the behavior of the Authority
	 * and the used threading model.
	 */
	//@{

	/**
	 * Checks in and therefore makes the authority valid by providing a
	 * namespace and a name that are given as fully qualified resource name.
	 * @pre The authority must not be checked in.
	 * @post The authority will be valid and checked in.
	 * @param[in] fullname The fully qualified name of the authority, including
	 *                     the namespace.
	 * @param[in] runtime  The runtime the authority should use. If not
	 *            specified the authority will create its own runtime.
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified namespace/name is invalid.
	 */
	void checkin(const ResourceName& fullname,
	             AuthorityRuntimePtr runtime = AuthorityRuntimePtr());

	/**
	 * Checks in and therefore makes the authority valid by providing a
	 * namespace and a name. This is essentially the same as the above
	 * method, but allows to specify namespace and name separately.
	 * @pre The authority must not be checked in.
	 * @post The authority will be valid and checked in.
	 * @param[in] ns The namespace the authority will reside in
	 * @param[in] name The name of the authority
	 * @param[in] runtime The runtime the authority should use. If not
	 *            specified the authority will create its own runtime.
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified namespace/name is invalid.
	 */
	void checkin(const ResourceName& ns, const std::string& name,
	             AuthorityRuntimePtr runtime = AuthorityRuntimePtr());

	/**
	 * Checks in the authority as sub-authority of parent. Hence, this authority
	 * will share some of the parents settings and runtime threads.
	 * @pre The authority must not be checked in.
	 * @post The authority will be valid and checked in.
	 * @param[in] iParent The parent authority
	 * @param[in] ns The namespace the authority will reside in
	 * @param[in] name The name of the authority
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified namespace/name is invalid.
	 */
	void checkin(Authority& iParent, const ResourceName& ns,
	             const std::string& name);

	/**
	 * Checks in the authority as sub-authority of parent. Hence, this authority
	 * will share some of the parents settings (namespace, etc) and runtime
	 * threads.
	 * @pre The authority must not be checked in.
	 * @post The authority will be valid and checked in.
	 * @param[in] iParent The parent authority
	 * @param[in] name The name of the authority
	 *
	 * @throw XInvalidParameter If another authority with the same name exists
	 *                          or if the specified namespace/name is invalid.
	 */
	void checkin(Authority& iParent, const std::string& name);

	/**
	 * Checks out and therefore invalidates the authority
	 * @pre The authority must be checked in and valid
	 * @post Authority is checked out. Every attempt to use the authority object
	 *       again will throw an exception.
	 */
	void checkout();

	/**
	 * Returns true, if this Authority was initialized and is checked in,
	 * otherwise false.
	 */
	bool isValid() const {
		return mValid;
	}

	/**
	 * Check if authority is checked in and therefore valid.
	 * @throw XInvalidConfig if not checked in
	 */
	void validate() const {
		if ( !isValid() )
			MIRA_THROW(XInvalidConfig, "Authority is not registered.");
	}

	/**
	 * Returns a shared_ptr to the runtime that this authority uses. The
	 * returned pointer never is nullptr if this Authority was checked in
	 * successfully. The returned runtime may be shared between different
	 * Authorities.
	 */
	AuthorityRuntimePtr getRuntime() {
		return mRuntime;
	}

	/**
	 * Returns a set with all children / sub-authorities.
	 */
	std::set<Authority*> getChildren() {
		return mChildren;
	}

	/**
	 * Adds a runnable that is invoked immediately. After the handler is
	 * processed it will be removed from the list of registered handlers.
	 * @pre The authority must be checked in and valid
	 */
	void addImmediateHandler(DiagnosticRunnablePtr runnable);

	/**
	 * Adds a handler function that is invoked immediately. After the handler is
	 * processed it will be removed from the list of registered handlers.
	 * Creates a Runnable internally.
	 * @pre The authority must be checked in and valid
	 */
	template <typename F>
	void addImmediateHandlerFunction(F&& fn);

	/**
	 * Adds a runnable that is executed ONCE just before the thread terminates.
	 * After the handler is processed it will be removed from the list of
	 * registered handlers.
	 * @pre The authority must be checked in and valid
	 */
	void addFinalizeHandler(DiagnosticRunnablePtr runnable);

	/**
	 * Adds a function that is executed ONCE just before the thread terminates.
	 * Creates a Runnable internally. After the handler is processed it will
	 * be removed from the list of registered handlers.
	 * @pre The authority must be checked in and valid
	 */
	template <typename F>
	void addFinalizeHandlerFunction(F&& fn);

	/**
	 * Creates and adds a timer that gets called cyclic in a given period
	 * (if oneshot is false) and calls the given callback on every invocation.
	 * @pre The authority must be checked in and valid
	 * @param[in] period Period the timer is called in
	 * @param[in] callback Function that gets called every time the timer fires.
	 * @param[in] oneshot If true callback is only called once.
	 * @return pointer to created timer. Can be used for removeTimer().
	 */
	TimerPtr createTimer(Duration period, TimerCallback callback,
	                     bool oneshot=false);
	
	/**
	 * Creates and adds a timer that gets called cyclic in a given period
	 * (if oneshot is false) and calls the given callback on every invocation.
	 * Additionally a warning is given when the call to the timer exceeds the
	 * scheduled invocation time about the specified tolerance.
	 * @pre The authority must be checked in and valid
	 * @param[in] period Period the timer is called in
	 * @param[in] tolerance The tolerance for exceeding the period.
	 * @param[in] callback Function that gets called every time the timer fires.
	 * @param[in] oneshot If true callback is only called once.
	 * @return pointer to created timer. Can be used for removeTimer().
	 */
	TimerPtr createTimer(Duration period, Duration tolerance,
	                     TimerCallback callback, bool oneshot=false);

	/// Provided for convenience. See above createTimer method for more details.
	template<typename Class>
	TimerPtr createTimer(Duration period,
	                     void (Class::*f)(const Timer&), Class* obj,
	                     bool oneshot=false);

	/// Provided for convenience. See above createTimer method for more details.
	template<typename Class>
	TimerPtr createTimer(Duration period, void (Class::*f)(const Timer&),
	                     bool oneshot=false);

	/// Provided for convenience. See above createTimer method for more details.
	template<typename Class>
	TimerPtr createTimer(Duration period, Duration tolerance,
	                     void (Class::*f)(const Timer&), Class* obj,
	                     bool oneshot=false);

	/// Provided for convenience. See above createTimer method for more details.
	template<typename Class>
	TimerPtr createTimer(Duration period, Duration tolerance,
	                     void (Class::*f)(const Timer&), bool oneshot=false);

	/**
	 * Creates a timer that calls callback exactly once at the given time.
	 * @pre The authority must be checked in and valid
	 * @param[in] time Time when the callback should be called.
	 * @param[in] callback Function that gets called at the given time.
	 * @return pointer to created timer. Can be used for removeTimer().
	 */
	TimerPtr createTimer(Time time, TimerCallback callback);

	/// Provided for convenience. See above createTimer method for more details.
	template<typename Class>
	TimerPtr createTimer(Time time, void (Class::*f)(const Timer&), Class* obj);

	/// Provided for convenience. See above createTimer method for more details.
	template<typename Class>
	TimerPtr createTimer(Time time, void (Class::*f)(const Timer&));

	/**
	 * Removes the given timer (timer callback will not be called again and
	 * timer will not be rescheduled). Calling start on a removed timer will
	 * not start it again.
	 */
	void removeTimer(TimerPtr timer);

	/**
	 * Starts/Resumes the main dispatcher thread if it is not yet started since
	 * the authority was created and checked in manually.
	 * Starts also dispatcher threads for rpc service calls.
	 * @pre The authority must be checked in and valid
	 */
	virtual void start();

	/**
	 * Stops/Pauses the main dispatcher thread. To continue call start.
	 * @pre The authority must be checked in and valid
	 */
	virtual void stop();

	/**
	 * Executes a single step of the main dispatcher of this authority.
	 * If this authority does not own the AuthorityRuntime with the main
	 * dispatcher, this method does nothing.
	 * The method returns true, if the caller should continue to call spin()
	 * periodically. If false is returned, spin() should not be called any
	 * longer.
	 *
	 * Usually the spin() method waits for events and signals of the dispatcher,
	 * you can specify the max. time that is spent in the call using the
	 * maxWait parameter. If this parameter is set to 0, spin() will not block
	 * at all. If it is set to Duration::infinity (the default), it will wait
	 * until the next signal or event or stop request.
	 */
	bool spin(const Duration& maxWait = Duration::infinity());

	/**
	 * Returns true, if there was a failure while processing the immediate
	 * handlers.
	 * This is an unrecoverable failure that will disallow the further execution
	 * of the authority.
	 * @pre The authority must be checked in and valid
	 */
	virtual bool hasUnrecoverableFailure() const;

	/**
	 * Returns true, if the authority was started (by calling start()).
	 * It will return false, if the authority was not yet started or is stopped.
	 */
	virtual bool isStarted() const;

	/**
	 * Return if the authority's main dispatcher is running
	 * @pre The authority must be checked in and valid
	 */
	virtual bool isRunning() const;

	//@}

public:

	/**
	 * Enables/disables subscriptions. If enabled, subscribed callbacks
	 * will be called upon changes in the channels. 
	 * If false, callbacks will not be called.
	 */
	void enableSubscriberCallbacks(bool enable);

public:

	////////////////////////////////////////////////////////////////////////////
	// Channel framework
	/**
	 * @name Channel Communication Access.
	 * The following methods gain access to the channels for communication and
	 * allow to control the subscriptions of the Authority
	 */
	//@{

	/**
	 * Resolve a given name by taking the namespace of the authority into account
	 * @pre The authority must be checked in and valid
	 * @param[in] name to be resolved
	 * @return resolved name
	 */
	std::string resolveName(const std::string& name) const;

	/**
	 * Resolves the given name of the service by taking the namespace of the
	 * authority into account. This method does essentially the same as
	 * resolveName() but takes a possible "#buildin" suffix into account.
	 * @pre The authority must be checked in and valid
	 * @param[in] name to be resolved
	 * @return resolved name
	 */
	std::string resolveServiceName(const std::string& name) const;

	/**
	 * Publishes a channel with the given id and marks the authority
	 * as a publisher of this channel. If the channel does not exist it will be
	 * created.
	 * @pre The authority must be checked in and valid
	 * @pre The channel must not exist or the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The channel id
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data
	 * @note The returned channel object only allows access according to the
	 *       current access level of the authority at calling time.
	 *       If you want to gain  read and write access you have to call
	 *       subscribe() and publish() and  store the channel on the second call.
	 *       Otherwise you can call  subscribe() and publish() and obtain
	 *       the channel object later by calling getChannel().
	 */
	template<typename T>
	Channel<T> publish(const std::string& channelID)
	{
		static_assert(std::is_void<T>::value!=true, 
		              "You must specify a typename for untyped channels");
		return publish<T>(channelID, typeName<T>());
	}

	/**
	 * Same as publish(channelID). 
	 * Use this for untyped channels Channel<void> to specify a typename.
	 */
	template<typename T>
	Channel<T> publish(const std::string& channelID, const Typename& type);

	/**
	 * Unpublish a channel with the given channel id
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidConfig if authority is not valid
	 * @param[in] channelID The channel id
	 */
	void unpublish(const std::string& channelID);

	/**
	 * Subscribes authority to a given channel.
	 * It returns a channel object that the caller can store to poll/read data
	 * from the channel. The channel is created if it does not exists.
	 * @pre The authority must be checked in and valid
	 * @pre The channel must not exist or the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @param[in] storageDuration The requested duration of how long data is
	 *            stored in the channel (history). Note that the real history
	 *            length may differ from this request because the number of
	 *            slots in the channel is limited by its maximum length.
	 *            Assuming we have only one writer to the channel that updates
	 *            the channel  each T milliseconds the real storage duration D
	 *            can be computed by D = (maxSlots-1) * T.
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data
	 * @note The returned channel object only allows access accordig
	 *       to the current access level of the authority at calling time.
	 *       If you want to gain read and write access you have to call
	 *       subscribe() and publish() and store the channel on the second call.
	 *       Otherwise you can call subscribe() and publish() and obtain
	 *       the channel object later by calling getChannel().
	 */
	template<typename T>
	Channel<T> subscribe(const std::string& channelID, 
	                     const Duration& storageDuration = Duration::seconds(0));

	/**
	 * Subscribes authority to a given channel and adds callback function fn.
	 * At least one thread is created per authority for the callbacks.
	 * If independentThread is false the main thread for this authority is used
	 * for the channel callback. If independentThread is true
	 * an independent thread for this channel is created for the authority.
	 * @pre The authority must be checked in and valid
	 * @pre The channel must not exist or the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @param[in] fn The callback function that is called whenever the data
	 *            on the channel changes.
	 * @param[in] storageDuration The requested duration of how long data
	 *            is stored in the channel (history).
	 *            Note that the real history length may differ from this request 
	 *            because the number of slots in the channel is limited by
	 *            its maximum length. Assuming we have only one writer
	 *            to the channel that updates the channel each T milliseconds
	 *            the maximum  storage duration maxD can be computed by
	 *            maxD = (maxSlots-1) * T.
	 * @param[in] independentThread If true, an independent thread is created
	 *            for this channel callback
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data
	 * @note The returned channel object only allows access according
	 *       to the current access level of the authority at calling time.
	 *       If you want to gain read and write access you have to call
	 *       subscribe() and publish() and store the channel on the second call.
	 *       Otherwise you can call subscribe() and publish() and obtain
	 *       the channel object later by calling getChannel().
	 */
	template<typename T>
	Channel<T> subscribe(const std::string& channelID,
	                     boost::function<void (ChannelRead<T>)> fn,
	                     const Duration& storageDuration = Duration::seconds(0),
	                     bool independentThread = false);

	/// Provided for convenience. See above subscribe method for more details.
	template<typename T>
	Channel<T> subscribe(const std::string& channelID,
	                     boost::function<void (ChannelRead<T>)> fn,
	                     bool independentThread);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above subscribe methods for more details.
	 */
	template<typename T, typename Class>
	Channel<T> subscribe(const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>),
	                     Class* obj,
	                     const Duration& storageDuration = Duration::seconds(0),
	                     bool independentThread = false);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above subscribe methods for more details.
	 */
	template<typename T, typename Class>
	Channel<T> subscribe(const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>),
	                     Class* obj, bool independentThread);

	/**
	 * Provided for simplified registration of a member function of a class
	 * as callback.
	 *
	 * Example:
	 *
	 * \code
	 * class MyClass
	 * {
	 * public:
	 *     void initialize()
	 *     {
	 *         authority.subscribe<int>("MyChannel", &MyClass::myChannelCallback);
	 *     }
	 *
	 *     void myChannelCallback(ChannelRead<int> data);
	 * };
	 * \endcode
	 *
	 * @see Authority::subscribe() for details.
	 */
	template<typename T, typename Class>
	Channel<T> subscribe(const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>),
	                     const Duration& storageDuration = Duration::seconds(0),
	                     bool independentThread = false)
	{
		Class* obj = dynamic_cast<Class*>(this);
		assert(obj!=NULL);
		return subscribe<T>(channelID, f, obj, storageDuration, independentThread);
	}

	/**
	 * Provided for simplified registration of a member function as callback.
	 * @see Authority::subscribe() for details.
	 */
	template<typename T, typename Class>
	Channel<T> subscribe(const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>), bool independentThread)
	{
		Class* obj = dynamic_cast<Class*>(this);
		assert(obj!=NULL);
		return subscribe<T>(channelID, f, obj, independentThread);
	}

	/**
	 * Publishes a channel with the given id and marks the authority
	 * as a publisher of this channel. Also subscribes authority to the channel.
	 * It returns a channel object that the caller can store to poll/read
	 * and write data from/to the channel.
	 * The channel is created if it does not exists.
	 * @pre The authority must be checked in and valid
	 * @pre The channel must not exist or the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @param[in] storageDuration The requested duration of how long data is
	 *            stored in the channel (history). Note that the real history
	 *            length may differ from this request because the number of
	 *            slots in the channel is limited by its maximum length.
	 *            Assuming we have only one writer to the channel that updates
	 *            the channel  each T milliseconds the real storage duration D
	 *            can be computed by D = (maxSlots-1) * T.
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data
	 */
	template<typename T>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  const Duration& storageDuration = Duration::seconds(0));

	/**
	 * Publishes a channel with the given id and marks the authority
	 * as a publisher of this channel. Also subscribes authority to the channel
	 * and adds callback function fn.
	 * At least one thread is created per authority for the callbacks.
	 * If independentThread is false the main thread for this authority is used
	 * for the channel callback. If independentThread is true
	 * an independent thread for this channel is created for the authority.
	 * @pre The authority must be checked in and valid
	 * @pre The channel must not exist or the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @param[in] fn The callback function that is called whenever the data
	 *            on the channel changes.
	 * @param[in] storageDuration The requested duration of how long data
	 *            is stored in the channel (history).
	 *            Note that the real history length may differ from this request
	 *            because the number of slots in the channel is limited by
	 *            its maximum length. Assuming we have only one writer
	 *            to the channel that updates the channel each T milliseconds
	 *            the maximum  storage duration maxD can be computed by
	 *            maxD = (maxSlots-1) * T.
	 * @param[in] independentThread If true, an independent thread is created
	 *            for this channel callback
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data.
	 */
	template<typename T>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  boost::function<void (ChannelRead<T>)> fn,
	                                  const Duration& storageDuration = Duration::seconds(0),
	                                  bool independentThread = false);

	/// Provided for convenience. See above publishAndSubscribe method for more details.
	template<typename T>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  boost::function<void (ChannelRead<T>)> fn,
	                                  bool independentThread);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above publishAndSubscribe methods for more details.
	 */
	template<typename T, typename Class>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  void (Class::*f)(ChannelRead<T>),
	                                  Class* obj,
	                                  const Duration& storageDuration = Duration::seconds(0),
	                                  bool independentThread = false);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above publishAndSubscribe methods for more details.
	 */
	template<typename T, typename Class>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  void (Class::*f)(ChannelRead<T>),
	                                  Class* obj, bool independentThread);

	/**
	 * Provided for simplified registration of a member function of a class
	 * as callback.
	 *
	 * Example:
	 *
	 * \code
	 * class MyClass
	 * {
	 * public:
	 *     void initialize()
	 *     {
	 *         authority.publishAndSubscribe<int>("MyChannel", &MyClass::myChannelCallback);
	 *     }
	 *
	 *     void myChannelCallback(ChannelRead<int> data);
	 * };
	 * \endcode
	 *
	 * @see Authority::publishAndSubscribe() for details.
	 */
	template<typename T, typename Class>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  void (Class::*f)(ChannelRead<T>),
	                                  const Duration& storageDuration = Duration::seconds(0),
	                                  bool independentThread = false)
	{
		Class* obj = dynamic_cast<Class*>(this);
		assert(obj!=NULL);
		return publishAndSubscribe<T>(channelID, f, obj, storageDuration, independentThread);
	}

	/**
	 * Provided for simplified registration of a member function as callback.
	 * @see Authority::subscribe() for details.
	 */
	template<typename T, typename Class>
	Channel<T> publishAndSubscribe(const std::string& channelID,
	                                  void (Class::*f)(ChannelRead<T>),
	                                  bool independentThread)
	{
		Class* obj = dynamic_cast<Class*>(this);
		assert(obj!=NULL);
		return publishAndSubscribe<T>(channelID, f, obj, independentThread);
	}

	/**
	 * Subscribes authority to a given channel.
	 * In contrast to the above subscribe methods this method subscribes a
	 * callback that takes a ChannelReadInterval as parameter. Instead of
	 * a single value, a whole interval of data is passed. The interval contains
	 * all values that were stored within the channel since the last callback
	 * (or since the subscription). Hence, by using this subscribe mechanism
	 * you can make sure, that no data is missed.
	 *
	 * @pre The authority must be checked in and valid
	 * @pre The channel must not exist or the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel exists but the types do not match.
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @param[in] fn The callback function that is called whenever the data
	 *            in the interval on the channel changes.
	 * @param[in] storageDuration The requested duration of how long data is
	 *            stored in the channel (history). This parameter is especially
	 *            important for subscribing to intervals, as it must be larger
	 *            than the delay that may occur within the callback to make
	 *            sure that no data is lost. Note that the real history
	 *            length may differ from this request because the number of
	 *            slots in the channel is limited by its maximum length.
	 *            Assuming we have only one writer to the channel that updates
	 *            the channel  each T milliseconds the real storage duration D
	 *            can be computed by D = (maxSlots-1) * T.
	 * @param[in] independentThread If true, an independent thread is created
	 *            for this channel callback
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data
	 * @note The returned channel object only allows access according
	 *       to the current access level of the authority at calling time.
	 *       If you want to gain read and write access you have to call
	 *       subscribe() and publish() and store the channel on the second call.
	 *       Otherwise you can call subscribe() and publish() and obtain
	 *       the channel object later by calling getChannel().
	 */

	template<typename T>
	Channel<T> subscribeInterval(const std::string& channelID,
	                     boost::function<void (ChannelReadInterval<T>)> fn,
	                     const Duration& storageDuration,
	                     bool independentThread = false);

	/*
	 * Provided for simplified registration of a member function as callback
	 * function. See the above subscribe methods for more details.
	 */
	template<typename T, typename Class>
	Channel<T> subscribeInterval(const std::string& channelID,
	                     void (Class::*f)(ChannelReadInterval<T>),
	                     Class* obj,
	                     const Duration& storageDuration,
	                     bool independentThread = false);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function. See the above subscribe methods for more details.
	 */
	template<typename T, typename Class>
	Channel<T> subscribeInterval(const std::string& channelID,
	                     void (Class::*f)(ChannelReadInterval<T>),
	                     const Duration& storageDuration,
	                     bool independentThread = false);

 	/**
	 * Provided for simplified registration of a member function taking
	 * a simple ChannelRead as parameter. This is useful especially when
	 * trying to get every single value (e.g. message passing), even if
	 * processing of a single value takes some longer.
	 */
	template<typename T, typename Class>
	Channel<T> subscribeIntervalByElements(const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>),
	                     Class* obj,
	                     const Duration& storageDuration = Duration::seconds(0),
	                     bool independentThread = false);

	/**
	 * General case accepting boost function instead of member function for
	 * subscribing to an interval with only a ChannelRead function given.
	 */
	template <typename T>
	Channel<T> subscribeIntervalByElements(const std::string& channelID,
	                     boost::function<void(ChannelRead<T>)>,
	                     const Duration& storageDuration = Duration::seconds(0),
	                     bool independentThread = false);

	/**
	 * Simplified notation automatically taking authority object as this
	 * argument. Same as respective subscribeInterval function taking
	 * ChannelReadInterval as parameter.
	 */
	template<typename T, typename Class>
	Channel<T> subscribeIntervalByElements(const std::string& channelID,
	                     void (Class::*f)(ChannelRead<T>),
	                     const Duration& storageDuration = Duration::seconds(0),
	                     bool independentThread = false);

	/**
	 * Waits until data in the specified channel becomes available. If a
	 * timeout it specified, the method times out if no data becomes
	 * available after the given time.
	 * @pre The authority must be checked in and valid
	 * @pre The authority must be subscribed to the given channel
	 * @param[in] channel The channel we are waiting for
	 * @param[in] timeout Wait for this duration or forever if Duration::infinity() (default)
	 * @return A valid ChannelRead, if data became available, or an invalid
	 *         ChannelRead, if the method timed out or the thread was interrupted.
	 */
	template <typename T>
	ChannelRead<T> waitForData(Channel<T> channel,
	                           Duration timeout = Duration::infinity()) const;

	/**
	 * Unsubscribe from a given channel.
	 * If subscribed with a callback function this function is no longer called.
	 * If independentThread was true when subscribing
	 * the independent thread for this channel is stopped.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidConfig if authority is not valid
	 * @param[in] channelID The id of the data channel to subscribe to
	 * @note If subscribed with an independent callback function
	 *       don't call this method inside the callback function. It will block
	 *       forever because this method will stop the callback thread.
	 */
	template <typename T>
	void unsubscribe(const std::string& channelID);

	/**
	 * Returns channel object that can be used to obtain read and write
	 * access on the channel.
	 * @pre The authority must be checked in and valid
	 * @pre The channel must exist and the type of the channel must match
	 * @throw XInvalidConfig if authority is not valid
	 * @throw XBadCast if the channel types do not match.
	 * @throw XAccessViolation if the channel does not exist
	 * @return A Channel object that the caller can store to have fast access
	 *         to the channel data
	 * @note The returned channel object only allows access according
	 *       to the current access level of the authority at calling time.
	 *       If you want to gain read and write access you have to call
	 *       subscribe() and publish() and store the channel on the second call.
	 *       Otherwise you can call subscribe() and publish() and obtain
	 *       the channel object later by calling getChannel().
	 */
	template <typename T>
	Channel<T> getChannel(const std::string& channelID);

	/**
	 * Returns if this authority is subscribed to a channel with the given channel id
	 */
	bool isSubscribedOn(const std::string& channelID) const;

	MIRA_DEPRECATED("Please use isSubscribedOn() instead",
	/// @deprecated Please use isSubscribedOn() instead.
	bool hasSubscribedOn(const std::string& channelID) const) {
		return isSubscribedOn(channelID);
	}

	/**
	 * Returns if this authority has published a channel with the given channel id
	 */
	bool hasPublished(const std::string& channelID) const;

	/**
	 * Returns if a channel with the given channel id exists.
	 */
	bool doesChannelExist(const std::string& channelID) const;

	/**
	 * Waits until a channel with the given channel id becomes available.
	 */
	bool waitForChannel(const std::string& channelID,
	                    const Duration& timeout = Duration::infinity()) const;

	//@}

public:

	////////////////////////////////////////////////////////////////////////////
	// Transform framework

	/**
	 * @name Access to Transform Framework.
	 * The following methods allow to use the transformation framework.
	 */
	//@{

	/**
	 * Checks if a transform from source frame to target frame is available.
	 * This means a path in the transform tree exists from source to target and
	 * all nodes on that path contain data.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target or source node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @return true if transform is available, false otherwise
	 */
	bool isTransformAvailable(const std::string& targetID, const std::string& sourceID) const;

	/**
	 * Checks if a transform from source frame to target frame via fixed frame is
	 * available. This means a path in the transform tree exists from source to target 
	 * via fixed and all nodes on that path contain data.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target, source or fixed node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param fixedID The frame id of the fixed node
	 * @return true if transform is available, false otherwise
	 */
	bool isTransformAvailable(const std::string& targetID, const std::string& sourceID,
	                          const std::string& fixedID) const;

	/**
	 * Waits until a transform from source frame to target frame is available.
	 * This means a path in the transform tree exists from source to target and
	 * all nodes on that path contain data.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target or source node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param timeout Wait for this duration or forever if Duration::infinity() (default)
	 * @return true if transform is available, false on timeout
	 */
	bool waitForTransform(const std::string& targetID, const std::string& sourceID,
	                      Duration timeout = Duration::invalid()) const;

	/**
	 * Waits until a transform from source frame to target frame via fixed frame 
	 * is available. This means a path in the transform tree exists from source 
	 * to target via fixed and all nodes on that path contain data.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target, source or fixed node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param fixedID The frame id of the fixed node
	 * @param timeout Wait for this duration or forever if Duration::infinity() (default)
	 * @return true if transform is available, false on timeout
	 */
	bool waitForTransform(const std::string& targetID, const std::string& sourceID,
	                      const std::string& fixedID, Duration timeout = Duration::invalid()) const;


	/**
	 * Subscribes to a transformation between the given target and source frame.
	 * Usually, the computation of the transformation between the target and
	 * source frame depends on multiple frames along the path between the
	 * 'target' and 'source' within the transformation tree.
	 * The installed callback function will be called whenever one of these
	 * frames and hence the transformation between the target and source frame
	 * changes. The latest transformation and the corresponding timestamp
	 * will be passed as parameter to the callback function.
	 * The transformation between target and source frame must exist, otherwise
	 * an XTransform exception is thrown. You can use waitForTransform to
	 * ensure that the transformation exists.
	 *
	 * @pre The authority must be checked in and valid.
	 * @throw TransformerBase::XTransform If the transformation between source and target does
	 *                   not exist.
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param fn       The callback that is called when the transformation changes,
	 *                 the new transformation and the corresponding timestamp
	 *                 are passed as parameter.
	 * @param filter The filter used to interpolate between the transformations.
	 */
	template<typename Transform, typename Filter>
	void subscribeTransform(const std::string& targetID,
	                        const std::string& sourceID,
	                        boost::function<void (Transform, Time)> fn,
	                        Filter&& filter);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above subscribe methods for more details.
	 */
	template<typename Transform, typename Filter, typename Class>
	void subscribeTransform(const std::string& targetID,
	                        const std::string& sourceID,
	                        void (Class::*f)(Transform, Time), Class* obj,
	                        Filter&& filter);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above subscribe methods for more details.
	 */
	template<typename Transform, typename Filter, typename Class>
	void subscribeTransform(const std::string& targetID,
	                        const std::string& sourceID,
	                        void (Class::*f)(Transform, Time),
	                         Filter&& filter);

	/**
	 * Subscribes to a transformation between the given target and source frame.
	 * Usually, the computation of the transformation between the target and
	 * source frame depends on multiple frames along the path between the
	 * 'target' and 'source' within the transformation tree.
	 * The installed callback function will be called whenever one of these
	 * frames and hence the transformation between the target and source frame
	 * changes. The latest transformation and the corresponding timestamp
	 * will be passed as parameter to the callback function.
	 * The transformation between target and source frame must exist, otherwise
	 * an XTransform exception is thrown. You can use waitForTransform to
	 * ensure that the transformation exists.
	 *
	 * @pre The authority must be checked in and valid.
	 * @throw TransformerBase::XTransform If the transformation between source and target does
	 *                   not exist.
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param fn       The callback that is called when the transformation changes,
	 *                 the new transformation and the corresponding timestamp
	 *                 are passed as parameter.
	 */
	template<typename Transform>
	void subscribeTransform(const std::string& targetID,
	                        const std::string& sourceID,
	                        boost::function<void (Transform, Time)> fn);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above subscribe methods for more details.
	 */
	template<typename Transform, typename Class>
	void subscribeTransform(const std::string& targetID,
	                        const std::string& sourceID,
	                        void (Class::*f)(Transform, Time), Class* obj);

	/**
	 * Provided for simplified registration of a member function as callback
	 * function.
	 * See the above subscribe methods for more details.
	 */
	template<typename Transform, typename Class>
	void subscribeTransform(const std::string& targetID,
	                        const std::string& sourceID,
	                        void (Class::*f)(Transform, Time));



	/**
	 * Get a transformation from source to target frame at a given time stamp.
	 * The specified filter is used. If time is "not_a_date_time" e.g. a default
	 * constructed Time(), the latest available transform is returned.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target or source node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param time The time of the requested transform
	 * @param filter The filter used to interpolate between the transforms
	 * @return The requested transformation
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const std::string& targetID,
	                       const std::string& sourceID,
	                       const Time& time, Filter&& filter) const;

	/**
	 * @see getTransform(const std::string&, const std::string&,
	                     const Time&, Filter&&)
	 * Here by default the NearestNeighborInterpolator is used
	 * for interpolation between transforms.
     * If time is "not_a_date_time" (the default value), the latest available
     * transform is returned.
	 */
	template<typename Transform>
	Transform getTransform(const std::string& targetID,
	                       const std::string& sourceID,
	                       const Time& time = Time()) const
	{
		return getTransform<Transform, NearestNeighborInterpolator>(
						targetID, sourceID, time, NearestNeighborInterpolator());
	}

	/**
	 * Get a transformation from source to target via fixed frame at the given
	 * time stamps. The specified filter is used.
	 * This method can be used to do time traveling.
	 * For example if one wants to know what was the position of the right arm
	 * two seconds ago relative to the position the left arm has now:
	 * \code
	 * Time t = Time::now();
	 * getTransform("LeftArmFrame", t,
	 *              "RightArmFrame", t-Duration::seconds(2),
	 *              "RobotFrame");
	 * \endcode
	 * The "RobotFrame" is chosen as "fixed frame" since it is fixed over time
	 * relative to the arm positions and hence can be used as reference frame.
	 * 
	 * One can read the parameters of this function like:
	 * Give the transform from frame targetID at targetTime to
	 * frame sourceID at time sourceTime specifying the frame fixedID that does
	 * not change over time
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target, source or fixed node does not exist
	 * @param targetID The frame id of the target node
	 * @param targetTime The time of the target transform
	 * @param sourceID The frame id of the source node
	 * @param sourceTime The time of the source transform
	 * @param fixedID The frame id of the fixed node
	 * @param filter The filter used to interpolate between the transforms
	 * @return The requested transformation
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const std::string& targetID, const Time& targetTime,
	                       const std::string& sourceID, const Time& sourceTime,
	                       const std::string& fixedID,
	                       Filter&& filter) const;

	/**
	 * @see getTransform(const std::string& targetID, const Time& targetTime,
	 *                   const std::string& sourceID, const Time& sourceTime,
	 *                   const std::string& fixedID,
	 *                   Filter&&)
	 * Here by default the NearestNeighborInterpolator is used 
	 * for interpolation between transforms.
	 */
	template<typename Transform>
	Transform getTransform(const std::string& targetID, const Time& targetTime,
                           const std::string& sourceID, const Time& sourceTime,
	                       const std::string& fixedID) const
	{
		return getTransform<Transform, NearestNeighborInterpolator>(
						targetID, targetTime, sourceID, sourceTime, fixedID,
						NearestNeighborInterpolator());
	}




	/**
	 * Get a transformation node for the given frame.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if node does not exist
	 * @param frameID The frame id of the requested node.
	 * @return A pointer to the node
	 */
	FrameworkTransformerNode* getTransformNode(const std::string& frameID) const;

	/**
	 * Prepares a transformation chain from a source node to a target node.
	 * The chain can later be used to obtain a transform using
	 * getTransform(const TransformDesc&, const Time& time, Filter&&);
	 * Preparing the transform chain only once can save performance when
	 * calculating the transform very often.
	 * @note It is important that after preparing the transform chain the
	 *       topology must not change.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target or source node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @return The prepared chain
	 */
	TransformDesc prepareTransform(const std::string& targetID,
	                               const std::string& sourceID) const;

	/**
	 * Prepares a transformation chain from a source node to a target node via
	 * fixed node. The chain can later be used to obtain a transform using
	 * getTransform(const TransformDesc&, const Time&, const Time&, Filter&&);
	 * Together with the above mentioned method the same functionality as calling
	 * getTransform(const std::string&, const Time&, const std::string&,
	 *              const Time&, const std::string&, Filter&&) can be achieved,
	 * but preparing the transform chain only once can save performance when
	 * calculating the transform very often.
	 * @note It is important that after preparing the transform chain the
	 *       topology must not change.
	 * @pre The authority must be checked in and valid
	 * @throw XInvalidRead if either the target or source node does not exist
	 * @param targetID The frame id of the target node
	 * @param sourceID The frame id of the source node
	 * @param fixedID The frame id of the fixed node
	 * @return The prepared chain
	 */
	TransformDesc prepareTransform(const std::string& targetID,
	                               const std::string& sourceID,
	                               const std::string& fixedID) const;

	/**
	 * Get a transformation at a given time using the prepared transform chain
	 * and a given filter.
	 * If time is "not_a_date_time" e.g. a default
	 * constructed Time(), the latest available transform is returned.
	 * @param desc The prepared transform chain
	 * @param time The time of the requested transform
	 * @param filter The filter used to interpolate between the transforms
	 * @return The requested transformation
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const TransformDesc& desc, const Time& time,
	                       Filter&& filter) const;

	/**
	 * @see getTransform(const TransformDesc&, const Time&, Filter&&)
	 * Here by default the NearestNeighborInterpolator is used
	 * for interpolation between transforms. 
	 * If time is "not_a_date_time" (the default value), the latest available
	 * transform is returned.
	 */
	template<typename Transform>
	Transform getTransform(const TransformDesc& desc, const Time& time = Time()) const
	{
		return getTransform<Transform, NearestNeighborInterpolator>(
						desc, time, NearestNeighborInterpolator());
	}

	/**
	 * Get a transformation at the given times using the prepared transform chain
	 * and a given filter. This method can be used to do time traveling.
	 * Together with
	 * prepareTransform(const std::string&, const std::string&, const std::string&)
	 * the same functionality can be achieved as by calling
	 * getTransform(const std::string&, const Time&, const std::string&,
	 *              const Time&, const std::string&, Filter&&)
	 * @param desc The prepared transform chain
	 * @param targetTime The time of requested target transform
	 * @param sourceTime The time of requested source transform
	 * @param filter The filter used to interpolate between the transforms
	 * @return The requested transformation
	 */
	template<typename Transform, typename Filter>
	Transform getTransform(const TransformDesc& desc, const Time& targetTime,
	                       const Time& sourceTime, Filter&& filter) const;

	/**
	 * @see getTransform(const TransformDesc&, const Time&,
	                     const Time&, Filter&&);
	 * Here by default the NearestNeighborInterpolator is used
	 * for interpolation between transforms.
	 */
	template<typename Transform>
	Transform getTransform(const TransformDesc& desc, const Time& targetTime,
	                       const Time& sourceTime) const
	{
		return getTransform<Transform, NearestNeighborInterpolator>(
						desc, targetTime, sourceTime, NearestNeighborInterpolator());
	}

	/**
	 * Publish a transformation for a given frame
	 * @pre The authority must be checked in and valid
	 * @param[in] frameID The id of the frame
	 * @param[in] transform Either a 2d or 3d transformation
	 * @param[in] time The time of the transformation
	 */
	template<typename Transform>
	void publishTransform(const std::string& frameID, const Transform& transform,
	                      const Time& time) const;

	/**
	 * Publish a transformation (frameID) indirect by specifying a direct
	 * transform between two nodes (sourceID and targetID).
	 * These two nodes are connected indirectly over the transformation (frameID)
	 * This is, the transformation (frameID) is inferred. In the case you know
	 * the direct 'transform' between 'targetID' and 'sourceID' you can use this
	 * method to publish the desired transform on the link 'frameID' at the
	 * specified 'time' and using the specified 'filter' for interpolation.
	 * In the following example there is a indirect link from sourceID to
	 * targetID via n1, n2, frameID and n3. The transform between source and
	 * target is known. The transform in frameID is inferred and published
	 * by calling this method.
	 * \code
	 *   n2 -------- n1 --- 'sourceID'
	 *   |                    .
	 *   | ???                . 'transform' (known)
	 *   |                    .
	 *   v                    v
	 * 'frameID' --- n3 --- 'targetID'
	 * \endcode
	 * @pre The authority must be checked in and valid
	 * @throw XRuntime if either the frame, target or source frames does not exist
	 * @param[in] frameID The id of the frame to publish indirectly
	 * @param[in] targetID The frame id of the target node
	 * @param[in] sourceID The frame id of the source node
	 * @param[in] transform Either a 2d or 3d transformation
	 * @param[in] time The time of the transformation
	 * @param[in] filter The filter used to interpolate between the transforms
	 */
	template<typename Transform, typename Filter>
	void publishTransformIndirect(const std::string& frameID,
	                              const std::string& targetID,
	                              const std::string& sourceID,
	                              const Transform& transform, const Time& time,
	                              Filter&& filter) const;

	/**
	 * @see publishTransformIndirect(const std::string&,
	 *                               const std::string&,
	 *                               const std::string&,
	 *                               const Transform&, const Time&,
	 *                               Filter&&);
	 * Here by default the NearestNeighborInterpolator is used 
	 * for interpolation between transforms.
	 */
	template<typename Transform>
	void publishTransformIndirect(const std::string& frameID,
	                              const std::string& targetID,
	                              const std::string& sourceID,
	                              const Transform& transform, const Time& time)
	{
		return publishTransformIndirect<Transform, NearestNeighborInterpolator>(
				frameID, targetID, sourceID, transform, time, NearestNeighborInterpolator());
	}

	/**
	 * Add a transformation link between the given child and parent.
	 * Creates child and/or parent if not exist.
	 * @pre The authority must be checked in and valid
	 * @param[in] childID The frame id of the child
	 * @param[in] parentID The frame id of the parent
	 */
	void addTransformLink(const std::string& childID, const std::string& parentID) const;

	//@}

public:

	////////////////////////////////////////////////////////////////////////////
	// RPC Framework

	/**
	 * @name Remote Procedure Calls.
	 * The following methods allow to register and use remote procedure calls.
	 */
	//@{

	/**
	 * Publish a service to register RPC functions at the RPC manager.
	 * You can publish more than one service object with the same service name
	 * @param[in] name The name under which the service is registered. Please
	 *                 note that the name is resolved depending on the namespace
	 *                 the authority is located in.
	 * @param[in] iService The service object to register.
	 * @param[in] handler The handler that is used for processing the RPC calls
	 *            if null internal handler is used (default)
	 */
	template<typename Service>
	void publishService(const std::string& name, Service& iService,
	                    RPCHandlerPtr handler = RPCHandlerPtr());

	/**
	 * Publish a service to register RPC functions at the RPC manager.
	 * As name of the service the id of the authority is used.
	 * @param[in] iService The service object to register.
	 * @param[in] handler The handler that is used for processing the RPC calls
	 *            if null internal handler is used (default)
	 */
	template<typename Service>
	void publishService(Service& iService,
	                    RPCHandlerPtr handler = RPCHandlerPtr());

	/**
	 * Unpublish this service by unregistering the service
	 * with the id of the authority from the RPC manager.
	 */
	void unpublishService();

	/**
	 * Unpublish a service with a given name by unregistering the service
	 * with the name from the RPC manager.
	 */
	void unpublishService(const std::string& name);

	// the code below uses boost::preprocessor to produce multiple callService
	// methods with different parameter counts
	#define RPCGEN_CALLSERVICE(z,n,_)                                          \
		template<typename R  BOOST_PP_ENUM_TRAILING_PARAMS_Z(z,n,typename P)>  \
			RPCFuture<R> callService(const std::string& service, const std::string& method  \
									BOOST_PP_REPEAT_ ## z(n,RPCGEN_CALL_PARAM_DECL,nil)) const;

	BOOST_PP_REPEAT(BOOST_PP_INC(RPC_METHODS_MAX_PARAMS),RPCGEN_CALLSERVICE, nil);
	#undef RPCGEN_CALLSERVICE

#ifdef DOXYGEN
	// this is a fake method to generate a doxygen documentation for the
	// callService() methods created above

	/**
	 * Performs an RPC call.
	 * This method calls the specified RPC method of the specified service.
	 * Depending on the called method the parameters of the RPC call need to
	 * be specified as additional parameters.
	 * The result of the RPC call can be obtained via the RPCFuture that is
	 * returned by this method. The RPCFuture acts as a proxy for the result
	 * of the asynchronous RPC call. It can be used to block until
	 * the RPC call has finished and the result is available.
	 *
	 * @param service      The name of the service
	 * @param method       The method name of the requested call
	 * @param p1           First optional parameter see pn
	 * @param pn           Variable number of optional parameters. The number
	 *                     and type depends on the requested RPC method. Both
	 *                     will be checked by the receiver of the RPC call.
	 *                     If the parameter types do not match to the called
	 *                     RPC method, the server will return an error.
	 * @return Returns an RPCFuture object. The type of the RPCFuture depends
	 *         on the return type of the RPC method. It allows the caller to
	 *         wait for the RPC call to finish and to retrieve the return value.
	 *
	 * When calling this method the return type of the RPC method needs to be
	 * specified as template parameter.
	 *
	 * Example:
	 * \code
	 *
	 *     RPCFuture<float> result = callService<float>("myservice", "mymethod", 123, 1.23f);
	 *
	 *     result.wait(); // wait for the result
	 *
	 *     float r = result.get(); // get the result value
	 * \endcode
	 */
	template<typename R, typename P1, ..., typename Pn>
	RPCFuture<R> callService(const std::string& service,
	                         const std::string& method, P1 p1, ..., Pn pn) const;
#endif

	/**
	 * Invokes a JSON RPC call. jsonCall must follow the JSON RPC 2.0 protocol.
	 * It must contain:
	 *  - "jsonrpc" with exactly the value "2.0"
	 *  - "id" an id that can be anything (int, string, null) but should be an
	 *         unique identifier
	 *  - "method" a service-method combination of the service to call
	 *             (e.g. "/MyService.myMethod")
	 *  - "params" Optional. Required only for methods with parameters.
	 *             An array containing the parameters of the method to call
	 *             (e.g. [1, "Hello World"]). 
	 * \code
	 * {
	 * 		"jsonrpc" : "2.0",
	 * 		"id" : 1,
	 * 		"method" : "/ns/Service.setInt",
	 * 		"params" : [255]
	 * }
	 * \endcode
	 * @return Returns a future that will enclose the JSON RPC response containing
	 *         the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callServiceJSON(const json::Value& jsonRequest) const;

	/**
	 * Provided for convenience. See callJSON above. Will convert jsonCall into
	 * a json::Value and calls callJSON(const json::Value& jsonCall) internally.
	 * @return Returns a future that will enclose the JSON RPC response
	 *         containing the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callServiceJSON(const std::string& jsonString) const;

	/**
	 * Invokes a JSON RPC call, on the specified service and method with the
	 * given params by generating a JSON RPC 2.0 conform json request.
	 * Call id will be automatically generated.
	 * @param[in] service The service to call
	 * @param[in] method The method on the service to call
	 * @param[in] params The parameters for the method to call
	 *            (must be a json::Array or json null value (default))
	 * @return Returns a future that will enclose the JSON RPC response
	 *         containing the result of the call.
	 */
	RPCFuture<JSONRPCResponse> callServiceJSON(const std::string& service,
	                                           const std::string& method,
	                                           const json::Value& params = json::Value()) const;

	/**
	 * Provided for convenience. See callJSON above. Will convert params string into
	 * an json::Array and calls callJSON(const std::string&, const std::string&,
	 * const json::Value&) internally.
	 * @note params must not contain [].
	 */
	RPCFuture<JSONRPCResponse> callServiceJSON(const std::string& service,
	                                           const std::string& method,
	                                           const std::string& params) const;

	/**
	 * Returns true, if a service with the given name exists, otherwise false.
	 */
	bool existsService(const std::string& name) const;

	/**
	 * Returns true, if a service with the given name implements the
	 * given interface, otherwise false.
	 */
	bool implementsInterface(const std::string& name,
	                         const std::string& interface) const;

	/**
	 * Waits until a service with the specified name becomes available.
	 * If a timeout is specified, the method times out if no such service
	 * becomes available after the given time.
	 * Returns true, if the desired services became available, or false, if
	 * the method timed out or the thread was interrupted.
	 */
	bool waitForService(const std::string& name, 
	                    Duration timeout = Duration::invalid()) const;

	/**
	 * Waits until a service with the specified interface becomes available.
	 * If a timeout is specified, the method times out if no such service
	 * becomes available after the given time.
	 * Returns the name of the service, if the desired services became
	 * available, or an empty string, if the method timed out or the thread
	 * was interrupted.
	 */
	std::string waitForServiceInterface(const std::string& interface, 
	                                    Duration timeout = Duration::invalid()) const;

	/**
	 * Returns a string list with the names of all registered services, that
	 * implement the specified interface.
	 */
	std::list<std::string> queryServicesForInterface(const std::string& interface) const;

	/**
	 * Registers a callback that is called whenever a service registers the
	 * given interface, which is specified as first parameter.
	 * The callback method must take two parameters of the type std::string:
	 *    method(const std::string& interface, const std::string& service);
	 * When the method is called, the name of the interface and the name of the
	 * service that is registering is passed. The callback is processed
	 * within the main dispatcher thread of the authority.
	 */
	void registerCallbackForInterface(const std::string& interface,
	                                  ServiceInterfaceHandler::Callback cb);

	/**
	 *  Same as above but providing a function and class pointer instead of
	 *  a boost function. Provided for convenience.
	 */
	template<typename Class>
	void registerCallbackForInterface(const std::string& interface,
	                                  void (Class::*f)(const std::string&,
	                                                   const std::string&),
	                                  Class* obj);

	//@}

public:

	StatusManager& getStatusManager();

	/**
	 * @name Implementation of AbstractAuthority
	 */
	//{@ 

	virtual StatusManager::StatusMap getStatusMap();

	virtual boost::shared_ptr<PropertyNode> getProperties();

	virtual std::set<std::string> getPublishedChannels();

	virtual std::set<std::string> getSubscribedChannels();

	virtual std::set<std::string> getServiceInterfaces();

	virtual ChannelNameMapping getPublishedChannelNames();

	virtual ChannelNameMapping getSubscribedChannelNames();

	//@}

protected:

	template<typename T>
	Channel<T> toProxy(ConcreteChannel<T>* channel);

	void setChannelReadAccess(const std::string& channelID);
	void setChannelWriteAccess(const std::string& channelID);
	void removeChannelReadAccess(const std::string& channelID);
	void removeChannelWriteAccess(const std::string& channelID);
	void insertChannelNameMapping(AbstractAuthority::ChannelNameMapping& map,
	                              const std::string& local,
	                              const std::string& global);
	void eraseChannelNameMapping(AbstractAuthority::ChannelNameMapping& map,
	                             const std::string& local,
	                             const std::string& global);

	void remotePublishService(const std::string& service) const;

	friend struct AuthorityService;
	friend class AuthorityManager;

	PropertyTree getPropertiesRPC();

	void setProperty(const std::string& property, const std::string& value);
	std::string getProperty(const std::string& property) const;
	void setPropertyJSON(const std::string& property, const json::Value& value);
	json::Value getPropertyJSON(const std::string& property) const;

protected:

	/// Channel access flags
	struct RWAccessFlags {
		RWAccessFlags() : flags(CHANNEL_ACCESS_NONE) {}
		ChannelAccessFlags flags;
	};

	typedef std::map<std::string, RWAccessFlags> RWAccessMap;
	typedef std::list<AbstractChannelSubscriberPtr> SubscribePtrList;

private:

	void init();

private:

	/// The used runtime
	AuthorityRuntimePtr mRuntime;
	/// Is the runtime owned by us ore is it shared
	bool mOwnsRuntime;
	/// Valid flag
	bool mValid;
	/// Whether the authority was started
	bool mStarted;
	/// Is this Authority internal to the local framework
	bool mIsInternal;
	/// Is this Authority hidden in terms of publishing and subscribing
	bool mIsInvisiblePublisherSubscriber;

	Flags mFlags;
	/// The channel access map (read, write, no access)
	RWAccessMap mAccessMap;
	/// The parent of this Authority or NULL if top level
	Authority* mParent;
	/// The children of this Authority
	std::set<Authority*> mChildren;

	/// Are we getting updates from our subscribed channels
	bool mSubsciberEnabled;

	typedef boost::shared_ptr<DispatcherThread> DispatcherThreadPtr;

	/**
	 * Independent subscribers used for subscriptions
	 * that have their own independent thread (subscribe() called with
	 * independentThread=true
	 */
	typedef std::list<DispatcherThreadPtr> IndependentSubscribers;

	/// Subscribe handlers for subscriptions that use an independent thread
	IndependentSubscribers mIndependentSubscriberDispatchers;

	/// Thread dispatcher used when we want to have a separate thread for subscriptions
	DispatcherThreadPtr mSubscriberDispatcher;
	/// Thread dispatcher used when we want to have a separate thread for RPC calls
	DispatcherThreadPtr mRPCDispatcher;

	/// Handler for RPC calls
	RPCHandlerPtr mRPCHandler;
	/// Handler for notifications about registered service interfaces
	ServiceInterfaceHandlerPtr mServiceInterfaceHandler;
	/// List of all our subscriptions
	SubscribePtrList mSubscriberList;
	/// Services published by this authority
	std::set<std::string> mPublishedServices;
	/// Mutex for protecting the name maps
	boost::mutex mNameMutex;
	/// Pairs of global and local names for published channels
	ChannelNameMapping mPublishedChannelNames;
	/// Pairs of global and local names for subscribed channels
	ChannelNameMapping mSubscribedChannelNames;
	/// The status manager used for registering diagnostic modules and managing the overall status
	StatusManager mStatusManager;
};

/// Typedef for Authority pointers
typedef boost::shared_ptr<Authority> AuthorityPtr;

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
