/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Main.C
 *    $Tool for generating manifest files.$.
 *
 * @author Ronny Stricker
 * @date   2011/02/23
 */

#include <stdio.h>
#include <stdlib.h>
#include <iostream>

#include <boost/program_options.hpp>

#include <utils/SharedLibrary.h>
#include <factory/Factory.h>
#include <factory/ManifestAgent.h>
#include <error/LogTxtStreamSink.h>
#include <utils/Path.h>
#include <utils/PathFinder.h>

#ifdef MIRA_LINUX
#  include <sys/wait.h>
#  include <unistd.h>
#endif

using namespace std;
using namespace mira;

#ifdef MIRA_WINDOWS
  #  define MIRA_LIB_FOLDER "bin"
#else
  #  define MIRA_LIB_FOLDER "lib"
#endif

//////////////////////////////////////////////////////////////////////////////

class SharedLibraryLoaderSingleton :
	public LazySingleton<SharedLibraryLoaderSingleton>,
	public SharedLibraryLoader
{
};

//////////////////////////////////////////////////////////////////////////////

// try to load the given library
void loadLibrary( string const& iLibrary, PathVector& additionalLibraryPaths )
{
	MIRA_LOG( NOTICE ) << "    Trying to load library " << iLibrary;
	// we want to load our own libraries only. Therefore, we have to check
	// if the given library is located in a known MIRA_PATH location

	bool tMIRALibrary = false;
	Path tLibraryPath( iLibrary );
	// check if the given path is absolute
	if ( Path(iLibrary).root_directory().empty() ) {
		// no absolute path
		// try to find library
		Path tResolvedPath = resolveLibraryName( iLibrary );

		vector<Path> tLibraries;
		// search for libraries in the mira lib folder first of all
		tLibraries = findProjectFiles(MIRA_LIB_FOLDER/tResolvedPath, false);
				
		// search for library in the additional paths
		PathVector secondary = findFiles( additionalLibraryPaths, tResolvedPath, false );
		tLibraries.insert( tLibraries.end(), secondary.begin(), secondary.end() );

		foreach( Path const& tPath, tLibraries)
			MIRA_LOG( NOTICE ) << "      found library " << tPath.string();

		if ( tLibraries.size() > 1 ) {
			MIRA_THROW( XLogical, string("found ")
					+ toString<size_t>(tLibraries.size())
					+ " occurrences of "
					+ iLibrary + "!");
		}
		if ( tLibraries.size() == 1 ) {
			tMIRALibrary = true;
			tLibraryPath = tLibraries[0];
		}
	}
	else {
		// absolute path
		foreach( Path const& tPath, extractPaths("$MIRA_PATH") ) {
			if ( iLibrary.find( tPath.string() ) == 0 ) {
				tMIRALibrary = true;
				break;
			}
		}
	}

	// now that we know that the library belongs to us, we can load it
	if ( tMIRALibrary ) {
		MIRA_LOG( NOTICE ) << "      load library " << tLibraryPath;
		SharedLibraryLoaderSingleton::instance().loadLibrary( tLibraryPath );
	}
	else
		MIRA_LOG( NOTICE ) << "      skip loading library";
}

///////////////////////////////////////////////////////////////////////////////

int main( int argc, char *argv[] )
{
	namespace po = boost::program_options;

	// define command line options
	po::options_description tCmdOptions( "Allowed options" );
	tCmdOptions.add_options()
			("help", "produce help message")
			("preload,P", po::value< vector< string > >()->multitoken(),
					"Libraries to load before the factory starts to extract the "
					"contained objects.")
			("library,L", po::value< string >(),
					"The library which should be used to create the manifest "
					"file." )
			("ignore,I", po::value< vector< string > >()->multitoken(),
					"Class names which should be ignored and excluded from the "
					"manifest file.")
			("target,T", po::value< string >(), "Target filename." )
			("libPath", po::value< vector< string > >()->multitoken(),
					"The manifest generator starts a (non recursive) "
					"search for libraries in the MIRA_PATH/lib folders by default. "
					"Use this flag to add additional search paths, which will be."
					"used during the (nun recursive) search phase.")
			("silent,S","do not produce text output")
			("fork,F","UNIX only: fork a child process to do the work (workaround for the 'TBB failed to destroy TLS storage' problem on some systems)")
			("delete,D", po::value< string >(),"The given file will be deleted if the program fails to generate the manifest file.");

	po::variables_map tCmdValues;

	// parse command line
	try {
		po::store( po::command_line_parser( argc, argv)
			.options( tCmdOptions ).run(), tCmdValues );
		po::notify( tCmdValues );
	}
	catch ( po::error& tException ) {
		cout << tException.what() << "\n\n";
		cout << tCmdOptions << "\n";
		return 1;
	}

	// Workaround for 'TBB failed to destroy TLS storage' bug (see #212)
#ifdef MIRA_LINUX
	if ( tCmdValues.count("fork") != 0 ) {
		if(fork()!=0) {
			// wait for child and return 0 (indicating no error)
			int status;
			// catch our own return code
			wait(&status);
			if ( status == 10*256 ) {
				return -1;
			}
			return 0;
		}
		// child process reaches here and continues, it may abort with 'TBB failed to destroy TLS storage' while terminating the
		// its process, however, the parent process will return success above anyway (thats the workaround)
	}
#endif // MIRA_LINUX

	PathVector tAdditionalSearchPaths;

	if ( tCmdValues.count("libPath") == 1 ) {
		std::vector< std::string > tVct = tCmdValues["libPath"].as< vector< string > >();
		foreach( string const& path, tVct ) {
			tAdditionalSearchPaths.push_back( path );
		}
	}

	if ( tCmdValues.count("silent") == 0 ) {
		MIRA_LOGGER.registerSink( LogTxtStreamSink( std::cout ) );
	}

	if ( tCmdValues.count("help") || argc == 1 ||
			tCmdValues.count("library") == 0 ) {
		cout << tCmdOptions << "\n";
		return 1;
	}

	string tLibraryName = tCmdValues["library"].as< string >();

	// set name of destination .mani file
	string tTargetFile = tLibraryName+".manifest";
	if ( tCmdValues.count("target") ) {
		tTargetFile = tCmdValues["target"].as< string >();
	}

	// To analyze the classes which are contained in the given libary we might
	// have to preload some dependent libraries to build the "background"
	// classes which are not directly included in the lib
	typedef map<string, ClassProxy > DerivedClassesMap;

	DerivedClassesMap tBackgroundClasses;

	MIRA_LOG( NOTICE ) << "Creating manifest for library " << tLibraryName;
	try {
		if ( tCmdValues.count("preload") ) {
			MIRA_LOG( NOTICE ) << "  preloading libraries";
			for ( uint32 i=0;
					i < tCmdValues["preload"].as< vector< string > >().size(); ++i ) {
				loadLibrary( tCmdValues["preload"].as< vector< string > >()[i],
						tAdditionalSearchPaths );
			}
			// get the available classes at this point (before the target library
			// has been loaded)
			tBackgroundClasses = ClassFactory::getDerivedClasses();
		}

		// NOW we can load the real shit
		MIRA_LOG( NOTICE ) << "  finally loading the library " << tLibraryName;
		loadLibrary( tLibraryName, tAdditionalSearchPaths );
	}
	catch ( Exception& ex ) {
		cout << "***************************************\n";
		cout << "***       Manifest Error!!!         ***\n";
		cout << "***************************************\n";
		cout << ex.what();

		if ( tCmdValues.count("delete") != 0 ) {
			string tFile = resolveLibraryName( tCmdValues["delete"].as<string>() ).string();

			if( remove( tFile.c_str() ) != 0 )
				cout << "\nCannot remove \"" << tFile << "\"!" << endl;
			else
				cout << "\nRemoved \"" << tFile << "\"!" << endl;
		}

		return 10;
	}

	// list of classes, which should be ignored
	std::set<std::string> ignoreClasses;
	if ( tCmdValues.count("ignore") ) {
		for( uint32 i = 0;
				i < tCmdValues["ignore"].as< vector< string > >().size(); ++i )
		{
			ignoreClasses.insert( tCmdValues["ignore"].as< vector< string > >()[i] );
		}
	}

	// get the available classes after the target library has been loaded
	DerivedClassesMap tAllClasses = ClassFactory::getDerivedClasses();

	MIRA_LOG( NOTICE ) << "NEW classes count: "
			<< tAllClasses.size() - tBackgroundClasses.size();

	vector< ClassProxy > tNewClasses;

	int ignoredClassesCnt = 0;
	foreach( DerivedClassesMap::value_type const& tClass, tAllClasses ) {
		if ( tBackgroundClasses.find( tClass.first ) == tBackgroundClasses.end() ) {
			// Bingo Bongo. This is a new class from our library,
			if (ignoreClasses.find(tClass.first) == ignoreClasses.end())
				tNewClasses.push_back( tClass.second );
			else
				ignoredClassesCnt++;
		}
	}
	if (ignoredClassesCnt > 0) {
		MIRA_LOG( NOTICE ) << "IGNORED classes count: " << ignoredClassesCnt;
	}

	ManifestAgent tManifestAgent( tLibraryName, tNewClasses );

	MIRA_LOG( NOTICE ) << "write manifest for library "
			<< tCmdValues["library"].as< string >();

	// write a manifest file
	tManifestAgent.saveToFile( tTargetFile );

	////////////////////////////////////////////////////////////////////////////

	MIRA_LOG( NOTICE ) << "done.";

	return 0;
}

//////////////////////////////////////////////////////////////////////////////
