/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PathFinderTest.C
 *    Test cases for PathFinder.h.
 *
 * @author Christian Martin
 * @date   2011/09/29
 */

#include <boost/test/unit_test.hpp>

#include <utils/PathFinder.h>
#include <utils/Time.h>

using namespace std;
using namespace mira;

void createFile(const Path& path)
{
	std::ofstream tFile(path.native());
	tFile << "MIRA";
}

BOOST_AUTO_TEST_CASE( testFindFiles )
{
	Path tTmp = getTempDirectory();

	Path root  = tTmp/"root";
	Path path1 = tTmp/"root/foo";
	Path path2 = tTmp/"root/bar";

	PathVector all;
	all.push_back(path1);
	all.push_back(path2);

	boost::filesystem::create_directory(root);
	boost::filesystem::create_directory(path1);
	boost::filesystem::create_directory(path1/"a");
	boost::filesystem::create_directory(path1/"a/a");
	boost::filesystem::create_directory(path2);
	boost::filesystem::create_directory(path2/"a");
	boost::filesystem::create_directory(path2/"a/a");

	createFile(path1/"a/foo.txt");
	createFile(path1/"a/bar.txt");
	createFile(path1/"a/a/foo.txt");
	createFile(path1/"a/a/bar.txt");
	createFile(path2/"a/foo.txt");
	createFile(path2/"a/bar.txt");
	createFile(path2/"a/a/foo.txt");
	createFile(path2/"a/a/bar.txt");

	BOOST_CHECK_EQUAL(0, findFiles(root, "foo.txt", false).size());
	BOOST_CHECK_EQUAL(0, findFiles(root, "*.txt", false).size());
	BOOST_CHECK_EQUAL(4, findFiles(root, "foo.txt", true).size());
	BOOST_CHECK_EQUAL(8, findFiles(root, "*.txt", true).size());

	BOOST_CHECK_EQUAL(1, findFiles(path1, "a/foo.txt", false).size());
	BOOST_CHECK_EQUAL(2, findFiles(path1, "a/foo.txt", true).size());
	BOOST_CHECK_EQUAL(4, findFiles(root, "a/foo.txt", true).size());

	BOOST_CHECK_EQUAL(1, findFiles(path1, "a/a/foo.txt", true).size());
	BOOST_CHECK_EQUAL(2, findFiles(path1, "a/a/*.txt", true).size());

	BOOST_CHECK_EQUAL(2, findFiles(all, "a/foo.txt", false).size());
	BOOST_CHECK_EQUAL(4, findFiles(all, "a/foo.txt", true).size());
	BOOST_CHECK_EQUAL(2, findFiles(all, "a/a/foo.txt", false).size());
	BOOST_CHECK_EQUAL(2, findFiles(all, "a/a/foo.txt", true).size());

	BOOST_CHECK_EQUAL(8, findFiles(all, "a/*.txt", true).size());
	BOOST_CHECK_EQUAL(4, findFiles(all, "a/a/*.txt", true).size());

	boost::filesystem::remove_all(root);
}

BOOST_AUTO_TEST_CASE( testFindPath )
{
	Path tTmp = getTempDirectory();
	Path tHome = getHomeDirectory();

	Path root  = tTmp/"root";
	Path path1 = tTmp/"root/foo";
	Path path2 = tTmp/"root/bar";

	PathVector all;
	all.push_back(path1);
	all.push_back(path2);

	boost::filesystem::create_directory(root);
	boost::filesystem::create_directory(path1);
	boost::filesystem::create_directory(path1/"a");
	boost::filesystem::create_directory(path1/"a/a");
	boost::filesystem::create_directory(path1/"a/a/a");
	boost::filesystem::create_directory(path2);
	boost::filesystem::create_directory(path2/"a");
	boost::filesystem::create_directory(path2/"a/a");
	boost::filesystem::create_directory(path2/"a/a/a");

	BOOST_CHECK_EQUAL(0, findDirectories(root, "a", false).size());
	BOOST_CHECK_EQUAL(6, findDirectories(root, "a", true).size());
	BOOST_CHECK_EQUAL(2, findDirectories(all, "a", false).size());
	BOOST_CHECK_EQUAL(6, findDirectories(all, "a", true).size());
	BOOST_CHECK_EQUAL(2, findDirectories(all, "a/a", false).size());
	BOOST_CHECK_EQUAL(4, findDirectories(all, "a/a", true).size());
	BOOST_CHECK_EQUAL(2, findDirectories(all, "a/a/a", false).size());
	BOOST_CHECK_EQUAL(2, findDirectories(all, "a/a/a", true).size());

	boost::filesystem::remove_all(root);
}

#if defined(MIRA_LINUX)
BOOST_AUTO_TEST_CASE( testFindMIRAPath )
{
	Path tTmp  = getTempDirectory();
	Path root  = tTmp/"root";
	Path m1    = root/"mira";
	Path m2    = root/"mira-pkg";
	boost::filesystem::create_directory(root);
	boost::filesystem::create_directory(m1);
	boost::filesystem::create_directory(m2);

	setenv("MIRA_PATH", (m1.string()+":"+m2.string()).c_str(), 1);

	BOOST_CHECK_EQUAL(findContainingMIRAPath(m1/"hallo/welt"), m1);
	BOOST_CHECK_EQUAL(findContainingMIRAPath(m2), m2);
	BOOST_CHECK_EQUAL(findContainingMIRAPath(m2/"test/it"), m2);
	BOOST_CHECK_THROW(findContainingMIRAPath(root), XLogical);

	BOOST_CHECK_EQUAL(chopMIRAPath(m1/"hallo/welt"), "hallo/welt");
	BOOST_CHECK_EQUAL(chopMIRAPath(m1), Path());

	boost::filesystem::remove_all(root);
}
#endif
