/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Benchmark.C
 *
 * @author Erik Einhorn
 */

#include <sstream>

#include <boost/test/unit_test.hpp>
#include <boost/archive/binary_oarchive.hpp>
#include <boost/archive/binary_iarchive.hpp>
#include <boost/archive/text_oarchive.hpp>
#include <boost/archive/xml_oarchive.hpp>

#include <boost/serialization/vector.hpp>
#include <boost/serialization/list.hpp>
#include <boost/serialization/shared_ptr.hpp>

#include <error/Logging.h>

#include <serialization/BinarySerializer.h>
#include <serialization/adapters/std/vector>
#include <serialization/adapters/std/list>
#include <serialization/adapters/boost/shared_ptr.hpp>

#include <utils/Buffer.h>


using namespace mira;
using namespace std;

struct MyClass
{

	MyClass() {}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("a",a,"");
		r.member("b",b,"");
	}

	template<typename Archive>
	void serialize(Archive& ar, const unsigned int v) {
		ar & boost::serialization::make_nvp("a", a);
		ar & boost::serialization::make_nvp("b", b);
	}

	int   a;
	float b;
};

struct MyClass2
{
	MyClass2(int) { ptr=NULL; }

	MyClass2() {
		ptr=this;

		c1.reset(new MyClass);
		c2=c1;
	}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("ptr",ptr,"");
		r.member("a",a,"");
		r.member("b",b,"");
		r.member("c1",c1,"");
		r.member("c2",c2,"");

	}

	template<typename Archive>
	void serialize(Archive& ar, const unsigned int v) {
		ar & boost::serialization::make_nvp("ptr", ptr);
		ar & boost::serialization::make_nvp("a", a);
		ar & boost::serialization::make_nvp("b", b);
		ar & boost::serialization::make_nvp("c1", c1);
		ar & boost::serialization::make_nvp("c2", c2);
	}

	MyClass2* ptr;
	int   a;
	float b;

	boost::shared_ptr<MyClass> c1;
	boost::shared_ptr<MyClass> c2;

};


struct MyClass3
{

	MyClass3() {}

	template<typename Reflector>
	void reflect(Reflector& r) {
		r.member("a",a,"");
		r.member("b",b,"");
	}

	template<typename Archive>
	void serialize(Archive& ar, const unsigned int v) {
		ar & boost::serialization::make_nvp("a", a);
		ar & boost::serialization::make_nvp("b", b);
	}

	int   a;
	float b;
};

///////////////////////////////////////////////////////////////////////////////

template <typename T, typename Stream>
Duration serializeMemcopy(const std::string& info, T& data, Stream& stream)
{
	std::size_t s = sizeof(data);
	stream.resize(s);
	MIRA_LOGTIMER(NOTICE, Timer, "serialize memcopy: " + info);
	memcpy(stream.data(),&data, s);
	Duration t = Timer.peek();
	std::cout << "raw memcopy    " + info + ": " << t << std::endl;
	return t;
}

template <typename T, typename Stream>
Duration serializeMIRA(const std::string& info, T& data, Stream& stream)
{
	BinaryBufferSerializer s(&stream);
	MIRA_LOGTIMER(NOTICE, Timer, "serialize MIRA: " + info);
	s.serialize(data);
	Duration t = Timer.peek();
	std::cout << "serializeMIRA   " + info + ": " << t << std::endl;
	return t;
}


template <typename T, typename Stream>
Duration serializeBoost(const std::string& info, T& data, Stream& stream)
{
	boost::archive::binary_oarchive oa(stream);
	MIRA_LOGTIMER(NOTICE, Timer, "serialize Boost: " + info);
	oa << boost::serialization::make_nvp("data", data);
	Duration t = Timer.peek();
	std::cout << "serializeBoost " + info + ": " << t << std::endl;
	return t;
}

template <typename T, typename Stream>
Duration deserializeMIRA(const std::string& info, T& data, Stream& stream)
{
	BinaryBufferDeserializer s(&stream);
	MIRA_LOGTIMER(NOTICE, Timer, "deserialize MIRA: " + info);
	s.deserialize(data);
	Duration t = Timer.peek();
	std::cout << "deserializeMIRA   " + info + ": " << t << std::endl;
	return t;
}


template <typename T, typename Stream>
Duration deserializeBoost(const std::string& info, T& data, Stream& stream)
{
	boost::archive::binary_iarchive ia(stream);
	MIRA_LOGTIMER(NOTICE, Timer, "deserialize Boost: " + info);
	ia >> boost::serialization::make_nvp("data", data);
	Duration t = Timer.peek();
	std::cout << "deserializeBoost " + info + ": " << t << std::endl;
	return t;
}


///////////////////////////////////////////////////////////////////////////////

void compareTimes(Duration tboost, Duration tmira)
{
	float t1 = tboost.totalMicroseconds();
	float t2 = tmira.totalMicroseconds();
	float speedup = t1/t2;
	std::cout << "Speedup: " << speedup;

	if(speedup>=1.0) {
		std::cout << " (faster than boost)";
	} else {
		std::cout << " (SLOWER than boost)";
	}
	std::cout << std::endl;
	BOOST_CHECK(speedup>=0.5 && "MUCH SLOWER than boost");

	std::cout << std::endl;
}

///////////////////////////////////////////////////////////////////////////////

BOOST_AUTO_TEST_CASE( SerializationBenchmark )
{
	const int N = 256*1024;

	Duration t1,t2;

	std::cout << "CTEST_FULL_OUTPUT (Avoid ctest truncation of output)" << std::endl;
	std::cout << "The following benchmarks are performed for different containers with "
			<< N << " elements of different type." << std::endl;

	std::stringstream s1;
	BinaryBufferOstream::buffer_type s2;
	BinaryBufferOstream::buffer_type s3;

	float array1[N];
	serializeMemcopy("array1", array1, s3);
	t1=serializeBoost("array1", array1, s1);
	t2=serializeMIRA("array1", array1, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("array1", array1, s1);
	t2=deserializeMIRA("array1", array1, s2);
	compareTimes(t1,t2);

	Buffer<float> buffer1(N);
	serializeMIRA("buffer1", buffer1, s2);

	std::vector<float> vector1(N);
	t1=serializeBoost("vector1", vector1, s1);
	t2=serializeMIRA("vector1", vector1, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("vector1", vector1, s1);
	t2=deserializeMIRA("vector1", vector1, s2);
	compareTimes(t1,t2);

	std::list<float> list1(N);
	t1=serializeBoost("list1", list1, s1);
	t2=serializeMIRA("list1", list1, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("list1", list1, s1);
	t2=deserializeMIRA("list1", list1, s2);
// here mira is MUCH_SLOWER (fails test) with boost 1.58 (e.g. Mint 18)
//	compareTimes(t1,t2);

	std::vector<MyClass2> vector2(N);
	t1=serializeBoost("vector2", vector2, s1);
	t2=serializeMIRA("vector2", vector2, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("vector2", vector2, s1);
	t2=deserializeMIRA("vector2", vector2, s2);
	compareTimes(t1,t2);

	std::list<MyClass2> list2(N);
	t1=serializeBoost("list2", list2, s1);
	t2=serializeMIRA("list2", list2, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("list2", list2, s1);
	t2=deserializeMIRA("list2", list2, s2);
	compareTimes(t1,t2);

	std::vector<MyClass> vector3(N);
	t1=serializeBoost("vector3", vector3, s1);
	t2=serializeMIRA("vector3", vector3, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("vector3", vector3, s1);
	t2=deserializeMIRA("vector3", vector3, s2);
	compareTimes(t1,t2);

	std::list<MyClass> list3(N);
	t1=serializeBoost("list3", list3, s1);
	t2=serializeMIRA("list3", list3, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("list3", list3, s1);
	t2=deserializeMIRA("list3", list3, s2);
	compareTimes(t1,t2);

	std::list<MyClass3> list4(N);
	t1=serializeBoost("list4", list4, s1);
	t2=serializeMIRA("list4", list4, s2);
	compareTimes(t1,t2);
	t1=deserializeBoost("list4", list4, s1);
	t2=deserializeMIRA("list4", list4, s2);
	compareTimes(t1,t2);
}
