/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file XMLDom.C
 *    Implementation of a STL conform DOM reader/writer for XML.
 *
 * @author Tim Langner
 * @date   2010/07/01
 */

#include <xml/XMLDom.h>

#include <assert.h>

#include <map>
#include <boost/shared_ptr.hpp>
#include <boost/algorithm/string.hpp>
#include <boost/thread/mutex.hpp>

#include <libxml/tree.h>
#include <libxml/parser.h>
#include <libxml/encoding.h>

#include <utils/Functionals.h>

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

const static xmlElementType sXMLNodeTypeMap[] = { XML_TEXT_NODE, 
                                                  XML_COMMENT_NODE };

///////////////////////////////////////////////////////////////////////////////

/*
 * The following code takes care that the URI of the files where the XML nodes
 * were read from is preserved, even if the XML nodes are added to different
 * XMLDom documents using the insert_before() and insert_after() methods (which
 * are e.g. used by the include modifier).
 * This is achieved by using the _private member of each XML node. If a node
 * is moved to a different XMLDom document, the _private member of that node
 * and all of its sub nodes is set to an XMLSourceInfo object that contains
 * the original URI of the nodes before they were moved. This is implemented in
 * the XMLpreserveSource() method.
 *
 * All XMLSourceInfo objects are collected in a single static map which can
 * be accessed via the thread-safe obtainSourceInfo() method.
 *
 * Now the URI of an XML node can be obtained as follows:
 * 1. if the _private member is not NULL, this node was moved from a previous
 *    XMLDom document and the _private member contains a valid XMLSourceInfo
 *    with the original URI of the document, where the node was read from.
 * 2. if the _private member is NULL, the node always belonged to the current
 *    document and the URI can be determined via node->doc->URL.
 *
 * This is done by the XMLDom::iterator_base::uri() method.
 */

/**
 * Stores source information of XML nodes, i.e. the URI of the file, where
 * the XML nodes were originally parsed from.
 */
struct XMLSourceInfo {
	XMLSourceInfo() {}
	XMLSourceInfo(const std::string& iUri) : uri(iUri) {}

	/// the URI of the file, the corresponding XML node was found in
	std::string uri;
};

/// shared pointer to previous struct
typedef boost::shared_ptr<XMLSourceInfo> XMLSourceInfoPtr;

/**
 * Returns an XMLSourceInfo pointer that contains the specified URL. If the
 * specified URL was not used before, a new XMLSourceInfo object will be created
 * otherwise, the existing one is returned.
 * This method is thread-safe.
 */
static XMLSourceInfoPtr obtainSourceInfo(const std::string& uri)
{
	static boost::mutex sMutex;
	static std::map<std::string, XMLSourceInfoPtr> sSourceInfo;

	boost::mutex::scoped_lock lock(sMutex);

	auto it = sSourceInfo.find(uri);
	if(it==sSourceInfo.end()) {
		XMLSourceInfoPtr info(new XMLSourceInfo(uri));
		sSourceInfo[uri] = info;
		return info;
	} else
		return it->second;
}

/**
 * Returns an XMLSourceInfo pointer that contains the specified URL. If the
 * specified URL was not used before, a new XMLSourceInfo object will be created
 * otherwise, the existing one is returned.
 * This method is thread-safe.
 */
static XMLSourceInfoPtr obtainSourceInfo(const xmlChar* url)
{
	if(url==NULL)
		return XMLSourceInfoPtr();

	std::string uri((const char*)url);
	return obtainSourceInfo(uri);
}

/**
 * Makes sure that the dest-node and all of its sub nodes contain a reference
 * to a XMLSourceInfo with the correct URI that describes there original
 * location. This reference is stored in the _private member of the nodes.
 * To do so, the src tree and the dest tree are walked through in parallel.
 * If nodes in the src tree already contain XMLSourceInfos these infos are
 * copied directly into the nodes of the dest tree. All other nodes of the dest
 * tree are set to the XMLSourceInfo that is specified in the first parameter.
 */
static void XMLpreserveSource(XMLSourceInfoPtr info, _xmlNode* src, _xmlNode* dest,
                              bool useNext = false)
{
	// walk through both trees in parallel and copy the source information
	for(;src!=NULL;src=src->next, dest=dest->next)
	{
		assert(dest!=NULL);
		if(src->_private!=NULL)
			dest->_private = src->_private;
		else
			// if there was no information in source then set it to the specified info
			dest->_private = (void*)info.get();

		if(src->children!=NULL) {
			assert(dest->children!=NULL);
			XMLpreserveSource(info, src->children, dest->children, true);
		}
		if (!useNext)
			return;
	}
}

static void XMLsetSource(XMLSourceInfoPtr info, _xmlNode* dest, bool useNext = false)
{
	// walk through the trees and set the source information
	for(;dest!=NULL; dest=dest->next)
	{
		dest->_private = (void*)info.get();

		if(dest->children!=NULL) {
			XMLsetSource(info, dest->children, true);
		}
		if (!useNext)
			return;
	}
}

/// recursively copies the private data of both trees
static void XMLcopyPrivate(_xmlNode* src, _xmlNode* dest)
{
	// walk through both trees in parallel and copy the source information
	for(;src!=NULL;src=src->next, dest=dest->next)
	{
		assert(dest!=NULL);
		dest->_private = src->_private;
		dest->doc = src->doc;
		if(src->children)
			XMLcopyPrivate(src->children, dest->children);
	}
}

///////////////////////////////////////////////////////////////////////////////

// same as xmlCopyNode, but it makes sure, that the _private member is copied, too
static _xmlNode* XMLcopyNode(_xmlNode* node, int recursive)
{
	_xmlNode* newNode = xmlCopyNode(node, recursive);
	newNode->_private = node->_private;
	newNode->doc = node->doc;

	if(recursive && node->children)
		XMLcopyPrivate(node->children, newNode->children);

	return newNode;
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::const_attribute_iterator

XMLDom::const_attribute_iterator::const_attribute_iterator() :
	mNode(NULL),
	mAttribute(NULL)
{
}

XMLDom::const_attribute_iterator::const_attribute_iterator(xmlNode* iNode, 
                                                           xmlAttr* iAttribute) :
	mNode(iNode),
	mAttribute(iAttribute)
{
}

bool XMLDom::const_attribute_iterator::operator==(const const_attribute_iterator& other) const
{
	return mNode == other.mNode && mAttribute == other.mAttribute;
}

bool XMLDom::const_attribute_iterator::operator!=(const const_attribute_iterator& other) const
{
	return mNode != other.mNode || mAttribute != other.mAttribute;
}

XMLDom::const_attribute_iterator& XMLDom::const_attribute_iterator::operator++()
{
	if (mAttribute == NULL)
		return *this;

	mAttribute = mAttribute->next;
	if (mAttribute == NULL)
	{
		mNode = NULL;
		return *this;
	}
	return *this;
}

XMLDom::const_attribute_iterator& XMLDom::const_attribute_iterator::operator--()
{
	if (mAttribute == NULL)
		return *this;

	mAttribute = mAttribute->prev;
	if (mAttribute == NULL)
	{
		mNode = NULL;
		return *this;
	}
	return *this;
}

XMLDom::Attribute XMLDom::const_attribute_iterator::operator*() const
{
	if (mAttribute == NULL || mAttribute->name == NULL)
		return Attribute("", "");
	if (mAttribute->children  == NULL || 
		mAttribute->children->content == NULL)
		return Attribute((char*) mAttribute->name, "");

	return Attribute((char*) mAttribute->name, 
	                 (char*) mAttribute->children->content);
}

string XMLDom::const_attribute_iterator::name() const
{
	if (mAttribute == NULL || mAttribute->name == NULL)
		return "";

	return (char*) mAttribute->name;
}

string XMLDom::const_attribute_iterator::value() const
{
	if (mAttribute == NULL || 
		mAttribute->children == NULL || 
		mAttribute->children->content == NULL)
		return "";

	return (char*) mAttribute->children->content;
}

XMLDom::NameSpace XMLDom::const_attribute_iterator::nameSpace() const
{
	NameSpace ns;
	if (mAttribute != NULL && mAttribute->ns != NULL)
	{
		if (mAttribute->ns->href != NULL)
			 ns.href = (char*)mAttribute->ns->href;
		if (mAttribute->ns->prefix != NULL)
			 ns.prefix = (char*)mAttribute->ns->prefix;
	}
	return ns;
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::attribute_iterator

XMLDom::attribute_iterator::attribute_iterator() :
	mNode(NULL),
	mAttribute(NULL)
{
}

XMLDom::attribute_iterator::attribute_iterator(xmlNode* iNode, 
                                               xmlAttr* iAttribute) :
	mNode(iNode),
	mAttribute(iAttribute)
{
}

bool XMLDom::attribute_iterator::operator==(const attribute_iterator& other) const
{
	return mNode == other.mNode && mAttribute == other.mAttribute;
}

bool XMLDom::attribute_iterator::operator!=(const attribute_iterator& other) const
{
	return mNode != other.mNode || mAttribute != other.mAttribute;
}

XMLDom::attribute_iterator& XMLDom::attribute_iterator::operator++()
{
	if (mAttribute == NULL)
		return *this;

	mAttribute = mAttribute->next;
	if (mAttribute == NULL)
	{
		mNode = NULL;
		return *this;
	}
	return *this;
}

XMLDom::attribute_iterator& XMLDom::attribute_iterator::operator--()
{
	if (mAttribute == NULL)
		return *this;

	mAttribute = mAttribute->prev;
	if (mAttribute == NULL)
	{
		mNode = NULL;
		return *this;
	}
	return *this;
}

XMLDom::Attribute XMLDom::attribute_iterator::operator*() const
{
	if (mAttribute == NULL || mAttribute->name == NULL)
		return Attribute("", "");
	if (mAttribute->children == NULL || 
		mAttribute->children->content == NULL )
		return Attribute((char*) mAttribute->name, "");

	return Attribute((char*) mAttribute->name, 
	                 (char*) mAttribute->children->content);
}

string XMLDom::attribute_iterator::name() const
{
	if (mAttribute == NULL || mAttribute->name == NULL)
		return "";

	return (char*) mAttribute->name;
}

string XMLDom::attribute_iterator::value() const
{
	if (mAttribute == NULL || 
		mAttribute->children == NULL || 
		mAttribute->children->content == NULL)
		return "";

	return (char*) mAttribute->children->content;
}

XMLDom::NameSpace XMLDom::attribute_iterator::nameSpace() const
{
	NameSpace ns;
	if (mAttribute != NULL && mAttribute->ns != NULL)
	{
		if (mAttribute->ns->href != NULL)
			 ns.href = (char*)mAttribute->ns->href;
		if (mAttribute->ns->prefix != NULL)
			 ns.prefix = (char*)mAttribute->ns->prefix;
	}
	return ns;
}

const XMLDom::attribute_iterator& XMLDom::attribute_iterator::operator=(const string& value)
{
	if (mAttribute == NULL || mAttribute->children == NULL)
		return *this;

	mAttribute->children->content = xmlCharStrndup(value.c_str(), value.size());
	return *this;
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::const_data_iterator

template <int Type>
XMLDom::const_data_iterator<Type>::const_data_iterator() :
	mNode(NULL)
{
}

template <int Type>
XMLDom::const_data_iterator<Type>::const_data_iterator(xmlNode* iNode) :
	mNode(iNode)
{
}

template <int Type>
bool XMLDom::const_data_iterator<Type>::operator==(const const_data_iterator& other) const
{
	return mNode == other.mNode;
}

template <int Type>
bool XMLDom::const_data_iterator<Type>::operator!=(const const_data_iterator& other) const
{
	return mNode != other.mNode;
}

template <int Type>
XMLDom::const_data_iterator<Type>& XMLDom::const_data_iterator<Type>::operator++()
{
	if (mNode == NULL)
		return *this;

	mNode = mNode->next;

	while (mNode != NULL)
	{
		if (mNode->type == sXMLNodeTypeMap[Type])
			return *this;
		mNode = mNode->next;
	}

	return *this;
}

template <int Type>
XMLDom::const_data_iterator<Type>& XMLDom::const_data_iterator<Type>::operator--()
{
	if (mNode == NULL)
		return *this;

	mNode = mNode->prev;

	while (mNode != NULL)
	{
		if (mNode->type == sXMLNodeTypeMap[Type])
			return *this;
		mNode = mNode->prev;
	}

	return *this;
}

template <int Type>
string XMLDom::const_data_iterator<Type>::operator*() const
{
	if (mNode == NULL || mNode->content == NULL)
		return "";

	return (char*)mNode->content;
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::data_iterator

template <int Type>
XMLDom::data_iterator<Type>::data_iterator() :
	mNode(NULL)
{
}

template <int Type>
XMLDom::data_iterator<Type>::data_iterator(xmlNode* iNode) :
	mNode(iNode)
{
}

template <int Type>
bool XMLDom::data_iterator<Type>::operator==(const data_iterator& other) const
{
	return mNode == other.mNode;
}

template <int Type>
bool XMLDom::data_iterator<Type>::operator!=(const data_iterator& other) const
{
	return mNode != other.mNode;
}

template <int Type>
XMLDom::data_iterator<Type>& XMLDom::data_iterator<Type>::operator++()
{
	if (mNode == NULL)
		return *this;

	mNode = mNode->next;

	while (mNode != NULL)
	{
		if (mNode->type == sXMLNodeTypeMap[Type])
			return *this;
		mNode = mNode->next;
	}

	return *this;
}

template <int Type>
XMLDom::data_iterator<Type>& XMLDom::data_iterator<Type>::operator--()
{
	if (mNode == NULL)
		return *this;

	mNode = mNode->prev;

	while (mNode != NULL)
	{
		if (mNode->type == sXMLNodeTypeMap[Type])
			return *this;
		mNode = mNode->prev;
	}

	return *this;
}

template <int Type>
string XMLDom::data_iterator<Type>::operator*() const
{
	if (mNode == NULL || mNode->content == NULL)
		return "";

	return (char*)mNode->content;
}

template <int Type>
const XMLDom::data_iterator<Type>& XMLDom::data_iterator<Type>::operator=(const string& value)
{
	if (mNode == NULL)
		return *this;

	mNode->content = xmlCharStrndup(value.c_str(), value.size());
	return *this;
}

template <int Type>
XMLDom::data_iterator<Type> XMLDom::data_iterator<Type>::remove()
{
	if (mNode == NULL)
		return *this;

	_xmlNode* nextNode = mNode->next;
	xmlUnlinkNode(mNode);
	xmlFreeNode(mNode);
	mNode = nextNode;
	return *this;
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::iterator_base

XMLDom::iterator_base::iterator_base(xmlNode* iNode, const std::string& name) :
	mNode(iNode),
	mName(name)
{
}

string XMLDom::iterator_base::operator*() const
{
	if (mNode == NULL || mNode->name == NULL)
		return "";
	return (char*)mNode->name;
}

XMLDom::NameSpace XMLDom::iterator_base::nameSpace() const
{
	NameSpace ns;
	if (mNode != NULL && mNode->ns != NULL)
	{
		if (mNode->ns->href != NULL)
			 ns.href = (char*)mNode->ns->href;
		if (mNode->ns->prefix != NULL)
			 ns.prefix = (char*)mNode->ns->prefix;
	}
	return ns;
}

string XMLDom::iterator_base::uri() const
{
	if (mNode == NULL)
		return "";

	if (mNode->_private!=NULL) {
		XMLSourceInfo* info = (XMLSourceInfo*)mNode->_private;
		return info->uri;
	}

	if(mNode->doc == NULL || mNode->doc->URL == NULL)
		return "";

	return (char*)mNode->doc->URL;
}

void XMLDom::iterator_base::setUri(const std::string& uri)
{
	if (mNode == NULL)
		return;
	XMLsetSource(obtainSourceInfo(uri), mNode);
}

uint32 XMLDom::iterator_base::line() const
{
	if (mNode == NULL)
		return 0;
	return mNode->line;
}

XMLDom::const_sibling_iterator XMLDom::iterator_base::cparent() const
{
	if (mNode == NULL)
		return const_sibling_iterator(NULL);
	return const_sibling_iterator(mNode->parent);
}

XMLDom::const_sibling_iterator XMLDom::iterator_base::cbegin() const
{
	if (mNode == NULL)
		return const_sibling_iterator(NULL);

	xmlNode* n = mNode->children;
	while (n != NULL)
	{
		if (n->type == XML_ELEMENT_NODE)
			return const_sibling_iterator(n);
		n = n->next;
	}

	return const_sibling_iterator(NULL);
}

XMLDom::const_sibling_iterator XMLDom::iterator_base::cbegin(const std::string& name) const
{
	const_sibling_iterator i = std::find(begin(), end(), name);
	return const_sibling_iterator(i.mNode, name);
}

XMLDom::const_sibling_iterator XMLDom::iterator_base::cend() const
{
	return const_sibling_iterator(NULL);
}

XMLDom::const_attribute_iterator XMLDom::iterator_base::attribute_cbegin() const
{
	if (mNode == NULL)
		return const_attribute_iterator(NULL, NULL);

	xmlAttr* n = mNode->properties;
	while (n != NULL)
	{
		if (n->type == XML_ATTRIBUTE_NODE)
			return const_attribute_iterator(mNode, n);
		n = n->next;
	}

	return const_attribute_iterator(NULL, NULL);
}

XMLDom::const_attribute_iterator XMLDom::iterator_base::attribute_cend() const
{
	return const_attribute_iterator(NULL, NULL);
}

XMLDom::const_attribute_iterator XMLDom::iterator_base::find_attribute(const string& name) const
{
	return std::find_if(attribute_begin(), attribute_end(), 
	                    evalFirstOf<Attribute>(std::equal_to<string>(), name));
}

XMLDom::const_content_iterator XMLDom::iterator_base::content_cbegin() const
{
	if (mNode == NULL)
		return const_content_iterator(NULL);

	xmlNode* n = mNode->children;
	while (n != NULL)
	{
		if (n->type == XML_TEXT_NODE || n->type == XML_CDATA_SECTION_NODE)
			return const_content_iterator(n);
		n = n->next;
	}

	return const_content_iterator(NULL);
}

XMLDom::const_content_iterator XMLDom::iterator_base::content_cend() const
{
	return const_content_iterator(NULL);
}

XMLDom::const_comment_iterator XMLDom::iterator_base::comment_cbegin() const
{
	if (mNode == NULL)
		return const_comment_iterator(NULL);

	xmlNode* n = mNode->children;
	while (n != NULL)
	{
		if (n->type == XML_COMMENT_NODE)
			return const_comment_iterator(n);
		n = n->next;
	}

	return const_comment_iterator(NULL);
}

XMLDom::const_comment_iterator XMLDom::iterator_base::comment_cend() const
{
	return const_comment_iterator(NULL);
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::const_sibling_iterator

XMLDom::const_sibling_iterator::const_sibling_iterator() :
	iterator_base(NULL, "")
{
}

XMLDom::const_sibling_iterator::const_sibling_iterator(xmlNode* iNode, const std::string& name) :
	iterator_base(iNode, name)
{
}

XMLDom::const_sibling_iterator& XMLDom::const_sibling_iterator::operator++()
{
	if (mNode == NULL)
		return *this;
	mNode = mNode->next;
	if (mName.empty())
	{
		while (mNode != NULL)
		{
			if (mNode->type == XML_ELEMENT_NODE)
				break;
			mNode = mNode->next;
		}
	}
	else
	{
		while (mNode && (mNode->type != XML_ELEMENT_NODE || !mNode->name || string((char*)mNode->name) != mName))
			mNode = mNode->next;
	}
	return *this;
}

XMLDom::const_sibling_iterator& XMLDom::const_sibling_iterator::operator+=(std::size_t increment)
{
	for(std::size_t i = 0; i < increment; i++)
		this->operator ++();
	return *this;
}

XMLDom::const_sibling_iterator XMLDom::const_sibling_iterator::operator+(std::size_t increment) const
{
	XMLDom::const_sibling_iterator ret = *this;
	ret += increment;
	return ret;
}

XMLDom::const_sibling_iterator& XMLDom::const_sibling_iterator::operator--()
{
	if (mNode == NULL)
		return *this;
	mNode = mNode->prev;
	if (mName.empty())
	{
		while (mNode != NULL)
		{
			if (mNode->type == XML_ELEMENT_NODE)
				break;
			mNode = mNode->prev;
		}
	}
	else
	{
		while (mNode && (mNode->type != XML_ELEMENT_NODE || !mNode->name || string((char*)mNode->name) != mName))
			mNode = mNode->prev;
	}
	return *this;
}

XMLDom::const_sibling_iterator XMLDom::const_sibling_iterator::find(const std::string& name,
                                                                    std::size_t nth) const
{
	std::vector<std::string> children;
	boost::split(children, name, boost::is_from_range('/','/'));
	if (children.size() == 0)
		return const_sibling_iterator(NULL);

	const_sibling_iterator parentNode = *this;
	for(std::size_t i=0; i<children.size()-1; ++i)
		parentNode = std::find(parentNode.begin(), parentNode.end(), children[i]);

	std::string leaf = children.back();
	const_sibling_iterator item = std::find(parentNode.begin(), parentNode.end(), leaf);
	for(std::size_t i = 0; i < nth && item!=parentNode.end(); ++i)
		item = std::find(++item, parentNode.end(), leaf);
	return item;
}


///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom::sibling_iterator

XMLDom::sibling_iterator::sibling_iterator() :
	iterator_base(NULL, "")
{
}

XMLDom::sibling_iterator::sibling_iterator(xmlNode* iNode, const std::string& name) :
	iterator_base(iNode, name)
{
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::operator++()
{
	if (mNode == NULL)
		return *this;
	mNode = mNode->next;
	if (mName.empty())
	{
		while (mNode != NULL)
		{
			if (mNode->type == XML_ELEMENT_NODE)
				break;
			mNode = mNode->next;
		}
	}
	else
	{
		while (mNode && (mNode->type != XML_ELEMENT_NODE || !mNode->name || string((char*)mNode->name) != mName))
			mNode = mNode->next;
	}
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::operator+=(std::size_t increment)
{
	for(std::size_t i = 0; i < increment; i++)
		this->operator ++();
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::operator+(std::size_t increment) const
{
	XMLDom::sibling_iterator ret = *this;
	ret += increment;
	return ret;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::operator--()
{
	if (mNode == NULL)
		return *this;
	mNode = mNode->prev;
	if (mName.empty())
	{
		while (mNode != NULL)
		{
			if (mNode->type == XML_ELEMENT_NODE)
				break;
			mNode = mNode->prev;
		}
	}
	else
	{
		while (mNode && (mNode->type != XML_ELEMENT_NODE || !mNode->name || string((char*)mNode->name) != mName))
			mNode = mNode->prev;
	}
	return *this;
}

const XMLDom::sibling_iterator& XMLDom::sibling_iterator::operator=(const string& name)
{
	if (mNode  == NULL || mNode->name == NULL)
		return *this;

	mNode->name = xmlCharStrndup(name.c_str(), name.size());
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::parent()
{
	if (mNode == NULL)
		return sibling_iterator(NULL);
	return sibling_iterator(mNode->parent);
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::begin()
{
	if (mNode == NULL)
		return sibling_iterator(NULL);

	xmlNode* n = mNode->children;
	while (n != NULL)
	{
		if (n->type == XML_ELEMENT_NODE )
			return sibling_iterator(n);
		n = n->next;
	}

	return sibling_iterator(NULL);
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::begin(const std::string& name)
{
	sibling_iterator i = std::find(begin(), end(), name);
	return sibling_iterator(i.mNode, name);
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::end()
{
	return sibling_iterator(NULL);
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::find(const string& name, std::size_t nth)
{
	std::vector<std::string> children;
	boost::split(children, name, boost::is_from_range('/','/'));
	if (children.size() == 0)
		return sibling_iterator(NULL);

	sibling_iterator parentNode = *this;
	for(std::size_t i=0; i<children.size()-1; ++i)
		parentNode = std::find(parentNode.begin(), parentNode.end(), children[i]);

	std::string leaf = children.back();
	sibling_iterator item = std::find(parentNode.begin(), parentNode.end(), leaf);
	for(std::size_t i = 0; i < nth && item!=parentNode.end(); ++i)
		item = std::find(++item, parentNode.end(), leaf);
	return item;
}

XMLDom::content_iterator XMLDom::sibling_iterator::content_begin()
{
	if (mNode == NULL)
		return content_iterator(NULL);

	xmlNode* n = mNode->children;
	while (n != NULL)
	{
		if (n->type == XML_TEXT_NODE || n->type == XML_CDATA_SECTION_NODE)
			return content_iterator(n);
		n = n->next;
	}

	return content_iterator(NULL);
}

XMLDom::content_iterator XMLDom::sibling_iterator::content_end()
{
	return content_iterator(NULL);
}

XMLDom::comment_iterator XMLDom::sibling_iterator::comment_begin()
{
	if (mNode == NULL)
		return comment_iterator(NULL);

	xmlNode* n = mNode->children;
	while (n != NULL)
	{
		if (n->type == XML_COMMENT_NODE)
			return comment_iterator(n);
		n = n->next;
	}

	return comment_iterator(NULL);
}

XMLDom::comment_iterator XMLDom::sibling_iterator::comment_end()
{
	return comment_iterator(NULL);
}

XMLDom::attribute_iterator XMLDom::sibling_iterator::attribute_begin()
{
	if (mNode == NULL)
		return attribute_iterator(NULL, NULL);

	xmlAttr* n = mNode->properties;
	while (n != NULL)
	{
		if (n->type == XML_ATTRIBUTE_NODE)
			return attribute_iterator(mNode, n);
		n = n->next;
	}

	return attribute_iterator(NULL, NULL);
}

XMLDom::attribute_iterator XMLDom::sibling_iterator::attribute_end()
{
	return attribute_iterator(NULL, NULL);
}

XMLDom::attribute_iterator XMLDom::sibling_iterator::find_attribute(const string& name)
{
	return std::find_if(attribute_begin(), attribute_end(), 
	                    evalFirstOf<Attribute>(std::equal_to<string>(), name));
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::add_attribute(const Attribute& attribute)
{
	xmlSetProp(mNode, BAD_CAST attribute.first.c_str(),
	           BAD_CAST attribute.second.c_str());
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::add_attribute(const string& pName,
                                                                  const string& pValue)
{
	if (xmlValidateName(BAD_CAST pName.c_str(), 0) != 0)
		MIRA_THROW(XIO, "'" << pName << "' is not a valid xml element name."
		           " Element names can contain numbers, letters and other"
		           " characters but must start with a letter."
		           " Elements names must not start with 'xml'");
	xmlSetProp(mNode, BAD_CAST pName.c_str(), BAD_CAST pValue.c_str());
	return *this;
}

void XMLDom::sibling_iterator::remove_attribute(attribute_iterator it)
{
	if(it.mNode == NULL)
		return; // do nothing for "empty" end sibling_iterator

	if(it.mNode != this->mNode)
		MIRA_THROW(XIO, "The specified attribute does not belong to this node");

	xmlRemoveProp(it.mAttribute);
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::add_comment(const string& comment)
{
	xmlNode* childnode = xmlNewComment(BAD_CAST comment.c_str());
	xmlAddChild(mNode, childnode);
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::add_content(const string& content)
{
	xmlNodeAddContent(mNode, BAD_CAST content.c_str());
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::add_cdata_content(const string& cdata_content)
{
	xmlNode* cdata = xmlNewCDataBlock(mNode->doc, BAD_CAST cdata_content.c_str(), cdata_content.size());
	xmlAddChild(mNode, cdata);
	return *this;
}

void validateNodeName(const string& name)
{
	if (xmlValidateName(BAD_CAST name.c_str(), 0) != 0)
		MIRA_THROW(XIO, "'" << name << "' is not a valid xml element name."
		           " Element names can contain numbers, letters and other"
		           " characters but must start with a letter."
		           " Elements names must not start with 'xml'");
}

_xmlNode* insertNodeBefore(_xmlNode* current, _xmlNode* newNode)
{
	if (current == NULL || current->parent == NULL)
		return current;

	if (newNode != NULL)
	{
		if(newNode->doc==current->doc)
			return xmlAddPrevSibling(current, newNode);

		// make sure to preserve the original source location,
		// if the document has changed
		XMLSourceInfoPtr sourceInfo = obtainSourceInfo(newNode->doc->URL);

		_xmlNode* addedNode = xmlAddPrevSibling(current, newNode); // changes newNode->doc!

		XMLpreserveSource(sourceInfo, newNode, addedNode);

		return addedNode;
	}
	return current;
}

_xmlNode* insertNodeAfter(_xmlNode* current, _xmlNode* newNode)
{
	if (current == NULL || current->parent == NULL)
		return current;

	if (newNode != NULL)
	{
		if(newNode->doc==current->doc)
			return xmlAddNextSibling(current, newNode);

		// make sure to preserve the original source location,
		// if the document has changed
		XMLSourceInfoPtr sourceInfo = obtainSourceInfo(newNode->doc->URL);

		_xmlNode* addedNode = xmlAddNextSibling(current, newNode); // changes newNode->doc!

		XMLpreserveSource(sourceInfo, newNode, addedNode);

		return addedNode;
	}
	return current;
}

_xmlNode* replaceNode(_xmlNode* current, _xmlNode* newNode)
{
	if (newNode != NULL)
	{
		_xmlNode* oldNode = xmlReplaceNode(current, newNode);
		xmlFreeNode(oldNode);
		return newNode;
	}
	return current;
}

_xmlNode* createNewNode(_xmlNode* current, const std::string& name, const XMLDom::NameSpace& ns)
{
	validateNodeName(name);

	xmlNs* usedNS = NULL;

	if(ns.prefix.empty())
		//Retrieve default namespace if it exists
		usedNS = xmlSearchNs(current->doc, current, 0);
	else
	{
		//Use the existing namespace if one exists:
		usedNS = xmlSearchNs(current->doc, current, BAD_CAST ns.prefix.c_str());
		if (usedNS == NULL)
			MIRA_THROW(XIO, "The namespace prefix (" + ns.prefix + ") has not been declared.");
	}

	return xmlNewNode(usedNS, BAD_CAST name.c_str());
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::add_child(const string& name, const NameSpace& ns)
{
	validateNodeName(name);
	_xmlNode* newNode = createNewNode(mNode, name, ns);
	newNode = xmlAddChild(mNode, newNode);
	return XMLDom::sibling_iterator(newNode);
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::add_child(const XMLDom::const_sibling_iterator& node)
{
	if (mNode == NULL || mNode->parent == NULL)
		return *this;

	_xmlNode* newNode = XMLcopyNode(node.mNode, 1);

	if (newNode != NULL) {
		newNode =xmlAddChild(mNode, newNode);

		// make sure to preserve the original source location,
		// if the document has changed
		if(node.mNode->doc!=mNode->doc) {
			XMLpreserveSource(obtainSourceInfo(node.mNode->doc->URL),
			                  node.mNode, newNode);
		}
	}

	return XMLDom::sibling_iterator(newNode);
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::insert_before(const XMLDom::const_sibling_iterator& node)
{
	if (mNode == NULL || mNode->parent == NULL)
		return *this;

	_xmlNode* newNode = XMLcopyNode(node.mNode, 1);
	return XMLDom::sibling_iterator(insertNodeBefore(mNode, newNode));
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::insert_before(const std::string& name, const NameSpace& ns)
{
	if (mNode == NULL || mNode->parent == NULL)
		return *this;
	_xmlNode* newNode = createNewNode(mNode, name, ns);
	return XMLDom::sibling_iterator(insertNodeBefore(mNode, newNode));
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::insert_after(const XMLDom::const_sibling_iterator& node)
{
	if (mNode == NULL || mNode->parent == NULL)
		return *this;
	_xmlNode* newNode = XMLcopyNode(node.mNode, 1);
	return XMLDom::sibling_iterator(insertNodeAfter(mNode, newNode));
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::insert_after(const std::string& name, const NameSpace& ns)
{
	if (mNode == NULL || mNode->parent == NULL)
		return *this;
	validateNodeName(name);
	_xmlNode* newNode = createNewNode(mNode, name, ns);
	return XMLDom::sibling_iterator(insertNodeAfter(mNode, newNode));
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_comment_before(const XMLDom::const_comment_iterator& comment)
{
	if (mNode != NULL)
		insertNodeBefore(mNode, XMLcopyNode(comment.mNode, 1));
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_comment_before(const std::string& comment)
{
	if (mNode != NULL)
		insertNodeBefore(mNode, xmlNewComment(BAD_CAST comment.c_str()));
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_comment_after(const XMLDom::const_comment_iterator& comment)
{
	if (mNode != NULL)
		insertNodeAfter(mNode, XMLcopyNode(comment.mNode, 1));
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_comment_after(const std::string& comment)
{
	if (mNode != NULL)
		insertNodeAfter(mNode, xmlNewComment(BAD_CAST comment.c_str()));
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_content_before(const XMLDom::const_content_iterator& content)
{
	if (mNode != NULL)
	{
		_xmlNode* newNode = XMLcopyNode(content.mNode, 1);
		_xmlNode* tmp = insertNodeBefore(mNode, newNode);
		if (tmp == newNode)
			tmp = xmlTextMerge(tmp->prev, tmp);
	}
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_content_before(const std::string& content)
{
	if (mNode != NULL)
	{
		_xmlNode* newNode = xmlNewText(BAD_CAST content.c_str());
		_xmlNode* tmp = insertNodeBefore(mNode, newNode);
		if (tmp == newNode)
			tmp = xmlTextMerge(tmp->prev, tmp);
	}
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_content_after(const XMLDom::const_content_iterator& content)
{
	if (mNode != NULL)
	{
		_xmlNode* newNode = XMLcopyNode(content.mNode, 1);
		_xmlNode* tmp = insertNodeAfter(mNode, newNode);
		if (tmp == newNode)
			tmp = xmlTextMerge(tmp, tmp->next);
	}
	return *this;
}

XMLDom::sibling_iterator& XMLDom::sibling_iterator::insert_content_after(const std::string& content)
{
	if (mNode != NULL)
	{
		_xmlNode* newNode = xmlNewText(BAD_CAST content.c_str());
		_xmlNode* tmp = insertNodeAfter(mNode, newNode);
		if (tmp == newNode)
			tmp = xmlTextMerge(tmp, tmp->next);
	}
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::replace(const XMLDom::const_sibling_iterator& node)
{
	if (mNode == NULL)
		return *this;
	mNode = replaceNode(mNode, XMLcopyNode(node.mNode, 1));
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::replace(const std::string& name, const NameSpace& ns)
{
	if (mNode == NULL)
		return *this;
	mNode = replaceNode(mNode, createNewNode(mNode, name, ns));
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::replace_by_comment(const XMLDom::const_comment_iterator& node)
{
	if (mNode == NULL)
		return *this;
	_xmlNode* oldNode = mNode;
	this->operator ++();
	replaceNode(oldNode, XMLcopyNode(node.mNode, 1));
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::replace_by_comment(const std::string& comment)
{
	if (mNode == NULL)
		return *this;
	_xmlNode* oldNode = mNode;
	this->operator ++();
	replaceNode(oldNode, xmlNewComment(BAD_CAST comment.c_str()));
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::replace_by_content(const XMLDom::const_content_iterator& node)
{
	if (mNode == NULL)
		return *this;
	_xmlNode* oldNode = mNode;
	this->operator ++();
	_xmlNode* newNode = XMLcopyNode(node.mNode, 1);
	_xmlNode* tmp = replaceNode(oldNode, newNode);
	if (tmp == newNode)
	{
		tmp = xmlTextMerge(tmp->prev, tmp);
		tmp = xmlTextMerge(tmp, tmp->next);
	}
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::replace_by_content(const std::string& content)
{
	if (mNode == NULL)
		return *this;
	_xmlNode* oldNode = mNode;
	this->operator ++();
	_xmlNode* newNode = xmlNewText(BAD_CAST content.c_str());
	_xmlNode* tmp = replaceNode(oldNode, newNode);
	if (tmp == newNode)
	{
		tmp = xmlTextMerge(tmp->prev, tmp);
		tmp = xmlTextMerge(tmp, tmp->next);
	}
	return *this;
}

XMLDom::sibling_iterator XMLDom::sibling_iterator::remove()
{
	if (mNode == NULL)
		return *this;

	_xmlNode* nextNode = mNode->next;
	xmlUnlinkNode(mNode);
	xmlFreeNode(mNode);
	mNode = nextNode;
	return *this;
}

///////////////////////////////////////////////////////////////////////////////
// Implementation of XMLDom

XMLDom::XMLDom(const std::string& rootNodeName) :
	mXMLDocument(NULL)
{
	mXMLDocument = xmlNewDoc(BAD_CAST XML_DEFAULT_VERSION);
	if (!mXMLDocument)
		MIRA_THROW(XBadAlloc, "Could not create XML document");

	_xmlNode* rootNode = xmlNewDocNode(mXMLDocument, 0, 
	                                   BAD_CAST rootNodeName.c_str(), 0);
	if (rootNode == NULL)
		MIRA_THROW(XBadAlloc, "Could not create XML root node");

	xmlDocSetRootElement(mXMLDocument, rootNode);
}

XMLDom::~XMLDom()
{
	if(mXMLDocument != NULL)
		xmlFreeDoc(mXMLDocument);
}

XMLDom::XMLDom(XMLDom&& other) : mXMLDocument(NULL)
{
	std::swap(mXMLDocument, other.mXMLDocument);
}

XMLDom& XMLDom::operator=(XMLDom&& other)
{
	std::swap(mXMLDocument, other.mXMLDocument);
	return *this;
}

void XMLDom::clear()
{
	// free old document
	xmlFreeDoc(mXMLDocument);

	mXMLDocument = xmlNewDoc(BAD_CAST XML_DEFAULT_VERSION);
	if (mXMLDocument == NULL)
		MIRA_THROW(XBadAlloc, "Could not create XML document");

	_xmlNode* rootNode = xmlNewDocNode(mXMLDocument, 0, BAD_CAST "root", 0);
	if (rootNode == NULL)
		MIRA_THROW(XBadAlloc, "Could not create XML root node");

	xmlDocSetRootElement(mXMLDocument, rootNode);
}

void XMLDom::loadFromString(const string& buffer)
{
	xmlFreeDoc(mXMLDocument);
	mXMLDocument = NULL;
	mXMLDocument = xmlReadMemory(buffer.c_str(), buffer.size(), "xml", 
	                             NULL, XML_PARSE_NOBLANKS);

	if (mXMLDocument == NULL)
		MIRA_THROW(XIO, "Cannot load xml document from memory");

	if (xmlDocGetRootElement(mXMLDocument) == NULL)
		MIRA_THROW(XIO, "Cannot load xml document from memory - no root node found");
}

void XMLDom::loadFromFile(const Path& filename)
{
	Path path = resolvePath(filename);

	if (!boost::filesystem::exists(path))
		MIRA_THROW(XFileNotFound, "The file " << path.string() << " does not exist!");

	xmlFreeDoc(mXMLDocument);
	mXMLDocument = NULL;

	mXMLDocument = xmlReadFile(path.string().c_str(),NULL, XML_PARSE_NOBLANKS);

	if (mXMLDocument == NULL)
		MIRA_THROW(XIO, "Cannot load xml document from file " << path.string());

	if (xmlDocGetRootElement(mXMLDocument) == NULL)
		MIRA_THROW(XIO, "Cannot load xml document from file " 
		          << path.string() << " - no root node found");
}

void XMLDom::saveToFile(const Path& filename, const string& encoding) const
{
	if (mXMLDocument == NULL)
		MIRA_THROW(XInvalidConfig, "No xml document loaded. "
		           "Cannot save empty xml document!");

	Path path = resolvePath(filename);
	Path folder = path.parent_path();
	// create destination folder if not exists
	if (!boost::filesystem::exists(folder))
		boost::filesystem::create_directory(folder);

	if (xmlSaveFormatFileEnc(path.string().c_str(), 
	                         mXMLDocument, encoding.c_str(), 1) < 0)
		MIRA_THROW(XIO, "Failed to write xml document to file " << filename);
}

std::string XMLDom::saveToString(const string& encoding) const
{
	if (mXMLDocument == NULL)
		MIRA_THROW(XInvalidConfig, "No xml document loaded. "
		           "Cannot save empty xml document!");

	xmlBuffer* buffer = xmlBufferCreate();
	xmlOutputBuffer* outputBuffer = xmlOutputBufferCreateBuffer(buffer, NULL);
	if (xmlSaveFormatFileTo(outputBuffer, mXMLDocument, encoding.c_str(), 1) < 0)
	{
		xmlBufferFree(buffer);
		MIRA_THROW(XIO, "Failed to write xml document to string");
	}
	std::string r((char*)buffer->content, buffer->use);
	xmlBufferFree(buffer);
	return r;
}

XMLDom::const_sibling_iterator XMLDom::croot() const
{
	return const_sibling_iterator(xmlDocGetRootElement(mXMLDocument));
}

XMLDom::sibling_iterator XMLDom::root()
{
	return sibling_iterator(xmlDocGetRootElement(mXMLDocument));
}

std::string XMLDom::uri() const
{
	if (mXMLDocument == NULL || mXMLDocument->URL == NULL)
		return "";
	return (char*)mXMLDocument->URL;
}

void XMLDom::setUri(const std::string& uri)
{
	if (mXMLDocument == NULL)
		return;
	mXMLDocument->URL = xmlCharStrndup(uri.c_str(), uri.size());
}

std::string XMLDom::encoding() const
{
	if (mXMLDocument == NULL || mXMLDocument->encoding == NULL)
			return "";
	return (char*)mXMLDocument->encoding;
}

///////////////////////////////////////////////////////////////////////////////

template class XMLDom::const_data_iterator<XMLDom::COMMENT_NODE>;
template class XMLDom::const_data_iterator<XMLDom::TEXT_NODE>;
template class XMLDom::data_iterator<XMLDom::COMMENT_NODE>;
template class XMLDom::data_iterator<XMLDom::TEXT_NODE>;

///////////////////////////////////////////////////////////////////////////////

}
