/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MetaSerializer.C
 *    Implementation of MetaSeriaizer.h.
 *
 * @author Erik Einhorn
 * @date   2011/07/13
 */

#include <serialization/MetaSerializer.h>

#include <error/Exceptions.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

void TypeMeta::setAtomicType(const std::string& typenam)
{
	assert(Base::size()==0);
	if(typenam=="char") setBaseAtomicType(TYPE_INT8);
	else if(typenam=="signed char") setBaseAtomicType(TYPE_INT8);
	else if(typenam=="unsigned char") setBaseAtomicType(TYPE_UINT8);
	else if(typenam=="short") setBaseAtomicType(TYPE_INT16);
	else if(typenam=="unsigned short") setBaseAtomicType(TYPE_UINT16);
	else if(typenam=="int") setBaseAtomicType(TYPE_INT32);
	else if(typenam=="unsigned int") setBaseAtomicType(TYPE_UINT32);
	else if(typenam=="int64") setBaseAtomicType(TYPE_INT64);
	else if(typenam=="unsigned int64") setBaseAtomicType(TYPE_UINT64);
	else if(typenam=="float") setBaseAtomicType(TYPE_FLOAT);
	else if(typenam=="double") setBaseAtomicType(TYPE_DOUBLE);
	else if(typenam=="std::string") setBaseAtomicType(TYPE_STRING);
	else if(typenam=="bool")   setBaseAtomicType(TYPE_BOOL);
	else if(typenam=="enum")   setBaseAtomicType(TYPE_ENUMERATION);
	// our own type strings (should never occur here actually)
	else if(typenam=="int8")   setBaseAtomicType( TYPE_INT8);
	else if(typenam=="uint8")  setBaseAtomicType(TYPE_UINT8);
	else if(typenam=="int16")  setBaseAtomicType( TYPE_INT16);
	else if(typenam=="uint16") setBaseAtomicType(TYPE_UINT16);
	else if(typenam=="int32")  setBaseAtomicType( TYPE_INT32);
	else if(typenam=="uint32") setBaseAtomicType(TYPE_UINT32);
	else if(typenam=="uint64") setBaseAtomicType(TYPE_UINT64);
	else if(typenam=="string") setBaseAtomicType(TYPE_STRING);
	else {
		push_back(TYPE_ATOMIC);
		mIdentifier = typenam;
	}
}

std::string TypeMeta::getTypename() const
{
	switch(getType())
	{
		case TYPE_CLASS:   return getIdentifier();
		case TYPE_ATOMIC:  return getIdentifier();
		case TYPE_VOID:    return "void";
		case TYPE_UINT8:   return "unsigned char";
		case TYPE_INT8:    return "char";
		case TYPE_UINT16:  return "unsigned short";
		case TYPE_INT16:   return "short";
		case TYPE_UINT32:  return "unsigned int";
		case TYPE_INT32:   return "int";
		case TYPE_UINT64:  return "unsigned int64";
		case TYPE_INT64:   return "int64";
		case TYPE_FLOAT:   return "float";
		case TYPE_DOUBLE:  return "double";
		case TYPE_STRING:  return "std::string";
		case TYPE_BOOL:    return "bool";
		case TYPE_ENUMERATION:  return "enum";
		default:           return "invalid";
	}
}

std::string TypeMeta::toString() const
{
	std::string r;

	const std::vector<uint8>& This = *this;
	foreach(uint8 v, This)
	{
		switch(v)
		{
			case TYPE_COLLECTION:   r = "[]" + r; break;
			case TYPE_POINTER: r = "@" + r; break;
			case TYPE_PROPERTY: break; // r = "property " + r; break;
			case TYPE_VERSION: r = "version" + r; break;
			case TYPE_ATOMIC:  r = "atomic " + getTypename() + r; break;
			default:  r = getTypename() + r; break;
		}
	}

	return r;
}


///////////////////////////////////////////////////////////////////////////////

std::string CompoundMeta::toString() const
{
	const std::string eol = "\n";

	std::string r;
	if(!mVersion)
		r+= " (no version)\n";
	else
		r+= " (version=" + mira::toString(*mVersion) + ")\n";
	if (!members.empty())
		r+= "-members\n";
	foreach(const Member& m, members)
		r += "    " + m.type->toString() + " " + m.name + " (" + m.comment + ")" + eol;
	if (!interfaces.empty())
		r+= "-interfaces\n";
	foreach(const std::string& i, interfaces)
		r += "    " + i + eol;
	if (!methods.empty())
		r+= "-methods\n";
	foreach(const MethodMetaPtr& m, methods)
	{
		r += "    " + m->returnType->toString();
		r += " " + m->name;
		r += "(";
		for(auto p = m->parameters.begin(); p!=m->parameters.end(); ++p)
		{
			if(p!=m->parameters.begin())
				r += ",";
			r += p->type->toString();
			if (!p->name.empty())
				r += " " + p->name;
		}
		r += ")" + eol;
		r += "        " + m->comment + eol;
		for(auto p = m->parameters.begin(); p!=m->parameters.end(); ++p)
		{
			if (!p->name.empty() && !p->description.empty())
				r += "        " + p->name + " \t: " + p->description + eol;
		}		
	}
	return r;
}

///////////////////////////////////////////////////////////////////////////////

void MetaTypeDatabase::generateDependentTypesDB(TypeMetaPtr type,
                                                MetaTypeDatabase& db)
{
	if(type->getType()!=TypeMeta::TYPE_CLASS)
		return;

	auto it = this->find(type->getIdentifier());
	if(it==this->end()) {
		MIRA_THROW(XLogical, "Cannot provide all required meta information "
			"since the class '" << type->getIdentifier() <<
			"' is missing in the database");
	}

	// insert the required type
	db.insert(*it);

	CompoundMetaPtr meta = it->second;

	// handle members recursively
	foreach(CompoundMeta::Member& m, meta->members)
		generateDependentTypesDB(m.type,db);
}


///////////////////////////////////////////////////////////////////////////////

}

