/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file CallStackLinux.C
 *    Implementation of OS dependent parts in CallStack.h
 *
 * @author Erik Einhorn
 * @date   2009/12/26
 */

#include <error/CallStack.h>

#include <utils/ToString.h>

#include <error/Logging.h>

#include <boost/format.hpp>
#include <boost/system/error_code.hpp>
#include <boost/algorithm/string/trim.hpp>

using namespace boost;
using namespace std;

// these macros are defined by the build system depending whether
// libiberty.a libbfd.a, demangle.h and bfd.h are available
#ifdef CALLSTACK_LINUX_USE_DEMANGLE
#  include <string.h>
#define HAVE_DECL_BASENAME 1

#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
#  pragma GCC diagnostic push
#  pragma GCC diagnostic ignored "-Wredundant-decls"
# endif
#endif

#  include <demangle.h>

#if defined(MIRA_GNUC_VERSION)
# if MIRA_GNUC_VERSION >= 40600
#  pragma GCC diagnostic pop
# endif
#endif

#endif

#ifdef CALLSTACK_LINUX_USE_SYMBOLTABLE
#  include <error/private/SymbolTableLinux.h>
#endif

namespace mira {

///////////////////////////////////////////////////////////////////////////////

string CallStack::Symbol::sourceLocation() const
{
#ifdef CALLSTACK_LINUX_USE_SYMBOLTABLE
	SymbolTable::SourceLocation loc =
		SymbolTable::forBinary(file.c_str())->sourceLocation(addressStr.c_str(),
			name.empty() ? NULL : name.c_str(), offset);
	if(loc.filename().empty())
		return "";
	else
		return loc.filename() + ":" + boost::lexical_cast<string>(loc.line());
#endif
	return "";
}

///////////////////////////////////////////////////////////////////////////////

CallStack CallStack::backtrace(size_t maxSize, size_t stackStart)
{
	maxSize+=stackStart;
	void** stack = new void*[maxSize];
	size_t size = ::backtrace(stack, maxSize);
	return CallStack(stack, size, stackStart);
}

void CallStack::obtainSymbols() const
{
	if(mStack==NULL) // we have no valid call stack
		return;

	if(!mSymbols.empty()) // we already have the symbols
		return;

	char** syms = backtrace_symbols(mStack.get(), mStackSize);
	if(syms==NULL)
		return; // failed to get the symbols for some reason

	mSymbols.clear();

	// parse each symbol
	for(size_t i=0; i<mStackSize; ++i)
	{
		string s(syms[i]);
		//std::cout << "sym: " << i << " - :" << s << std::endl;

		Symbol symbol;
		symbol.address = 0;

		size_t pos;
		size_t fileEnd = string::npos; // position of file end

		// extract the address (if available)
		pos = s.rfind("[");
		if(pos!=string::npos) {
			fileEnd = pos;
			pos++;
			size_t end = s.rfind("]");
			if(end!=string::npos) {
				symbol.addressStr = s.substr(pos, end-pos);
				symbol.address = fromString<int64>(symbol.addressStr);
			}
		}

		// extract the (mangled) symbol name
		symbol.name = "??"; // default
		symbol.offset = 0;
		pos = s.rfind("(");
		if(pos!=string::npos) {
			fileEnd = pos;
			pos++;
			size_t end = s.find_first_of("+-)", pos);
			if(end!=string::npos) {
				symbol.name = s.substr(pos, end-pos);

				pos = s.find_first_of("+-", end-1);
				if(pos!=string::npos) {
					end = s.find_first_of(")", end);
					string offset = s.substr(pos+1, end-pos-1);

					try {
						symbol.offset = fromString<FromHex<int64>>(offset);
					} catch(...) {}

				}
			}

		}

		// extract the binaries file name
		symbol.file = s.substr(0, fileEnd);
		trim(symbol.file);

		// demangle the symbol
		symbol.nameDemangled = symbol.name;
# ifdef CALLSTACK_LINUX_USE_DEMANGLE
		char* demangled_symb = cplus_demangle(symbol.name.c_str(), DMGL_PARAMS);
		if(demangled_symb!=NULL) {
			symbol.nameDemangled = demangled_symb;
			free(demangled_symb);
		}
# endif

		mSymbols.push_back(symbol);
	}
	free(syms);
}

///////////////////////////////////////////////////////////////////////////////

}
