/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file SQLiteDB.C
 *    Implementation of SQLiteDB.h
 *
 * @author Tim Langner
 * @date   2010/10/25
 */

#include <database/SQLiteDB.h>

#include <sqlite3.h>

#include <utils/MakeString.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

SQLiteDB::SQLiteDB(const Path& db)
{
	Path dbFile = resolvePath(db);
	if ( sqlite3_open(dbFile.string().c_str(), &mDB) != SQLITE_OK )
	{
		const char* msg = sqlite3_errmsg(mDB);
		MIRA_THROW(XSQLite, "Error opening SQLite database '" << db.string() <<
		           "'. Error: " << msg);
	}
}

SQLiteDB::~SQLiteDB()
{
	if (mDB)
	{
		sqlite3_close(mDB);
		mDB = NULL;
	}
}

bool SQLiteDB::tableExists(const std::string& table)
{
	std::string sql = MakeString() << "select count(*) from sqlite_master "
	    "where type='table' and name='" << table << "'";
	if ( query(sql).getValue<int>(0) == 0 )
		return false;
	return true;
}

SQLiteQuery SQLiteDB::query(const std::string& sql)
{
	// Prepare the sql query statement
	sqlite3_stmt* statement = prepare(sql);
	// Evaluate the prepared statement for the first time and check for errors
	int ret = sqlite3_step(statement);
	if ( ret == SQLITE_ERROR || ret == SQLITE_BUSY || ret == SQLITE_MISUSE )
		MIRA_THROW(XSQLite, "Query '" << sql << "' failed.");

	// If evaluation returned that no more results 
	// are available this query is done
	if ( ret == SQLITE_DONE )
		return SQLiteQuery(mDB, statement, false);
	// We have a query with more results available
	return SQLiteQuery(mDB, statement, true);
}

int SQLiteDB::exec(const std::string& sql)
{
	char* error = NULL;
	if ( sqlite3_exec(mDB, sql.c_str(), 0, 0, &error) != SQLITE_OK )
		MIRA_THROW(XSQLite, "Exec '" << sql << "' failed. Error: " << error);
	return sqlite3_changes(mDB);
}


sqlite3_stmt* SQLiteDB::prepare(const std::string& sql)
{
	const char* tail = NULL;
	sqlite3_stmt* statement;

	if ( sqlite3_prepare_v2(mDB, sql.c_str(), -1, &statement, &tail) != SQLITE_OK )
		MIRA_THROW(XSQLite, "Error preparing statement '" << sql << "'");

	return statement;
}

///////////////////////////////////////////////////////////////////////////////

}
