/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Print.h
 *    Provides print() method that uses serialization to print objects.
 *
 * @author Erik Einhorn
 * @date   2011/10/15
 */

#ifndef _MIRA_PRINT_H_
#define _MIRA_PRINT_H_

#include <serialization/JSONSerializer.h>
#include <platform/Platform.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////
namespace Private {
///@cond INTERNAL

// implemented in Print.C
MIRA_BASE_EXPORT void jsonPrettyPrint(const JSONValue& value, std::ostream& os,
		                             bool prettyPrint);

/**
 * Helper class that is used by print()
 * @see JSONSerializer.h
 */
template <typename T>
class TPrintHelper
{
public:
	TPrintHelper(const T& value, bool prettyPrint=true) :
		mValue(value), mPrettyPrint(prettyPrint) {}

	/**
	 * Output stream operator for writing a matrix to stream in a given format.
	 * @param s The output stream
	 * @param format The format class
	 * @return The stream
	 */
	friend std::ostream& operator<< (std::ostream & s, const TPrintHelper& This)
	{
		JSONSerializer ts(true);
		JSONValue val = ts.serialize(This.mValue);
		jsonPrettyPrint(val, s, This.mPrettyPrint);
		return s;
	}

private:

	/// The value that is written to the stream
	const T& mValue;
	bool mPrettyPrint;
};

}
///@endcond

/**
 * @ingroup SerializationModule
 *
 * Convenient function for simple output of a serializable class
 * to a stream.
 *
 * Usage:
 * \code
 *   SomeClass c; ...
 *
 *   cout << "MyClass: " << print(c) << endl;
 * \endcode
 *
 * @param name  The name of the object.
 * @param value The object or value that should be serialized.
 *
 * @see @ref SerializationPage
 */
template <typename T>
Private::TPrintHelper<T> print(const T& value, bool prettyPrint=true)
{
	return Private::TPrintHelper<T>(value, prettyPrint);
}


///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
