/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file UniformRandomGenerator.h
 *    A random generator that samples random numbers within the
 *    interval that is passed to the constructor.
 *
 * @author Erik Einhorn
 * @date   2012/11/10
 */

#ifndef _MIRA_UNIFORMRANDOMGENERATOR_H_
#define _MIRA_UNIFORMRANDOMGENERATOR_H_

#include <math/RandomGenerator.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

// helper to choose correct boost uniform random distribution depending on
// whether the datatype is integral or floating point.
///@cond INTERNAL
namespace detail {

// workaround for BOOST Bug #6053: https://svn.boost.org/trac/boost/ticket/6053
template <typename T>
class UniformRealBoostWorkaround : public boost::uniform_real<T>
{
	typedef boost::uniform_real<T> Base;
public:

	explicit UniformRealBoostWorkaround(T min_arg = T(0.0),
	                                    T max_arg = T(1.0)) :
	    Base(min_arg, max_arg) {}

	template<class Engine>
	typename Base::result_type operator()(Engine& eng) const
	{
		// special treatment for min==max, otherwise boost::uniform_real runs into infinite loop (Bug #6053)
		if(this->min()==this->max())
			return this->min();
		else
			return Base::operator()(eng);
	}
};

template <typename T, bool IsInteger> // T is real
struct UniformRandomDistributionHelper {
#if BOOST_VERSION >= 104700
	typedef UniformRealBoostWorkaround<T> type;
#else
	typedef boost::uniform_real<T> type;
#endif
};

template <typename T> // T is integer
struct UniformRandomDistributionHelper<T,true> {
	typedef boost::uniform_int<T> type;
};

template <typename T>
struct UniformRandomDistributionChooser :
	public UniformRandomDistributionHelper<T,std::numeric_limits<T>::is_integer>
{};
}
///@endcond INTERNAL

///////////////////////////////////////////////////////////////////////////////

/**
 * A random generator that samples random numbers within the
 * interval that is passed to the constructor.
 * For integer types, the sampled values are within the interval [vmin,vmax].
 * For floating point types, the sampled values are within the interval [vmin,vmax).
 *
 * Example:
 * \code
 *    // create uniform random number generator for numbers between [1.0f,2.0f)
 *    UniformRandomGenerator<float> rnd(1.0f,2.0f);
 *
 *    // draw a sample:
 *    float sample = rnd();
 * \endcode
 *
 * @ingroup MathModule
 */
template <typename T>
class UniformRandomGenerator :
	public RandomGenerator<typename detail::UniformRandomDistributionChooser<T>::type>
{
public:
	MIRA_RANDOM_GENERATOR_COMMON(UniformRandomGenerator, typename detail::UniformRandomDistributionChooser<T>::type)
	explicit UniformRandomGenerator(T vmin=T(0), T vmax=T(1)) :
		Base(Distribution(vmin,vmax)) {}

	/**
	 * Resets the limits of the uniform distribution to the specified
	 * limits.
	 */
	void setLimits(T vmin, T vmax) {
		this->distribution() = Distribution(vmin, vmax);
	}
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
