/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file Wizard.h
 *
 * @author Erik Einhorn
 * @date   2011/09/16
 */


#ifndef _MIRA_MAINWIZARD_H_
#define _MIRA_MAINWIZARD_H_

#include <assert.h>
#include <QDir>
#include <QWizard>
#include <QFileDialog>
#include <QMessageBox>

#include <Backend.h>
#include <utils/Path.h>

#include <ui_StartPage.h>
#include <ui_FinishPage.h>
#include <ui_EnvironmentPage.h>
#include <ui_ExternalProjectPage1.h>
#include <ui_ExternalProjectPage2.h>
#include <ui_DomainToolboxPage.h>
#include <ui_ComponentPage.h>
#include <ui_WaitPage.h>

#include <iostream>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class Wizard: public QWizard
{
Q_OBJECT

public:
	enum {
		Page_Start, Page_Finish,
		Page_Environment,
		Page_ExternalProject1,
		Page_ExternalProject2,
		Page_DomainToolbox,
		Page_Component,
		Page_Wait
	};

public:

	Wizard(QWidget *parent = 0);

public:

	int prevId() {
		const QList<int> visited = visitedPages();
		if(visited.size()<2)
			return -1;
		return visited.at(visited.size()-2);
	}

public slots:

	void onCommit()
	{
		switch(prevId())
		{
		case Page_ExternalProject1:
		{
			Path path = resolvePath(field("ProjectPath").toString().toStdString());
			backend().createExternalProject(field("ProjectName").toString().toStdString(),
			                                path);
			break;
		}

		case Page_Environment:
			if(field("SetEnvironmentVariables").toBool()) {
				Path path = resolvePath(field("ProjectPath").toString().toStdString());
				Backend::EnvVars vars = backend().getMissingEnvVars(path);
				if(!backend().setEnvVarsAutomatically(vars))
					QMessageBox::critical(this, tr("Error"),
						tr("Failed to set environment variables"));
				else if ( vars.size() > 0 )
					QMessageBox::information(this, tr("Source environment variables"),
						tr("Environment variables have been set successfully. "
						   "You must source your environment variables file "
						   "or open a new terminal after restart."));
			}
			break;

		case Page_DomainToolbox: {
			setField("WaitText", tr("The wizard creates the necessary files and "
			         "creates the SVN branch. This may take a while, so please be "
			         "patient until this process has finished ..."));

			bool createUnit = field("CreateUnit").toBool();
			setField("WaitNextId", createUnit  ? Page_Component : Page_Finish);

			// give the wait dialog some time to show up
			for(int i=0; i<100; ++i)
				QApplication::processEvents();

			Path domainPath = resolvePath(field("DomainPath").toString().toStdString());
			backend().createDomainToolbox(field("DomainName").toString().toStdString(),
			                              field("DomainDescription").toString().toStdString(),
			                              field("DomainAuthor").toString().toStdString(),
			                              field("DomainTags").toString().toStdString(),
			                              field("DomainIncludeDir").toString().toStdString(),
			                              domainPath,
			                              field("CreateMountDir").toBool(),
			                              field("MountDir").toString().toStdString(),
			                              field("CreateBranch").toBool(),
			                              field("URL").toString().toStdString(),
			                              field("Branch").toString().toStdString(),
			                              field("CreateToolbox").toBool());

			mLastComponentNamespace = field("DomainName").toString().toLower();
			mLastAuthor = field("DomainAuthor").toString();
			mLastComponentSrcPath = QString::fromStdString( (domainPath / "src").string() );

			next();
			break;
		}

		case Page_Component: {
			Backend::ComponentType type = Backend::UNIT;

			if(field("ComponentUnit").toBool())             type = Backend::UNIT;
			else if(field("ComponentMicroUnit").toBool())   type = Backend::MICROUNIT;
			else if(field("ComponentVisualization2D").toBool()) type = Backend::VISUALIZATION2D;
			else if(field("ComponentVisualization3D").toBool()) type = Backend::VISUALIZATION3D;
			else if(field("ComponentViewPart").toBool())    type = Backend::VIEWPART;
			else if(field("ComponentEditorPart").toBool())  type = Backend::EDITORPART;

			Path componentPath = resolvePath(field("ComponentPath").toString().toStdString());

			backend().createComponent(field("ComponentName").toString().toStdString(),
			                          field("ComponentNamespace").toString().toStdString(),
			                          field("ComponentDescription").toString().toStdString(),
			                          field("ComponentAuthor").toString().toStdString(),
			                          componentPath,
			                          type);

			mLastAuthor = field("ComponentAuthor").toString();
			mLastComponentNamespace = field("ComponentNamespace").toString().toLower();
			mLastComponentSrcPath = QString::fromStdString( componentPath.string() );

			break;
		}
		default:
			break;
		}
	}

public:

	Backend& backend() {
		return mBackend;
	}

private:

	Backend mBackend;

public:

	QString mLastAuthor;
	QString mLastComponentNamespace;
	QString mLastComponentSrcPath;
};

// fields:
//     ProjectName
//     ProjectPath

///////////////////////////////////////////////////////////////////////////////

class WizardPage : public QWizardPage
{

public:

	Wizard* wizard() {
		Wizard* w = dynamic_cast<Wizard*>(QWizardPage::wizard());
		assert(w!=NULL);
		return w;
	}

	Backend& backend() {
		return wizard()->backend();
	}
};

///////////////////////////////////////////////////////////////////////////////


class StartPage : public WizardPage, private Ui::StartPage
{
Q_OBJECT
public:
	StartPage() {
		setupUi(this);
		connect(btExternalProject, SIGNAL(clicked(bool)), this, SLOT(buttonPressed()));
		connect(btToolbox, SIGNAL(clicked(bool)), this, SLOT(buttonPressed()));
		connect(btDomain, SIGNAL(clicked(bool)), this, SLOT(buttonPressed()));
		connect(btComponent, SIGNAL(clicked(bool)), this, SLOT(buttonPressed()));
	}

	virtual bool isComplete () const {
		return false; // to disable the "next" button
		// next is done using buttonPressed
	}

	virtual void initializePage() {
		mNextId = Wizard::Page_Finish;
	}

public:

	virtual int nextId () const {
		return mNextId;
	}

private slots:

	void buttonPressed()
	{
		assert(wizard()!=NULL);

		// reset some globally used fields
		setField("ProjectPath",	QDir::toNativeSeparators(QDir::currentPath()) + QDir::separator());
		setField("ComponentPath", QDir::toNativeSeparators(QDir::currentPath()) + QDir::separator());
		setField("CreateUnit",false);

		if(sender()==btExternalProject)
			mNextId = Wizard::Page_ExternalProject1;
		else if(sender()==btToolbox) {
			mNextId = Wizard::Page_DomainToolbox;
			setField("CreateToolbox",true);
		} else if(sender()==btDomain) {
			mNextId = Wizard::Page_DomainToolbox;
			setField("CreateDomain",true);
		} else if(sender()==btComponent) {
			mNextId = Wizard::Page_Component;
		}


		wizard()->next();
	}

private:
	int mNextId;
};

///////////////////////////////////////////////////////////////////////////////

class FinishPage : public WizardPage, private Ui::FinishPage
{
Q_OBJECT
public:
	FinishPage() {
		setupUi(this);
		setFinalPage(true);
		connect(btRestart, SIGNAL(clicked(bool)), this, SLOT(restart()));
	}

	virtual int nextId () const {
		return -1;
	}

private slots:

	void restart() {
		assert(wizard()!=NULL);
		wizard()->restart();
	}

};

///////////////////////////////////////////////////////////////////////////////

class WaitPage : public WizardPage, private Ui::WaitPage
{
public:
	WaitPage() {
		setupUi(this);
		registerField("WaitText", label, "text");
		registerField("WaitNextId", nextIdLabel, "text");
		setCommitPage(true);
	}

	virtual bool isComplete () const {
		return false;
	}

	virtual void initializePage()
	{
		nextIdLabel->setText("");
	}

	virtual int nextId () const {
		if(nextIdLabel->text().isEmpty())
			return -1;
		return nextIdLabel->text().toInt();
	}

};
///////////////////////////////////////////////////////////////////////////////

class EnvironmentPage : public WizardPage, private Ui::EnvironmentPage
{
public:
	EnvironmentPage() {
		setupUi(this);

		registerField("SetEnvironmentVariables", cbSetEnvironmentVariables);
	}

	virtual void initializePage()
	{
		QWizardPage::initializePage();
		Path path = resolvePath(field("ProjectPath").toString().toStdString());
		Backend::EnvVars vars = backend().getMissingEnvVars(path);

		if(!vars.empty()) {
			stackedWidget->setCurrentIndex(0);

			auto p = backend().getEnvVarsString(vars);

			cbSetEnvironmentVariables->setEnabled(p.second);
			teEnvironmentVariables->setPlainText(p.first.c_str());

			setCommitPage(true);

		} else {
			stackedWidget->setCurrentIndex(1);

			setCommitPage(false);
		}
	}

	virtual int nextId () const {
		return Wizard::Page_ExternalProject2;
	}

};

///////////////////////////////////////////////////////////////////////////////

class ExternalProjectPage1 : public WizardPage, private Ui::ExternalProjectPage1
{
Q_OBJECT
public:
	ExternalProjectPage1() {
		setupUi(this);
		setCommitPage(true);
		registerField("ProjectName*", leProjectName);
		registerField("ProjectPath*", lePath);

		QRegExp rx("[a-zA-Z][a-zA-Z0-9_-]*");
		QRegExpValidator* v = new QRegExpValidator(rx, this);
		leProjectName->setValidator(v);

		connect(btChoosePath, SIGNAL(clicked(bool)), this, SLOT(choosePath()));
		connect(leProjectName, SIGNAL(textEdited(const QString&)), this, SLOT(syncPath()));
		connect(lePath, SIGNAL(editingFinished()), this, SLOT(pathEdited()));
	}

	virtual void initializePage()
	{
		leProjectName->setText(""),

		QWizardPage::initializePage();

		parentPath = QDir::toNativeSeparators(QDir::currentPath()) + QDir::separator();
		syncPath();
	}

	virtual int nextId () const {
		return Wizard::Page_Environment;
	}

public slots:

	void choosePath()
	{
		QString p = QFileDialog::getExistingDirectory(this, tr("Choose parent directory"), parentPath);
		if(!p.isEmpty() && !p.isNull()) {
			parentPath = p + QDir::separator();
			syncPath();
		}
	}

	void pathEdited()
	{
		QString p = lePath->text();
		int t = p.lastIndexOf(QDir::separator());
		parentPath = p.mid(0, t+1);
	}

	void syncPath()
	{
		lePath->setText(parentPath+leProjectName->text());
	}

public:
	QString parentPath;
};

///////////////////////////////////////////////////////////////////////////////

class ExternalProjectPage2 : public WizardPage, private Ui::ExternalProjectPage2
{
public:
	ExternalProjectPage2() {
		setupUi(this);

		registerField("CreateToolbox", cbCreateToolbox);
		registerField("CreateDomain", cbCreateDomain);
		registerField("CreateUnit", cbCreateUnit);
	}

	virtual void initializePage()
	{
		cbCreateDomain->setChecked(true);
		cbCreateUnit->setChecked(true);
	}

	virtual int nextId () const {
		if(groupBox->isChecked())
			return Wizard::Page_DomainToolbox;
		else
			return Wizard::Page_Finish;
	}

};

///////////////////////////////////////////////////////////////////////////////

class DomainToolboxPage : public WizardPage, private Ui::DomainToolboxPage
{
Q_OBJECT
public:
	DomainToolboxPage() {
		setupUi(this);
		setCommitPage(true);

		registerField("DomainName*", leName);

		registerField("DomainDescription", leDescription);
		registerField("DomainAuthor", leAuthor);
		registerField("DomainTags", leTags);

		registerField("DomainIncludeDir", leIncludeDir);
		registerField("DomainPath*", lePath);

		registerField("CreateMountDir", cbCreateMountDir, "checked", "toggled(bool)");
		registerField("MountDir", leMountDir);

		registerField("CreateBranch", cbCreateBranch, "checked", "toggled(bool)");
		registerField("URL", cbURL, "currentText", "editTextChanged(const QString& text)");
		registerField("Branch", leBranch);

		QRegExp rx("[a-zA-Z][a-zA-Z0-9_]*");
		QRegExpValidator* v = new QRegExpValidator(rx, this);
		leName->setValidator(v);
		leIncludeDir->setValidator(v);

		connect(btChoosePath, SIGNAL(clicked(bool)), this, SLOT(choosePath()));
		connect(leName, SIGNAL(textEdited(const QString&)), this, SLOT(syncPath()));
		connect(lePath, SIGNAL(editingFinished()), this, SLOT(pathEdited()));
		connect(leBranch, SIGNAL(editingFinished()), this, SLOT(branchEdited()));

		connect(lePath, SIGNAL(textChanged(const QString&)), this, SLOT(syncMountDir()));
	}

	virtual int nextId () const {
		return Wizard::Page_Wait;
	}

	virtual void initializePage()
	{
		QString path = field("ProjectPath").toString() + QDir::separator();
		QString branch;

		if(field("CreateToolbox").toBool()) {
			path += "toolboxes";
			branch = "toolboxes/";
			lbIncludeDir->setEnabled(true);
			leIncludeDir->setEnabled(true);
		} else {
			path += "domains";
			branch = "domains/";
			lbIncludeDir->setEnabled(false);
			leIncludeDir->setEnabled(false);
		}

		path = QDir::toNativeSeparators(QDir::cleanPath(path));
		path += QDir::separator();

		parentPath = path;
		lePath->setText(path);

		cbURL->addItem("https://www.mira-project.org/svn/MIRA-pkg/branches");
		cbURL->addItem("https://www.metralabs-service.com/svn-intern/MetraLabs/branches");
		cbURL->addItem("http://miris.informatik.tu-ilmenau.de/svn/viros/branches");

		parentBranch = branch;
		leBranch->setText(branch);

		if(!wizard()->mLastAuthor.isEmpty())
			leAuthor->setText(wizard()->mLastAuthor);

		syncMountDir();
	}

	virtual bool validatePage()
	{
		if (!WizardPage::validatePage())
			return false;

		Path destPath(lePath->text().toStdString());
		Path projectPath;

		// First: Check, if the selected path in MIRA_PATH
		bool projectFound = false;
		PathVector projectPaths = extractPaths("$MIRA_PATH");
		foreach(const auto& p, projectPaths) {
			Path d = destPath;
			bool match = false;
			while(!d.empty()) {
				if (d == p) {
					projectFound = true;
					projectPath = p;
					break;
				}
				d = d.parent_path();
			}
			if (projectFound)
				break;
		}
		if (!projectFound) {
			// If we're reached here, it is possible, that we have a new
			// created project. Therefore, look for the relevant files.
			Path d = destPath;
			while(!d.empty()) {
				if (boost::filesystem::exists(d / "CMakeLists.txt") &&
					(boost::filesystem::exists(d / "FindMIRARoot.cmake") ||
					 boost::filesystem::exists(d / "mira.root")))
				{
					projectFound = true;
					projectPath = d;
					break;
				}
				d = d.parent_path();
			}
		}
		if (!projectFound) {
			QMessageBox::critical(this, tr("Error"),
			                      tr("The directory %1\nis not part of a MIRA project.").
			                      arg(lePath->text()));
			return(false);
		}

		// "touch" CMakeLists.txt by setting the last write time
		Path cmakePath = projectPath / "CMakeLists.txt";
		if (boost::filesystem::exists(cmakePath))
			boost::filesystem::last_write_time(cmakePath, std::time(0));

		return(true);
	}

public slots:

	void choosePath()
	{
		QString p = QFileDialog::getExistingDirectory(this,
						tr("Choose parent directory"), parentPath);
		if(!p.isEmpty() && !p.isNull()) {
			parentPath = p + QDir::separator();
			syncPath();
		}
	}

	void pathEdited()
	{
		QString p = lePath->text();
		int t = p.lastIndexOf(QDir::separator());
		parentPath = p.mid(0, t+1);
	}

	void branchEdited()
	{
		QString p = leBranch->text();
		int t = p.lastIndexOf('/');
		parentBranch = p.mid(0, t+1);
	}

	void syncPath()
	{
		lePath->setText(parentPath+leName->text());
		leBranch->setText(parentBranch+leName->text());
		if(field("CreateToolbox").toBool())
			leIncludeDir->setText(leName->text().toLower());
	}

	void syncMountDir()
	{
		QString projectPath = field("ProjectPath").toString() + QDir::separator();
		projectPath = QDir::toNativeSeparators(QDir::cleanPath(projectPath));

		QString path = lePath->text();
		if(path.startsWith(projectPath)) {
			path = path.mid(projectPath.length()+1);
			path = QDir::fromNativeSeparators(path);
			leMountDir->setText(path);
		}
	}

public:

	QString parentPath;
	QString parentBranch;
};


///////////////////////////////////////////////////////////////////////////////

class ComponentPage : public WizardPage, private Ui::ComponentPage
{
Q_OBJECT
public:

	ComponentPage() {
		setupUi(this);
		setCommitPage(true);

		registerField("ComponentAuthor*", leAuthor);
		registerField("ComponentNamespace*", leNamespace);
		registerField("ComponentName*", leName);
		registerField("ComponentDescription", leDescription);
		registerField("ComponentPath", lePath);

		registerField("ComponentUnit", rbUnit);
		registerField("ComponentMicroUnit", rbMicroUnit);
		registerField("ComponentVisualization2D", rbVisualization2D);
		registerField("ComponentVisualization3D", rbVisualization3D);
		registerField("ComponentEditorPart", rbEditorPart);
		registerField("ComponentViewPart", rbViewPart);

		QRegExp rx("[a-zA-Z][a-zA-Z0-9_]*");
		QRegExpValidator* v = new QRegExpValidator(rx, this);
		leName->setValidator(v);

		connect(btChoosePath, SIGNAL(clicked(bool)), this, SLOT(choosePath()));
	}

	void initializePage()
	{
		if(!wizard()->mLastComponentNamespace.isEmpty())
			leNamespace->setText(wizard()->mLastComponentNamespace);

		if(!wizard()->mLastComponentSrcPath.isEmpty())
			lePath->setText(wizard()->mLastComponentSrcPath);

		if(!wizard()->mLastAuthor.isEmpty())
			leAuthor->setText(wizard()->mLastAuthor);
	}

	virtual int nextId () const {
		return Wizard::Page_Finish;
	}

public slots:

	void choosePath()
	{
		QString p = QFileDialog::getExistingDirectory(this,
						tr("Choose parent directory"), lePath->text());
		if(!p.isEmpty() && !p.isNull())
			lePath->setText(p + QDir::separator());
	}
};

///////////////////////////////////////////////////////////////////////////////

inline Wizard::Wizard(QWidget* parent) : QWizard(parent)
{
	setPage(Page_Start, new StartPage);
	setPage(Page_Environment, new EnvironmentPage);
	setPage(Page_ExternalProject1, new ExternalProjectPage1);
	setPage(Page_ExternalProject2, new ExternalProjectPage2);
	setPage(Page_DomainToolbox, new DomainToolboxPage);
	setPage(Page_Component, new ComponentPage);
	setPage(Page_Finish, new FinishPage);
	setPage(Page_Wait, new WaitPage);

	/*
	setPixmap(QWizard::BannerPixmap, QPixmap(":/images/banner.png"));
	setPixmap(QWizard::BackgroundPixmap, QPixmap(":/images/background.png"));
	 */
	setWindowTitle(tr("MIRA Wizard"));

	connect(button(CommitButton),SIGNAL(clicked(bool)), this, SLOT(onCommit()));
}

///////////////////////////////////////////////////////////////////////////////

}

#endif
