/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TapePlotRenderer.h
 *    .
 *
 * @author Tim Langner
 * @date   2012/01/05
 */

#ifndef _MIRA_TAPEPLOTRENDERER_H_
#define _MIRA_TAPEPLOTRENDERER_H_

#include <math/Saturate.h>

#include <TypedTapeDataRenderer.h>
#include <TapeEditor.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class TapePlotRenderer : public TypedTapeDataRenderer<T>
{
public:
	typedef TypedTapeDataRenderer<T> Base;

	TapePlotRenderer()
	{
		this->mIndicatorHeight = 5;
		this->mDataHeight = 50;
		this->mDataMin = 0.0;
		this->mDataMax = 1000.0;
		this->mFeatures |= TapeDataRenderer::RENDER;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		Base::reflect(r);
		r.property("DataMin", mDataMin, "Minimum value of data");
		r.property("DataMax", mDataMax, "Maximum value of data");
		r.property("Member", mMember, "Member to render");
	}

	virtual int getDataWidth() const { return 5; }

	virtual QSize renderMessage(QPainter* painter, const QRect& maxRect,
	                            TapeChannelInfo& info,
	                            TapeChannelInfo::DataMap::iterator& message);

protected:

	QSize renderPlot(QPainter* painter, const QRect& maxRect,
	                 TapeChannelInfo& info,
	                 double message, double nextMessage, int64 timeDiff);

	double mDataMin;
	double mDataMax;
	std::string mMember;
};

template<typename T>
QSize TapePlotRenderer<T>::renderPlot(QPainter* painter, const QRect& maxRect,
                                      TapeChannelInfo& info,
                                      double message, double nextMessage, int64 timeDiff)
{
	int diffX = timeDiff / this->mEditor->getResolution();
	int nextX = maxRect.x() + diffX;
	if (nextX > maxRect.right())
		return QSize(0, maxRect.height());
	double factor = (double)(maxRect.height()-1) / (mDataMax-mDataMin);
	int y1 = (maxRect.height() -1) - (int)((saturate(message, mDataMin, mDataMax)-mDataMin) * factor);
	int y2 = (maxRect.height() -1) - (int)((saturate(nextMessage, mDataMin, mDataMax)-mDataMin) * factor);
	painter->setPen(QPen(info.color.darker()));
	painter->drawLine(maxRect.x(), maxRect.y()+y1, nextX, maxRect.y()+y2);
	return QSize(diffX, maxRect.height());
}

template<typename T>
QSize TapePlotRenderer<T>::renderMessage(QPainter* painter, const QRect& maxRect,
                                         TapeChannelInfo& info,
                                         TapeChannelInfo::DataMap::iterator& message)
{
	try
	{
		T data;
		BinaryBufferDeserializer bs(const_cast<Buffer<uint8>*>(&message->second.getData()));
		bs.deserialize(data, false);
		TapeChannelInfo::DataMap::iterator next = message;
		++next;
		if (next == info.data.end())
			return QSize(0, maxRect.height());
		T nextData;
		BinaryBufferDeserializer bsNext(const_cast<Buffer<uint8>*>(&next->second.getData()));
		bsNext.deserialize(nextData, false);
		JSONSerializer js;
		json::Value json = js.serialize(data);
		double value = json::getNumberElement(json, mMember);
		JSONSerializer jsNext;
		json = jsNext.serialize(nextData);
		double nextValue = json::getNumberElement(json, mMember);
		int64 timeDiff = next->first-message->first;
		return renderPlot(painter, maxRect, info, value, nextValue, timeDiff);
	}catch(...)
	{
	}
	return QSize(0, maxRect.height());
}

//////////////////////////////////////////////////////////////////////////////


#define MIRA_REGISTER_PLOTRENDERER(name, type)                \
class name : public TapePlotRenderer<type> {                 \
	MIRA_META_OBJECT(name,                                    \
	                ("Name", "Plot")                         \
	                ("Description", "Renders data as plot")) \
};

///////////////////////////////////////////////////////////////////////////////

} // namespace

#endif
