/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FTPRepository.C
 *    Description.
 *
 * @author Ronny Stricker, Christian Martin
 * @date   2011/09/02
 */
#include "core/FTPRepository.h"

#include <QApplication>

#include <serialization/Serialization.h>

#include "core/Tools.h"
#include "core/Package.h"
#include "core/PackageParser.h"

using namespace std;

namespace mira {

///////////////////////////////////////////////////////////////////////////////

FTPRepository::FTPRepository( QObject* parent) : FileRepository()
{
	protocol = "ftp";
	connect( &mFtp, SIGNAL(listInfo( QUrlInfo const& )), this, SLOT( dirEntry( QUrlInfo const& ) ));
//	connect( &mFtp, SIGNAL(stateChanged( int )), this, SLOT( stateChangedInfo( int ) ));
	connect( &mFtp, SIGNAL(commandFinished( int, bool )), this, SLOT( finished( int, bool ) ));

	name = getClass().getMetaInfo("RepoType");
	url += name;
	mForceLogin = false;
}

void FTPRepository::setUrl( Url const& iUrl )
{
	Url tUrl = url;
	try {
		Repository::setUrl( iUrl );
	}
	catch( Exception &ex ) {
		url = tUrl;
		MIRA_RETHROW( ex, "");
	}

	string protocol = url.substr( 0, url.find(":") );

	if ( protocol != "ftp" ) {
		MIRA_THROW( XLogical, "Dont know how to handle protocol " << protocol );
	}

	// extract the server and subpath string from the url
	server = removePrefix( url );
	subUrl = "";
	if ( server.find('/') != string::npos ) {
		subUrl = server.substr( server.find('/')+1 );
		server = server.substr( 0, server.find('/') );
	}
}

void FTPRepository::deepExamine( std::vector<Package*>& oPackages,
		boost::function<void (std::string const&,int,int)> progressListener )
{
	if ( login() ) {
		mFileList.clear();

		// connect to host
		browseDir( subUrl, "" );

		// count the interesting files
		int totalfiles = 0;
		if ( progressListener ) {
			foreach(Url const& u, mFileList) {
				if ( u.empty() )
					continue;
				Path p( u );
				if(p.extension()==".package" || p.extension()==".extpackage"
						|| p.filename()=="mountdir.xml") {
					++totalfiles;
				}
			}
		}

		int progress = 0;
		// look for *.(ext)package files
		foreach(Url const& u, mFileList) {
			if(u.empty())
				continue;
			Path p( u );
			if(p.extension()==".package" || p.extension()==".extpackage"
					|| p.filename()=="mountdir.xml") {
				if (p.filename()=="mountdir.xml") {
					string fileContent = catFile( u );
					parseMountDirXml( url / u, fileContent );
				}
				else {
					Package* tPackage = indexWebPackage(u, this);
					if ( tPackage )
						oPackages.push_back( tPackage );
				}
				++progress;
				if ( progressListener )
					progressListener(u, progress, int(mFileList.size()) );
			}
		}
	}
}

void FTPRepository::install( Package const& package, Path const& destPath )
{
	if ( login() ) {
		FileRepository::install(package, destPath);
	} else {
		MIRA_THROW( XIO, "Cannot login to repository " + name );
	}
}

vector<Url> FTPRepository::getZipFileList( Url const& subdir )
{
	vector<Url> tReturn;
	mFileList.clear();
	browseDir( subUrl / subdir, "", false );
	foreach( Url const& tFile, mFileList ) {
		if ( tFile.rfind(".zip") == tFile.size()-4 ) {
			tReturn.push_back( tFile );
		}
	}
	return tReturn;
}

bool FTPRepository::login()
{
//	// we have to remove the tailing '/' which was added in setUrl
//	// this is necessary since the QFtp class fails to connect otherwise
//	string url = baseUrl.size() > 0 ? baseUrl.substr(0,baseUrl.size()-1) : "";

	if ( mFtp.state() != QFtp::LoggedIn ) {
		if ( mFtp.state() != QFtp::Connected ) {
			mFtp.connectToHost( QString::fromStdString( server ) );
			forceExecution();
			if ( mFtp.state() != QFtp::Connected ) {
				prompt->showErrorMessage( string("cannot connect to ") + server );
				return false;
			}
		}

		// try to login (first attempt without empty username and password)
		credential.realm = url;

		if ( mForceLogin ) {
			prompt->getLogin( credential );
		}

		// repeat as long as we are not connected and the user tries
		// to enter new credential
		do {
			mFtp.login( QString::fromStdString( credential.user ),
				QString::fromStdString( credential.password ) );
			forceExecution();
		
		} while ( mFtp.state() != QFtp::LoggedIn && prompt->getLogin( credential ) );

		if ( mFtp.state() != QFtp::LoggedIn ) {
			prompt->showErrorMessage( string("failed to login to ") + server );
			return false;
		}
	}
	return true;
}

void FTPRepository::browseDir( Url const& subdir, Url const& currentDir, bool recursive )
{
	Url tCurrentDir = currentDir / subdir;

	mCurrentDirs.clear();
	mCurrentDir = tCurrentDir;
	MIRA_LOG(NOTICE) << "change directory: " << subdir;
	mFtp.cd( QString::fromStdString( subdir ) );
	forceExecution();

	MIRA_LOG(NOTICE) << "list directory: " << subdir;
	mFtp.list();
	forceExecution();

	if ( recursive ) {
		vector< Url > tDirs = mCurrentDirs;
		foreach( Url const& tDir, tDirs ) {
			browseDir( tDir, tCurrentDir );
		}
	}

	// step up as many directories as we have stepped down
	QString leaveCommand = "../";
	leaveCommand += leaveCommand.repeated(QString::fromStdString( subdir ).count('/'));
	MIRA_LOG(NOTICE) << "change directory: " << leaveCommand.toStdString();
	mFtp.cd( leaveCommand );
}

void FTPRepository::forceExecution()
{
	while ( mFtp.currentCommand() != QFtp::None || mFtp.hasPendingCommands() ) {
		QApplication::processEvents();
	}
}

void FTPRepository::dirEntry( QUrlInfo const& info )
{
	if ( info.name() != "." && info.name() != ".." ) {
		if ( info.isDir() )
			mCurrentDirs.push_back( info.name().toStdString() );
		else {
			string path = mCurrentDir / info.name().toStdString();
			// find the subUrl part even if the url starts with a slash
			size_t subUrlPos = path.find( subUrl );
			if ( subUrlPos != string::npos ) {
				// remove the subUrl part of the path
				path = path.substr( subUrlPos+subUrl.size()+1, string::npos );

				mFileList.push_back( path );
			}
			else
				MIRA_LOG(ERROR) << "  error(ftp): cannot find sub url " << subUrl
					<< "in current url " << path;
		}
	}
}

void FTPRepository::finished( int id, bool error )
{
	if ( mFtp.error() ) {
		MIRA_LOG(NOTICE) << "  error(ftp): " << mFtp.errorString().toStdString();
	}
}

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::FTPRepository, mira::Repository);
