/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file PackageTreeModel.h
 *    $Provide model used to display packages$.
 *
 * @author Ronny Stricker
 * @date   2011/07/26
 */

#ifndef PACKAGETREEMODEL_H_
#define PACKAGETREEMODEL_H_

#include <QAbstractItemModel>
#include <QPersistentModelIndex>
#include <QModelIndex>
#include <QVariant>
#include <QIcon>
#include <QItemDelegate>
#include <QComboBox>

#include "core/Database.h"

namespace mira {

//////////////////////////////////////////////////////////////////////////////

/**
 * @brief Editor to present model data as ComboBox.
 * Class provides functions to get and set the model data.
 */
class ComboBoxDelegate : public QItemDelegate
{
	Q_OBJECT

public:
	ComboBoxDelegate(QObject *parent = 0)  : QItemDelegate(parent) {}

	/**
	 * @brief Create QComboBox
	 */
	QWidget *createEditor(QWidget *parent, const QStyleOptionViewItem &option,
		                  const QModelIndex &index) const
	{
		QComboBox* editor = new QComboBox(parent);
		return editor;
	}

	/**
	 * @brief Read data from model and store the data in the comboBox
	 */
	void setEditorData(QWidget *editor, const QModelIndex &index) const
	{
		QComboBox* comboBox = static_cast<QComboBox*>(editor);

		QStringList list = index.model()->data(index, Qt::EditRole).toStringList();
		for (int i=0; i<list.size();++i) {
			comboBox->addItem( list.at(i) );
		}
	}

	/**
	 * @brief Read data from the comboBox and write it back to the model.
	 */
	void setModelData(QWidget *editor, QAbstractItemModel *model,
		               const QModelIndex &index) const
	{
		QComboBox *comboBox = static_cast<QComboBox*>(editor);
		QVariant value = comboBox->currentText();

		model->setData(index, value, Qt::EditRole);
	}

	void updateEditorGeometry(QWidget *editor,
		const QStyleOptionViewItem &option, const QModelIndex &index) const
	{
		editor->setGeometry(option.rect);
	}

};

/**
 * @brief The class wraps the data represented by a packageGroup tree to
 * enable usage of Qt model/view structure.
 */
class PackageTreeModel : public QAbstractItemModel
{
	Q_OBJECT

public:
	/**
	 * @brief Flags used to modify the behaviour of the model.
	 */
	enum Flags {
		NOFLAGS  = 0x1,
		EDITABLE = 0x2	///< active repository should be editable
	};

	MIRA_ENUM_TO_FLAGS_INCLASS( Flags )

	/**
	 * @brief Constructor.
	 * The model does not take ownership of the data. Therefore, the given data
	 * have to be valid as long as the class exist.
	 * The given Flags change the behavior of the class.
	 * @param[in] rootPackage data used by the model.
	 * @param[in] database the database associated with the root package
	 * @param[in] flags Flags for the model
	 */
	PackageTreeModel( PackageGroup const* rootPackage, Database* database,
			Flags flags = NOFLAGS, QObject *parent = 0);

	virtual ~PackageTreeModel();

	/**
	 * @brief provide data associated with the given model index.
	 */
	QVariant data(QModelIndex const& index, int role) const;

	/**
	 * @brief try to set the data of the given model index.
	 */
	bool setData(QModelIndex const& index, const QVariant &value, int role);

	/**
	 * @brief return Qt::ItemFlags for the given model index.
	 * see Qt Documentation for more information about this topic.
	 */
	Qt::ItemFlags flags(const QModelIndex &index) const;

	/**
	 * @brief provide header data for the different columns of the model.
	 * Note: number of columns depend on model flags.
	 */
	QVariant headerData(int section, Qt::Orientation orientation,
	                    int role = Qt::DisplayRole) const;

	/**
	 * @brief provide index for the given row and column.
	 * see Qt Documentation for more information about this topic.
	 */
	QModelIndex index(int row, int column,
	                  const QModelIndex &parent = QModelIndex()) const;

	/**
	 * @brief return the parent for the given model index.
	 * see Qt Documentation for more information about this topic.
	 */
	QModelIndex parent(const QModelIndex &index) const;

	/**
	 * @brief return number of rows provided by the model.
	 * see Qt Documentation for more information about this topic.
	 */
	int rowCount(const QModelIndex &parent = QModelIndex()) const;

	/**
	 * @brief return number of columns provided by the model.
	 * Note: number of columns depend on model flags.
	 */
	int columnCount(const QModelIndex &parent = QModelIndex()) const;

	/**
	 * @brief provide protected beginResetModel function.
	 * This isn't very British, however, it comes in handy sometimes.
	 */
	void beginReset() {
		beginResetModel();
	}

	/**
	 * @brief provide protected endResetModel function.
	 * This isn't very British, however, it comes in handy sometimes.
	 */
	void endReset() {
		endResetModel();
	}

signals:
	void installChangeRequested( QModelIndex const& index );

protected slots:
	void modelDataChanged( PackageGroup* group );

public:
	PackageGroup const* mRoot;	///< the data
	Database* mDatabase;		///< database associated with the data
	Flags mFlags;				///< active flags

	mutable std::map<PackageGroup*, std::map<int,QPersistentModelIndex> > mTrackedIndexes;
 };

//////////////////////////////////////////////////////////////////////////////

}

#endif /* PACKAGETREEMODEL_H_ */
