/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file FTPRepository.h
 *
 * @author Ronny Stricker
 * @date   2011/09/02
 */

#include <QFtp>

#include <utils/Path.h>
#include <utils/EnumToFlags.h>

#include <core/FileRepository.h>

#ifndef _MIRA_FTPREPOSITORY_H_
#define _MIRA_FTPREPOSITORY_H_

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * @brief The class enables the use of repositories via FTP for binary- and
 * extrapackages.
 */
class FTPRepository : public FileRepository {

	MIRA_META_OBJECT(FTPRepository,("RepoType","FTP"))
	Q_OBJECT

public:
	/**
	 * @brief Constructor.
	 * Setup the repository and create connections to Qt signals.
	 */
	FTPRepository( QObject* parent = NULL );

	virtual ~FTPRepository() {}

	virtual void setUrl( Url const& iUrl );

	/**
	 * @brief Ftp specific implementation of Repository::deepExamine().
	 */
	virtual void deepExamine( std::vector<Package*>& oPackages,
			boost::function<void (std::string const&,int,int)> progressListener = NULL );

	/**
	 * @brief Ftp specific implementation of FileRepository::install().
	 */
	virtual void install( Package const& package, mira::Path const& destPath );

	/// The reflect method.
	template <class Reflector>
	void reflect ( Reflector& r ) {
		FileRepository::reflect( r );
		r.member("forceLogin", mForceLogin,
				 "Force login to bypass anonymous logins.", false);
	}

public:

	bool mForceLogin;	///< User is forced to use login (to bypass anonymous login)

protected:
	/**
	 * @brief Browses the given dir for zip files and returns the list of files.
	 * @param subdir
	 * @return
	 */
	virtual std::vector<Url> getZipFileList( Url const& subdir );

protected:
	/**
	 * @brief Try to login to repository (if not already connected)
	 */
	bool login();

	/**
	 * @brief add files and directories of given directory to file and directory
	 * list and recursively call browseDir() for subdirectories.
	 */
	void browseDir( Url const& subdir, Url const& currentDir, bool recursive = true );

	/**
	 * @brief Since QFtp works asynchronous, this function forces the execution
	 * of pending commands, to provide a synchronous API
	 */
	void forceExecution();

protected slots:
	/**
	 * @brief add received file or directory info to member lists.
	 */
	void dirEntry( QUrlInfo const& info );

	/**
	 * @brief Internal function used for debugging purpose.
	 */
	void finished( int id, bool error );

protected:
	QFtp mFtp;	///< QFtp Object used to connect and communicate with the ftp server
	Credential mCredential;	///< credentials

	Url server;	// server part of the url
	Url subUrl;	// suburl part of the url

	Url mCurrentDir;	///< current directory used for browseDir()

	std::vector< Url > mFileList;		///< filelist obtained during browseDir()
	std::vector< Url > mCurrentDirs;	///< directory list obtained during browseDir()


};

///////////////////////////////////////////////////////////////////////////////

}

#endif /* _MIRA_FTPREPOSITORY_H_ */
