/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file RPCView.C
 *    Description.
 *
 * @author tim
 * @date   15.05.2011
 */

#include <QVBoxLayout>
#include <QHBoxLayout>
#include <QSplitter>
#include <QMenu>
#include <QCompleter>
#include <QTabWidget>
#include <QMessageBox>

#include <views/RPCView.h>
#include <serialization/Serialization.h>
#include <rpc/RPCManager.h>


///////////////////////////////////////////////////////////////////////////////

MIRA_CLASS_SERIALIZATION(mira::RPCView, mira::ViewPart);

///////////////////////////////////////////////////////////////////////////////

namespace mira {


RPCView::RPCView() :
	showHidden(false),
	ui(NULL)
{
}

QWidget* RPCView::createPartControl()
{
	ui = new UI(this);
	QMenu* menu = getViewMenu();
	QAction* action = menu->addAction("Show hidden services");
	action->setCheckable(true);
	action->setChecked(showHidden);
	connect(action, SIGNAL(toggled(bool)), this, SLOT(onShowHidden(bool)));

	return ui;
}

void RPCView::onShowHidden(bool show)
{
	showHidden = show;
	if (ui != NULL)
		ui->updateServices();
}

///////////////////////////////////////////////////////////////////////////////

RPCView::UI::UI(RPCView* parent) :
	QWidget(parent),
	rpcView(parent)
{
	QVBoxLayout *mainLayout = new QVBoxLayout(this);
	mainLayout->setContentsMargins(0, 4, 0, 0);

	QTabWidget* tab = new QTabWidget(this);
	mainLayout->addWidget(tab);

	//////////////////////////////////////////////////////////////////////////////
	// main tab

	{
		QWidget* mainTab = new QWidget();
		QVBoxLayout* mainLayout = new QVBoxLayout(mainTab);
		mainLayout->setContentsMargins(0, 4, 0, 0);
		QSplitter* splitterV = new QSplitter(mainTab);
		splitterV->setOrientation(Qt::Vertical);
		QSplitter* splitterH = new QSplitter(splitterV);
		splitterH->setOrientation(Qt::Horizontal);

		serviceList = new QTreeWidget(splitterH);
		serviceList->setMinimumHeight(20);
		serviceList->setColumnCount(2);
		QStringList l;
		l << "Service" << "Signature";
		serviceList->setHeaderLabels(l);
		serviceList->setColumnWidth(0, 200);
		serviceList->setContextMenuPolicy(Qt::CustomContextMenu);

		splitterH->addWidget(serviceList);
		QWidget* rightPane = new QWidget(splitterH);
		QVBoxLayout* rightLayoutV = new QVBoxLayout(rightPane);
		rightLayoutV->setContentsMargins(0, 0, 0, 0);
		rightLayoutV->setSpacing(0);

		QHBoxLayout* rightLayoutH = new QHBoxLayout();
		rightLayoutH->setContentsMargins(0, 0, 0, 0);
		rightLayoutH->setSpacing(0);

		rightLayoutV->addLayout(rightLayoutH);

		serviceName = new QLineEdit(rightPane);
		serviceName->setText(rpcView->method.c_str());
		rightLayoutH->addWidget(serviceName);

		callBtn = new QPushButton("call", rightPane);
		rightLayoutH->addWidget(callBtn);

		completer = new QCompleter(rpcView->history, mainTab);
		serviceParams = new TextEditAutoCompletion(rightPane);
		serviceParams->setMinimumHeight(20);
		serviceParams->setText(rpcView->params.c_str());
		serviceParams->setCompleter(completer);
		rightLayoutV->addWidget(serviceParams);

		splitterH->addWidget(rightPane);
		splitterV->addWidget(splitterH);

		answer = new QTextEdit(splitterV);
		answer->setMinimumHeight(20);
		splitterV->addWidget(answer);
		mainLayout->addWidget(splitterV);

		connect(callBtn, SIGNAL(clicked()), this, SLOT(onCall()));
		connect(serviceList, SIGNAL(itemDoubleClicked(QTreeWidgetItem*, int)),
				this, SLOT(onMethodClicked(QTreeWidgetItem*, int)));
		connect(serviceList, SIGNAL(customContextMenuRequested(const QPoint&)), this, 
					SLOT(onShowContextMenu(const QPoint&)));

		tab->addTab(mainTab, "Services");
	}

	//////////////////////////////////////////////////////////////////////////////
	// customized tab

	{
		QWidget* customizedTab = new QWidget();
		QVBoxLayout* mainLayout = new QVBoxLayout(customizedTab);
		mainLayout->setContentsMargins(0, 4, 0, 0);
		QSplitter* splitterV = new QSplitter(customizedTab);
		splitterV->setOrientation(Qt::Vertical);
		customServices = new QTableWidget(splitterV);
		QStringList headers;
		headers << "Method" << "Params" <<  "Call" << "Remove";
		customServices->setMinimumHeight(20);
		customServices->setColumnCount(4);
		customServices->setHorizontalHeaderLabels(headers);
		foreach(const auto& entry, rpcView->customEntries)
			addCustomMethod(entry.first, entry.second);
		splitterV->addWidget(customServices);
		customAnswer = new QTextEdit(splitterV);
		customAnswer->setMinimumHeight(20);
		splitterV->addWidget(customAnswer);
		mainLayout->addWidget(splitterV);

		tab->addTab(customizedTab, "Customized");
	}

	//////////////////////////////////////////////////////////////////////////////
	
	startTimer(5000);
	updateServices();
}

void RPCView::UI::timerEvent(QTimerEvent *e)
{
	updateServices();
}

void RPCView::UI::updateServices()
{
	std::set<std::string> localServices = MIRA_FW.getRPCManager().getLocalServices();
	std::set<std::string> remoteServices = MIRA_FW.getRPCManager().getRemoteServices();
	
	// first remove all services that are not longer known or that do not match the
	// settings and update remaining ones
	for(int i=0; i<serviceList->topLevelItemCount();)
	{
		std::string name = serviceList->topLevelItem(i)->text(0).toStdString();
		if (localServices.count(name) > 0)
		{
			// remove service from new service list
			localServices.erase(name);
			auto s = MIRA_FW.getRPCManager().getLocalService(name);
			if (!rpcView->showHidden &&
				name.find(MIRA_HIDDEN_AUTHORITY_PREFIX) != std::string::npos)
			{
				serviceList->takeTopLevelItem(i);
				continue;
			}
			// check if all methods are already in the method list of the service and
			// add new ones
			QTreeWidgetItem* item = serviceList->topLevelItem(i);
			foreach(const auto& m, s.methods)
			{
				bool alreadyExists = false;
				for(int j=0; j<item->childCount(); ++j)
					if (item->child(j)->text(0) == QString(m.signature.name.c_str()))
					{
						alreadyExists = true;
						break;
					}
				if (alreadyExists)
					continue;
				std::stringstream ss;
				ss << m.signature;
				QTreeWidgetItem* subItem = new QTreeWidgetItem(item);
				subItem->setText(0, m.signature.name.c_str());
				subItem->setText(1, ss.str().c_str());
				item->addChild(subItem);
			}
		}
		else if (remoteServices.count(name) > 0)
		{
			// remove service from new service list
			remoteServices.erase(name);
			auto s = MIRA_FW.getRPCManager().getRemoteService(name);
			if (!rpcView->showHidden &&
				name.find(MIRA_HIDDEN_AUTHORITY_PREFIX) != std::string::npos)
			{
				serviceList->takeTopLevelItem(i);
				continue;
			}
			// check if all methods are already in the method list of the service and
			// add new ones
			QTreeWidgetItem* item = serviceList->topLevelItem(i);
			foreach(const auto& m, s.methods)
			{
				bool alreadyExists = false;
				for(int j=0; j<item->childCount(); ++j)
					if (item->child(j)->text(0) == QString(m.signature.name.c_str()))
					{
						alreadyExists = true;
						break;
					}
				if (alreadyExists)
					continue;
				std::stringstream ss;
				ss << m.signature;
				QTreeWidgetItem* subItem = new QTreeWidgetItem(item);
				subItem->setText(0, m.signature.name.c_str());
				subItem->setText(1, ss.str().c_str());
				item->addChild(subItem);
			}
		}
		else
		{
			serviceList->takeTopLevelItem(i);
			continue;
		}

		++i;
	}

	// now add all remaining new services
	foreach(const std::string& service, localServices)
	{
		auto s = MIRA_FW.getRPCManager().getLocalService(service);
		if (!rpcView->showHidden &&
			service.find(MIRA_HIDDEN_AUTHORITY_PREFIX) != std::string::npos)
			continue;
		QTreeWidgetItem* item = new QTreeWidgetItem(serviceList);
		item->setText(0, service.c_str());
		serviceList->addTopLevelItem(item);
		foreach(const auto& m, s.methods)
		{
			std::stringstream ss;
			ss << m.signature;
			QTreeWidgetItem* subItem = new QTreeWidgetItem(item);
			subItem->setText(0, m.signature.name.c_str());
			subItem->setText(1, ss.str().c_str());
			item->addChild(subItem);
		}
	}
	foreach(const std::string& service, remoteServices)
	{
		auto s = MIRA_FW.getRPCManager().getRemoteService(service);
		if (!rpcView->showHidden &&
			service.find(MIRA_HIDDEN_AUTHORITY_PREFIX) != std::string::npos)
			continue;
		QTreeWidgetItem* item = new QTreeWidgetItem(serviceList);
		item->setText(0, service.c_str());
		serviceList->addTopLevelItem(item);
		foreach(const auto& m, s.methods)
		{
			std::stringstream ss;
			ss << m.signature;
			QTreeWidgetItem* subItem = new QTreeWidgetItem(item);
			subItem->setText(0, m.signature.name.c_str());
			subItem->setText(1, ss.str().c_str());
			item->addChild(subItem);
		}
	}
}

void RPCView::UI::onCall()
{
	answer->setText("");

	QString method = serviceName->text().trimmed();

	// Ensure, that method is not empty. Otherwise boost::split (see below)
	// will return an empty vector and this will cause crash.
	if (method.size() == 0)
		return;

	rpcView->method = method.toStdString();
	rpcView->params = serviceParams->toPlainText().toStdString();
	std::vector<std::string> v;
	boost::split(v, rpcView->method, boost::is_from_range('.','.'));
	auto future = MIRA_FW.getRPCManager().callJSON(*v.begin(), *v.rbegin(),
	                                               rpcView->params);

	if(future.timedWait(Duration::seconds(5))) {
		std::string a = json::write(future.get(),true);
		answer->setText(a.c_str());
	} else {
		MIRA_THROW(XRuntime, "Timeout while invoking RPC");
	}

	rpcView->history << serviceParams->toPlainText();
	serviceParams->setCompleter(NULL);
	delete completer;
	completer = new QCompleter(rpcView->history, this);
	serviceParams->setCompleter(completer);
}

void RPCView::UI::onCallCustomMethod()
{
	int row = findButton(2);
	if (row < 0)
		return;

	std::string service = customServices->item(row, 0)->text().toStdString();
	std::string params = customServices->item(row, 1)->text().toStdString();
	rpcView->customEntries[service] = params;
	std::vector<std::string> v;
	boost::split(v, service, boost::is_from_range('.','.'));
	auto future = MIRA_FW.getRPCManager().callJSON(*v.begin(), *v.rbegin(), params);
	if(future.timedWait(Duration::seconds(5))) {
		std::string a = json::write(future.get(),true);
		customAnswer->setText(a.c_str());
	} else {
		MIRA_THROW(XRuntime, "Timeout while invoking RPC");
	}
}

void RPCView::UI::onRemoveCustomMethod()
{
	int row = findButton(3);
	if (row < 0)
		return;
	std::string service = customServices->item(row, 0)->text().toStdString();
	rpcView->customEntries.erase(service);
	customServices->removeRow(row);
}

int RPCView::UI::findButton(int column)
{
	QPushButton* btn = (QPushButton*)sender();
	for(int r=0; r<customServices->rowCount(); ++r)
		if(btn == customServices->cellWidget(r, column))
			return r;
	return -1;
}

void RPCView::UI::addCustomMethod(const std::string& service, const std::string& params)
{
	int row = customServices->rowCount();
	customServices->insertRow(customServices->rowCount());
	QTableWidgetItem* entry = new QTableWidgetItem(service.c_str(), 1);
	customServices->setItem(row, 0, entry);
	entry = new QTableWidgetItem(params.c_str(), 1);
	customServices->setItem(row, 1, entry);

	QPushButton* callBtn = new QPushButton("call", this);
	QPushButton* remBtn = new QPushButton(this);
	remBtn->setIcon(QIcon(":/icons/Minus.png"));
	connect(callBtn, SIGNAL(clicked()), this, SLOT(onCallCustomMethod()));
	connect(remBtn, SIGNAL(clicked()), this, SLOT(onRemoveCustomMethod()));
	customServices->setCellWidget(row, 2, callBtn);
	customServices->setCellWidget(row, 3, remBtn);
}

void RPCView::UI::addCustomMethod(QTreeWidgetItem* item)
{
	std::string service = QString(item->parent()->text(0) + QString(".") + item->text(0)).toStdString();
	if (rpcView->customEntries.count(service) > 0)
	{
		QMessageBox::warning(this, "Duplicate entry", "An entry for this method does already exist");
		return;
	}
	rpcView->customEntries[service] = "";
	addCustomMethod(service, "");
}

void RPCView::UI::onShowContextMenu(const QPoint& p)
{
	QTreeWidgetItem* item = serviceList->itemAt(p);
	// if no item clicked or service instead of method selected do not show menu
	if (item == NULL || item->parent() == NULL)
		return;
	QPoint globalPos = serviceList->viewport()->mapToGlobal(p);
	QMenu myMenu;
	QAction* addAction = myMenu.addAction("Add new custom entry for this method");
	if (addAction == myMenu.exec(globalPos))
		addCustomMethod(item);
}

void RPCView::UI::onMethodClicked(QTreeWidgetItem* item, int c)
{
	if (item->parent() != NULL)
	{
		serviceName->setText(item->parent()->text(0) + QString(".") + item->text(0));
		serviceParams->setText("");
		answer->setText("");
		rpcView->method = serviceName->text().toStdString();
		rpcView->params = "";
	}
}


///////////////////////////////////////////////////////////////////////////////

} // namespace

