/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DriveView.C
 *    Implementation of DriveView.h.
 *
 * @author Tim Langner
 * @date   2011/04/26
 */

#include <gui/views/DriveView.h>

#include <serialization/PropertySerializer.h>
#include <transform/Velocity.h>

#include <QVBoxLayout>
#include <QMenu>

namespace mira { namespace robot {

///////////////////////////////////////////////////////////////////////////////

DriveView::DriveView() :
	velocityIncrement(0.1f),
	watchdogTimeout(Duration::seconds(5)),
	driveService("IDrive"),
	mControl(NULL),
	ui(NULL)
{
}


DriveView::~DriveView()
{
	delete mControl;
}

QWidget* DriveView::createPartControl()
{
	ui = new UI(this);
	return ui;
}

Object* DriveView::getAdapter(const Class& adapter)
{
	if(adapter == PropertyViewPage::CLASS())
	{
		if(mControl==NULL)
		{
			PropertySerializer s;
			DriveView* This = this;
			PropertyNode* p = s.reflectProperties(getClass().getName(), This);
			mControl = new PropertyViewPage(boost::shared_ptr<PropertyNode>(p));
			mControl->setWindowTitle(this->windowTitle());
		}
		return mControl;
	}

	return ViewPart::getAdapter(adapter);
}

///////////////////////////////////////////////////////////////////////////////

DriveView::UI::UI(DriveView* parent) :
	QWidget(parent),
	authority("/", "DriveView", Authority::ANONYMOUS | Authority::INTERNAL),
	mParent(parent)
{
	Ui::DriveViewWidget::setupUi(this);
	setFocusPolicy(Qt::StrongFocus);
	timer = new QTimer(this);
	timer->setInterval(100);
	timer->setSingleShot(false);
	serviceTimer = new QTimer(this);
	serviceTimer->setInterval(2000);
	serviceTimer->setSingleShot(false);

	forward->setIcon(QIcon(":/icons/ArrowUp.png"));
	backward->setIcon(QIcon(":/icons/ArrowDown.png"));
	left->setIcon(QIcon(":/icons/ArrowLeft.png"));
	right->setIcon(QIcon(":/icons/ArrowRight.png"));
	stop->setIcon(QIcon(":/icons/Cancel.png"));

	connect(forward, SIGNAL(clicked()), this, SLOT(onForward()));
	connect(backward, SIGNAL(clicked()), this, SLOT(onBackward()));
	connect(left, SIGNAL(clicked()), this, SLOT(onLeft()));
	connect(right, SIGNAL(clicked()), this, SLOT(onRight()));
	connect(stop, SIGNAL(clicked()), this, SLOT(onStop()));
	connect(bumper, SIGNAL(clicked()), this, SLOT(onResetBumper()));
	connect(odometry, SIGNAL(clicked()), this, SLOT(onResetOdometry()));
	connect(mBtEnable, SIGNAL(clicked()), this, SLOT(onEnableMotors()));
	connect(mBtDisable, SIGNAL(clicked()), this, SLOT(onDisableMotors()));
	connect(timer, SIGNAL(timeout()), this, SLOT(tick()));
	connect(serviceTimer, SIGNAL(timeout()), this, SLOT(updateService()));
	connect(mBtGrabKeyboard, SIGNAL(toggled(bool)), this, SLOT(onGrabKeyboard(bool)));

	updateService();
	serviceTimer->start();
	transSpeed = 0;
	rotSpeed = 0;
}

void DriveView::UI::tick()
{
	try {
		if ((Time::now() - lastCommand) > mParent->watchdogTimeout)
		{
			onStop();
			return;
		}
		authority.callService<void>(service, "setVelocity",
		                            Velocity2(transSpeed, 0.0f, rotSpeed));
	} catch(...) {
		timer->stop();
	}
}

void DriveView::UI::updateService()
{
	bool serviceChanged = false;
	// parent gives us a service so use it
	if (!mParent->driveService.empty())
	{
		if (mParent->driveService != service)
		{
			service = mParent->driveService;
			serviceChanged = true;
		}
	}

	if (service.empty())
	{
		auto l = authority.queryServicesForInterface("IMotorController");
		if (l.empty())
			l = authority.queryServicesForInterface("IDrive");
		if (l.empty())
		{
			forward->setEnabled(false);
			backward->setEnabled(false);
			left->setEnabled(false);
			right->setEnabled(false);
			stop->setEnabled(false);
			bumper->setEnabled(false);
			odometry->setEnabled(false);
			mBtEnable->setEnabled(false);
			mBtDisable->setEnabled(false);
			return;
		}
		service = *l.begin();
		serviceChanged = true;
	}
	if (serviceChanged)
	{
		forward->setEnabled(true);
		backward->setEnabled(true);
		left->setEnabled(true);
		right->setEnabled(true);
		stop->setEnabled(true);
		if (authority.implementsInterface(service, "IMotorController"))
		{
			bumper->setEnabled(true);
			odometry->setEnabled(true);
			mBtEnable->setEnabled(true);
			mBtDisable->setEnabled(true);
		}
		else
		{
			bumper->setEnabled(false);
			odometry->setEnabled(false);
			mBtEnable->setEnabled(false);
			mBtDisable->setEnabled(false);
		}
	}
}

void DriveView::UI::onForward()
{
	if (rotSpeed == 0)
		transSpeed += mParent->velocityIncrement;
	rotSpeed = 0;
	lastCommand = Time::now();
	timer->start();
	tick();
}

void DriveView::UI::onBackward()
{
	if (rotSpeed == 0)
		transSpeed -= mParent->velocityIncrement;
	rotSpeed = 0;
	lastCommand = Time::now();
	timer->start();
	tick();
}

void DriveView::UI::onLeft()
{
	rotSpeed += mParent->velocityIncrement;
	lastCommand = Time::now();
	timer->start();
	tick();
}

void DriveView::UI::onRight()
{
	rotSpeed -= mParent->velocityIncrement;
	lastCommand = Time::now();
	timer->start();
	tick();
}

void DriveView::UI::onStop()
{
	rotSpeed = 0;
	transSpeed = 0;
	timer->stop();
	try {
		authority.callService<void>(service, "setVelocity",
		                            Velocity2(transSpeed, 0.0f, rotSpeed));
	}
	catch(...)
	{}
}

void DriveView::UI::onResetBumper()
{
	rotSpeed = 0;
	transSpeed = 0;
	timer->stop();
	try {
		authority.callService<void>(service, "setVelocity",
		                            Velocity2(transSpeed, 0.0f, rotSpeed));
		authority.callService<void>(service, "resetMotorStop");
	}
	catch(...)
	{}
}

void DriveView::UI::onResetOdometry()
{
	try {
		authority.callService<void>(service, "resetOdometry");
	}
	catch(...)
	{}
}

void DriveView::UI::onEnableMotors()
{
	try {
			authority.callService<void>(service, "enableMotors", true);
		}
	catch(...)
	{}
}

void DriveView::UI::onDisableMotors()
{
	try {
			authority.callService<void>(service, "enableMotors", false);
		}
	catch(...)
	{}
}

void DriveView::UI::onGrabKeyboard(bool activate)
{
	if(activate)
		this->grabKeyboard();
	else
		this->releaseKeyboard();
}

void DriveView::UI::keyPressEvent(QKeyEvent *e)
{
	if (e->key() == Qt::Key_Right)
	{
		onRight();
		return;
	}

	if (e->key() == Qt::Key_Left)
	{
		onLeft();
		return;
	}

	if (e->key() == Qt::Key_Up)
	{
		onForward();
		return;
	}

	if (e->key() == Qt::Key_Down)
	{
		onBackward();
		return;
	}

	if (e->key() == Qt::Key_Space)
	{
		onStop();
		return;
	}


	if (e->key() == Qt::Key_Escape)
	{
		mBtGrabKeyboard->setChecked(false);
		return;
	}


	QWidget::keyPressEvent(e);
}

///////////////////////////////////////////////////////////////////////////////

}}

MIRA_CLASS_SERIALIZATION(mira::robot::DriveView, mira::ViewPart);
