/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file GeometryTest.C
 *    Test cases for geometry objects.
 *
 * @author Tim Langner
 * @date   2012/04/03
 */

#include <boost/test/unit_test.hpp>

#include <boost/geometry/geometries/polygon.hpp>

#include <model/RigidModel.h>

using namespace mira;

/**
 * Performs tests of getBoundingBox. The bounding box of the unit geometries
 * (diameter 1, length 1, size 1) must be a box with size (1x1x1)
 */
BOOST_AUTO_TEST_CASE(TestBoundingBoxes)
{
	// transform all geometries so that their origin is in 0,0,0
	Pose3 origin;
	origin.x() = 0.5f;	// move forward 0.5 m
	origin.y() = 0.5f;	// move left 0.5m
	origin.z() = 0.5f;	// move up 0.5m

	model::Box box;
	box.size = Size3f(1, 1, 1);
	Box3f bb = box.getBoundingBox(origin);
	BOOST_CHECK(bb.minCorner.isApprox(Point3f(0, 0, 0)));
	BOOST_CHECK(bb.maxCorner.isApprox(Point3f(1, 1, 1)));

	model::Sphere sphere;
	sphere.radius = 0.5f;
	bb = sphere.getBoundingBox(origin);
	BOOST_CHECK(bb.minCorner.isApprox(Point3f(0, 0, 0)));
	BOOST_CHECK(bb.maxCorner.isApprox(Point3f(1, 1, 1)));

	model::Cylinder cylinder;
	cylinder.radius = 0.5f;
	cylinder.length = 1.0f;
	bb = cylinder.getBoundingBox(origin);
	BOOST_CHECK(bb.minCorner.isApprox(Point3f(0, 0, 0)));
	BOOST_CHECK(bb.maxCorner.isApprox(Point3f(1, 1, 1)));

	model::Cone cone;
	cone.radius = 0.5f;
	cone.length = 1.0f;
	// origin of cone is at the bottom so no need to move by 0.5m up
	origin.z() = 0.0f;
	bb = cone.getBoundingBox(origin);
	BOOST_CHECK(bb.minCorner.isApprox(Point3f(0, 0, 0)));
	BOOST_CHECK(bb.maxCorner.isApprox(Point3f(1, 1, 1)));
}

/**
 *  Performs test of getFootPrint method.
 *  Rotates box, cylinder and cone geometry so that their footprint is either
 *  a rect or a triangle.
 *  Calculates the size of the area of the difference between the footprint
 *  and a rect/triangle reference. The area needs to be ~0.0f
 */
BOOST_AUTO_TEST_CASE(TestFootprints)
{
	typedef boost::geometry::model::polygon<Point2f> polygon;
	std::list<polygon> difference;
	Polygon2f reference;
	reference.push_back(Point2f(0.0f, 0.0f));
	reference.push_back(Point2f(0.0f, 1.0f));
	reference.push_back(Point2f(1.0f, 1.0f));
	reference.push_back(Point2f(1.0f, 0.0f));
	reference.push_back(Point2f(0.0f, 0.0f));

	// transform all geometries so that their origin is in 0,0,0
	Pose3 origin;
	origin.x() = 0.5f;	// move forward 0.5 m
	origin.y() = 0.5f;	// move left 0.5m
	origin.z() = 0.5f;	// move up 0.5m
	origin.r = quaternionFromYawPitchRoll(0.0f, half_pi<float>(), 0.0f);

	model::Box box;
	box.size = Size3f(1, 1, 1);
	boost::geometry::difference(reference, box.getFootprint(origin), difference);
	float area = 0.0f;
	foreach(const polygon& p, difference)
		area += (float)boost::geometry::area(p);
	BOOST_CHECK_SMALL(area, 0.0001f);

	model::Cylinder cylinder;
	cylinder.radius = 0.5f;
	cylinder.length = 1.0f;
	difference.clear();
	boost::geometry::difference(reference, box.getFootprint(origin), difference);
	area = 0.0f;
	foreach(const polygon& p, difference)
		area += (float)boost::geometry::area(p);
	BOOST_CHECK_SMALL(area, 0.0001f);

	model::Cone cone;
	cone.radius = 0.5f;
	cone.length = 1.0f;
	// origin of cone is at the bottom so no need to move by 0.5m x
	origin.x() = 0.0f;
	reference.clear();
	reference.push_back(Point2f(0.0f, 0.0f));
	reference.push_back(Point2f(0.0f, 1.0f));
	reference.push_back(Point2f(1.0f, 0.5f));
	reference.push_back(Point2f(0.0f, 0.0f));
	difference.clear();
	boost::geometry::difference(reference, cone.getFootprint(origin), difference);
	area = 0.0f;
	foreach(const polygon& p, difference)
		area += (float)boost::geometry::area(p);
	BOOST_CHECK_SMALL(area, 0.0001f);
}
