/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file VisualizationPlotCurveBasic.h
 *    TODO Add description.
 *
 * @author Tim Langner, Christian Vollmer
 * @date   2011/07/26
 */

#ifndef _MIRA_VISUALIZATIONPLOTCURVEBASIC_H_
#define _MIRA_VISUALIZATIONPLOTCURVEBASIC_H_

#include <qwt_plot_curve.h>

#if (QWT_VERSION >= 0x060000)
# include <qwt_compat.h>
#endif

#include <serialization/adapters/Qt/QColor>

#include <visualization/VisualizationPlot.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

template<typename T>
class VisualizationPlotCurveBasic : public VisualizationPlotCurve
{

public:

	VisualizationPlotCurveBasic(const std::string& channelName) :
		mName(channelName),
		mCurve(NULL),
		mColor(Qt::black),
		mLineWidth(0.f),
		mLineStyle(Qt::SolidLine),
		mYAxis(QwtPlot::yLeft)
	{
		mChannel.setDataIntervalChangedCallback(boost::bind(&VisualizationPlotCurveBasic::dataIntervalChanged, this, _1));
		mChannel.setChannelChangedCallback(boost::bind(&VisualizationPlotCurveBasic::channelChanged, this));
	}

	virtual ~VisualizationPlotCurveBasic()
	{
		if (mCurve)
		{
			mCurve->detach();
			delete mCurve;
		}
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		VisualizationPlotCurve::reflect(r);
		channelProperty(r, mName, mChannel, "");
		r.property("Color", mColor,
		           setter<QColor>(&VisualizationPlotCurveBasic::setColor, this),
		           "The color of the curve", Qt::black);
		r.property("LineWidth", mLineWidth,
		           setter(&VisualizationPlotCurveBasic::setLineWidth, this),
		           "The line width of the curve", 0.f,
		           PropertyHints::minimum(0.f));

		static const std::string styleHint = MakeString() << Qt::SolidLine << "=SolidLine;"
		                                                  << Qt::DashLine << "=DashLine;"
		                                                  << Qt::DotLine << "=DotLine;"
		                                                  << Qt::DashDotLine << "=DashDotLine;"
		                                                  << Qt::DashDotDotLine << "=DashDotDotLine";
		r.property("LineStyle", mLineStyle,
		           setter(&VisualizationPlotCurveBasic::setLineStyle, this),
		           "The line style of the grid", Qt::SolidLine,
		           PropertyHints::enumeration(styleHint));

		static const std::string axisHint = MakeString() << QwtPlot::yLeft << "=Left;"
		                                                 << QwtPlot::yRight << "=Right";

		r.property("YAxis", mYAxis, setter(&VisualizationPlotCurveBasic::setYAxis, this),
			       "Attach curve to left or right axis",
			       PropertyHints::enumeration(axisHint));
	}

	virtual void setupScene(IVisualizationPlotSite* site)
	{
		mCurve = createCurve();
	}

	virtual void clear()
	{
		if (!mCurve)
			return;
		mCurve->detach();
		delete mCurve;
		mCurve = createCurve();
	}

	virtual DataConnection getDataConnection() { return DataConnection(mChannel); }

	virtual void channelChanged()
	{
		try {
			this->setName(mChannel.getID());
		}
		catch (XRuntime&) {}
		clear();
	}

	virtual void dataChanged(ChannelRead<T> data) {}

	virtual void dataIntervalChanged(ChannelReadInterval<T> data) {}

	virtual void setEnabled(bool enabled)
	{
		VisualizationPlotCurve::setEnabled(enabled);
		if (mCurve)
			mCurve->setVisible(isEnabled());
	}

	void redraw()
	{
		if (!mCurve)
			return;

		mCurve->setPen(QPen(mColor, mLineWidth, mLineStyle));
		VisualizationPlotCurve::getSite()->getPlot()->replot();
	}

	void setColor(const QColor& color)
	{
		mColor = color;
		redraw();
	}

	void setLineWidth(float width)
	{
		mLineWidth = width;
		redraw();
	}

	void setLineStyle(const Qt::PenStyle& style)
	{
		mLineStyle = style;
		redraw();
	}

	void setYAxis(int axis)
	{
		mYAxis = axis;
		if (!mCurve)
			return;

		mCurve->setYAxis(mYAxis);
		VisualizationPlotCurve::getSite()->updateAxes();
	}

	virtual QwtPlotItem* getPlotItem()
	{
		return mCurve;
	}

	QwtPlotCurve* createCurve()
	{
		mCurve = new QwtPlotCurve(getCurveTitle());
		mCurve->setPen(QPen(mColor, mLineWidth, mLineStyle));
		#if (QWT_VERSION >= 0x060000)
		mCurve->setLegendAttribute(QwtPlotCurve::LegendShowLine);
		#endif
		mCurve->setVisible(VisualizationPlotCurve::isEnabled());
		mCurve->attach(VisualizationPlotCurve::getSite()->getPlot());
		mCurve->setYAxis(mYAxis);
		VisualizationPlotCurve::getSite()->getPlot()->replot();
		return mCurve;
	}

	virtual QString getCurveTitle()
	{
		return mChannel.getID().c_str();
	}

	virtual int getYAxis()
	{
		return mYAxis;
	}

protected:
	std::string mName;
	ChannelProperty<T> mChannel;
	QwtPlotCurve* mCurve;
	QColor mColor;
	float mLineWidth;
	Qt::PenStyle mLineStyle;
	int mYAxis;
};

///////////////////////////////////////////////////////////////////////////////

}

#endif 
