/*
 * Copyright (C) 2013 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file DistortionModel.h
 *    Forward and reverse distortion model for radial and tangential
 *    lense distortion
 *
 * @author Erik Einhorn
 * @date   13/07/16
 */

#ifndef _MIRA_CAMERA_DISTORTIONPARAMETERS_H_
#define _MIRA_CAMERA_DISTORTIONPARAMETERS_H_

#include <geometry/Point.h>
#include <cameraparameters/CameraParametersExports.h>

namespace mira { namespace camera {

///////////////////////////////////////////////////////////////////////////////

/**
 * The distortion parameters for radial and tangential distortion according
 * to the standard model:
 *    Brown, Duane C., "Decentering distortion of lenses",
 *    Photogrammetric Engineering 32(3), pp. 444–462, 1966
 */
struct MIRA_CAMERAPARAMETERS_EXPORT DistortionParameters
{
	DistortionParameters() : k1(0.f),k2(0.f),k3(0.f),k4(0.f),k5(0.f),k6(0.f),
	                         p1(0.f),p2(0.f) {}
	DistortionParameters(float iK1, float iK2, float iP1, float iP2,
	                     float iK3 = 0.f, float iK4 = 0.f,
	                     float iK5 = 0.f, float iK6 = 0.f) {
		k1=iK1;	k2=iK2;
		k3=iK3;	k4=iK4;
		k5=iK5;	k6=iK6;
		p1=iP1;	p2=iP2;
	}

	template <typename Reflector>
	void reflect(Reflector& r)
	{
		r.property("K1",  k1,  "radial distortion");
		r.property("K2",  k2,  "radial distortion");
		r.property("K3",  k3,  "radial distortion", 0.f);
		r.property("K4",  k4,  "radial distortion", 0.f);
		r.property("K5",  k5,  "radial distortion", 0.f);
		r.property("K6",  k6,  "radial distortion", 0.f);
		r.property("P1",  p1,  "tangential distortion");
		r.property("P2",  p2,  "tangential distortion");
	}

	union {
		struct {
			float k1; ///< radial distortion param
			float k2; ///< radial distortion param
			float k3; ///< radial distortion param
			float k4; ///< radial distortion param
			float k5; ///< radial distortion param
			float k6; ///< radial distortion param
		};
		float k[6]; ///< the radial distortion params (same as k1 to k6)
	};

	union {
		struct {
			float p1; ///< tangential distortion param
			float p2; ///< tangential distortion param
		};
		float p[2]; ///< the tangential distortion params (same as p1 and p2)
	};
	
	bool operator==(const DistortionParameters& other) const {
		return ((k1 == other.k1) && (k2 == other.k2) && (p1 == other.p1) && (p2 == other.p2) &&
		        (k3 == other.k3) && (k4 == other.k4) && (k5 == other.k5) && (k6 == other.k6));
	}	
};

///////////////////////////////////////////////////////////////////////////////

} } // namespace

#endif
