/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file OgreWidgetEx.C
 *    Implementation of OgreWidgetEx.h.
 *
 * @author Erik Einhorn
 * @date   2012/09/01
 */

#include <widgets/OgreWidgetEx.h>

#include <OGRE/OgreRoot.h>
#include <OGRE/OgreRenderWindow.h>
#include <OGRE/OgreQuaternion.h>

#include <OGRE/OgreRectangle2D.h>
#include <OGRE/OgreHardwareBufferManager.h>
#include <OGRE/OgreMaterialManager.h>


namespace mira {

static const float sNearClipping = 0.01f;
static const float sFarClipping = 10000.0f;

///////////////////////////////////////////////////////////////////////////////

/// render mode modifier (modifies the rendering technique, to enable wireframe modes)
class OgreWidgetEx::RenderModeTechniqueMod : public Ogre::RenderQueue::RenderableListener
{
public:

	RenderModeTechniqueMod(OgreWidgetEx* w) :
		widget(w), backgroundColor(0, 0, 0)
	{
		wireframeColor = Ogre::ColourValue::Red;

		wireframeMaterial = Ogre::MaterialManager::getSingleton().getByName("Vis3DWireframeMaterial");
		if (wireframeMaterial.isNull())
		{
			wireframeMaterial = Ogre::MaterialManager::getSingleton().create("Vis3DWireframeMaterial",
				Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
			Ogre::Pass* p = wireframeMaterial->getTechnique(0)->getPass(0);
			p->setLightingEnabled(false);
			p->setPolygonMode(Ogre::PM_WIREFRAME);
			p->setCullingMode(Ogre::CULL_NONE);

			Ogre::TextureUnitState* t = p->createTextureUnitState();
			t->setColourOperationEx(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL, Ogre::LBS_CURRENT, wireframeColor);
			wireframeMaterial->load();
		}

		hiddenLineMaterial = Ogre::MaterialManager::getSingleton().getByName("Vis3DHiddenLineMaterial");
		if (hiddenLineMaterial.isNull())
		{
			hiddenLineMaterial = Ogre::MaterialManager::getSingleton().create("Vis3DHiddenLineMaterial",
				Ogre::ResourceGroupManager::DEFAULT_RESOURCE_GROUP_NAME);
			Ogre::Pass* p = hiddenLineMaterial->getTechnique(0)->getPass(0);

			p->setLightingEnabled(false);
			Ogre::TextureUnitState* t = p->createTextureUnitState();
			t->setColourOperationEx(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL, Ogre::LBS_CURRENT, backgroundColor);

			p = hiddenLineMaterial->getTechnique(0)->createPass();
			p->setLightingEnabled(false);
			p->setDepthBias(10);
			p->setPolygonMode(Ogre::PM_WIREFRAME);

			t = p->createTextureUnitState();
			t->setColourOperationEx(Ogre::LBX_SOURCE1, Ogre::LBS_MANUAL, Ogre::LBS_CURRENT, wireframeColor);
			hiddenLineMaterial->load();
		}
	}

	~RenderModeTechniqueMod() {}

	bool renderableQueued(Ogre::Renderable* rend, Ogre::uint8 groupID,
		Ogre::ushort priority, Ogre::Technique** tech, Ogre::RenderQueue* queue)
	{
		if(widget->mRenderMode==OgreWidgetEx::NORMAL)
			return true;

		// use renderable pointer to determine what colour for now
		// may want to use a categorisation in future
		Ogre::RenderOperation op;
		rend->getRenderOperation(op);

		switch(widget->mRenderMode)
		{
		case OgreWidgetEx::WIREFRAME:
			*tech = wireframeMaterial->getSupportedTechnique(0);
			return true;
		case OgreWidgetEx::HIDDEN_LINE:
			*tech = hiddenLineMaterial->getSupportedTechnique(0);
			return true;
		default:
			return true;
		};
		return true;
	}

	OgreWidgetEx* widget;

	Ogre::ColourValue wireframeColor;
	Ogre::ColourValue backgroundColor;
	Ogre::MaterialPtr wireframeMaterial;
	Ogre::MaterialPtr hiddenLineMaterial;
};

class OgreWidgetEx::ColouredRectangle2D : public Ogre::Rectangle2D
{
public:

	static const int COLOUR_BINDING = 3;

	ColouredRectangle2D(bool includeTextureCoordinates = false) {
		Ogre::VertexDeclaration* decl = mRenderOp.vertexData->vertexDeclaration;

		decl->addElement(COLOUR_BINDING, 0, Ogre::VET_COLOUR, Ogre::VES_DIFFUSE);
		Ogre::VertexBufferBinding* bind = mRenderOp.vertexData->vertexBufferBinding;

		Ogre::HardwareVertexBufferSharedPtr vbuf =
			Ogre::HardwareBufferManager::getSingleton().createVertexBuffer(
				decl->getVertexSize(COLOUR_BINDING),
					mRenderOp.vertexData->vertexCount,
					Ogre::HardwareBuffer::HBU_STATIC_WRITE_ONLY);

		// Bind buffer
		bind->setBinding(COLOUR_BINDING, vbuf);

		setQueryFlags(0); // make sure, that we are not included in any scene query
	}

	void setColors(const Ogre::ColourValue& topLeft, const Ogre::ColourValue& bottomLeft,
	               const Ogre::ColourValue& topRight, const Ogre::ColourValue & bottomRight)
	{
		Ogre::HardwareVertexBufferSharedPtr vbuf =
		      mRenderOp.vertexData->vertexBufferBinding->getBuffer(COLOUR_BINDING);
		unsigned int* vertex = static_cast<unsigned int*>(vbuf->lock(Ogre::HardwareBuffer::HBL_DISCARD));
		const Ogre::VertexElementType srcType = Ogre::VertexElement::getBestColourVertexElementType();
		*vertex++ = Ogre::VertexElement::convertColourValue( topLeft, srcType );
		*vertex++ = Ogre::VertexElement::convertColourValue( bottomLeft, srcType );
		*vertex++ = Ogre::VertexElement::convertColourValue( topRight, srcType );
		*vertex++ = Ogre::VertexElement::convertColourValue( bottomRight, srcType );
		vbuf->unlock();
	}
};

///////////////////////////////////////////////////////////////////////////////

OgreWidgetEx::OgreWidgetEx(QWidget* parent) : OgreWidget(parent),
	mBackgroundMode(BACKGROUND_GRADIENT),
	mBackgroundColor1(Ogre::ColourValue(0.78f,0.86f,1.0f)),
	mBackgroundColor2(Ogre::ColourValue(0.90f,0.94f,1.0f)),
	mCameraMode(PERSPECTIVE),
	mRenderMode(NORMAL),
	mStereoMode(NONE),mStereoEyeDistance(0.2f),
	mShowBoundingBoxes(false),
	mLight(NULL), mCamera(NULL), mCameraRight(NULL),
	mViewport(NULL), mViewportRight(NULL),
	mSceneManager(NULL), mBackground(NULL), mRenderModeTechniqueMod(NULL)
{
	mCameraSettings.distance = 10.0f;
	Point3f from(-1.0f,-1.0f,1.0f);
	from.normalize();
	from *= mCameraSettings.distance;
	lookAt(from, Point3f(0,0,0));
	mCameraSettings.roll  = 0.0f;
}

OgreWidgetEx::~OgreWidgetEx()
{
	// destroy the scene manager if we had one
	if(mSceneManager!=NULL)
	{
		Ogre::Root::getSingletonPtr()->destroySceneManager(mSceneManager);
		delete mRenderModeTechniqueMod;
	}
}

void OgreWidgetEx::initOgre()
{
	Ogre::SceneType sceneManagerType = Ogre::ST_GENERIC;
	mSceneManager = Ogre::Root::getSingletonPtr()->createSceneManager(sceneManagerType);
	mSceneManager->setAmbientLight(Ogre::ColourValue(1,1,1));


	mCamera = mSceneManager->createCamera( "MainCamera" );
	mCamera->setFixedYawAxis(true, Ogre::Vector3::UNIT_Z);
	mCamera->setNearClipDistance(sNearClipping);
	mCamera->setFarClipDistance(sFarClipping);

	mViewport = getRenderWindow()->addViewport( mCamera, 0, 0.0f,0.0f, 1.0f,1.0f);
	mViewport->setBackgroundColour(mBackgroundColor1);
	mViewport->setClearEveryFrame(true);

	mCamera->setAspectRatio(Ogre::Real(width()) / Ogre::Real(height()));

	mCameraRight = mSceneManager->createCamera( "RightCamera" );
	mCameraRight->setFixedYawAxis(true, Ogre::Vector3::UNIT_Z);
	mCameraRight->setNearClipDistance(sNearClipping);
	mCameraRight->setFarClipDistance(sFarClipping);
	mCameraRight->setAspectRatio(Ogre::Real(width()) / Ogre::Real(height()));


	mLight = mSceneManager->createLight();
	mLight->setType(Ogre::Light::LT_DIRECTIONAL);
	mLight->setDiffuseColour(Ogre::ColourValue(1.0, 1.0, 1.0));
	mLight->setSpecularColour(Ogre::ColourValue(1.0, 1.0, 1.0));
	mLight->setDirection(Ogre::Vector3( 0.5, 1, -1.5).normalisedCopy());

	///////////////////////////////////////////////////////////////////////////////

	// Create background material
	std::string name = "Background" + toString(this);
	Ogre::MaterialPtr material = Ogre::MaterialManager::getSingleton().create(name, "General");
	material->getTechnique(0)->getPass(0)->setDepthCheckEnabled(false);
	material->getTechnique(0)->getPass(0)->setDepthWriteEnabled(false);
	material->getTechnique(0)->getPass(0)->setLightingEnabled(false);

	// Create background rectangle covering the whole screen
	ColouredRectangle2D* rect = new ColouredRectangle2D();
	rect->setCorners(-1.0, 1.0, 1.0, -1.0);
	rect->setMaterial(name);

	// Set the colours
	//rect->setColours( Ogre::ColourValue::Red, Ogre::ColourValue::Green, Ogre::ColourValue::Blue, Ogre::ColourValue::Black );

	// Render the background before everything else
	rect->setRenderQueueGroup(Ogre::RENDER_QUEUE_BACKGROUND);

	// Use infinite AAB to always stay visible
	Ogre::AxisAlignedBox aabInf;
	aabInf.setInfinite();
	rect->setBoundingBox(aabInf);

	// Attach background to the scene
	Ogre::SceneNode* node = mSceneManager->getRootSceneNode()->createChildSceneNode(name);
	node->attachObject(rect);

	mBackground = rect;

	///////////////////////////////////////////////////////////////////////////////

	mRenderModeTechniqueMod = new RenderModeTechniqueMod(this);
	mCamera->getSceneManager()->getRenderQueue()->setRenderableListener(mRenderModeTechniqueMod);


	setCameraMode(mCameraMode);
	setBackgroundMode(mBackgroundMode);
	setStereoMode(mStereoMode);
	showBoundingBoxes(mShowBoundingBoxes);

	updateCamera();
}

void OgreWidgetEx::setBackgroundMode(BackgroundMode mode)
{
	mBackgroundMode = mode;

	switch(mBackgroundMode)
	{
	case BACKGROUND_SOLID:
		mBackground->setVisible(false);
		mViewport->setBackgroundColour(mBackgroundColor1);
		if(mViewportRight)
			mViewportRight->setBackgroundColour(mBackgroundColor1);
		mRenderModeTechniqueMod->backgroundColor = mBackgroundColor1;
		break;
	case BACKGROUND_WHITE:
		mBackground->setVisible(false);
		mViewport->setBackgroundColour(Ogre::ColourValue::White);
		if(mViewportRight)
			mViewportRight->setBackgroundColour(Ogre::ColourValue::White);
		mRenderModeTechniqueMod->backgroundColor = Ogre::ColourValue::White;
		break;
	case BACKGROUND_BLACK:
		mBackground->setVisible(false);
		mViewport->setBackgroundColour(Ogre::ColourValue::Black);
		if(mViewportRight)
			mViewportRight->setBackgroundColour(Ogre::ColourValue::Black);
		mRenderModeTechniqueMod->backgroundColor = Ogre::ColourValue::Black;
		break;
	case BACKGROUND_GRADIENT:
		mBackground->setVisible(true);
		mBackground->setColors(mBackgroundColor1,mBackgroundColor2,mBackgroundColor1,mBackgroundColor2);
		break;
	}
}

void OgreWidgetEx::setBackgroundColor1(Ogre::ColourValue color1)
{
	mBackgroundColor1 = color1;
	setBackgroundMode(mBackgroundMode); // will set the colors internally
}

void OgreWidgetEx::setBackgroundColor2(Ogre::ColourValue color2)
{
	mBackgroundColor2 = color2;
	setBackgroundMode(mBackgroundMode); // will set the colors internally
}

void OgreWidgetEx::setBackgroundColors(Ogre::ColourValue color1,
                                       Ogre::ColourValue color2)
{
	mBackgroundColor1 = color1;
	mBackgroundColor2 = color2;
	setBackgroundMode(mBackgroundMode); // will set the colors internally
}

void OgreWidgetEx::setCameraMode(CameraMode mode)
{
	switch(mode)
	{
	case PERSPECTIVE:
		mCamera->setProjectionType(Ogre::PT_PERSPECTIVE);
		mCameraRight->setProjectionType(Ogre::PT_PERSPECTIVE);
		break;
	case ORTHOGRAPHIC:
		mCamera->setProjectionType(Ogre::PT_ORTHOGRAPHIC);
		mCamera->setOrthoWindowWidth(10);
		break;
	case BIRDS_EYE:
		mCamera->setProjectionType(Ogre::PT_ORTHOGRAPHIC);
		mCamera->setOrthoWindowWidth(10);
		break;
	}
}

void OgreWidgetEx::setRenderMode(RenderMode mode)
{
	mRenderMode = mode;
	if(mode != NORMAL && mBackgroundMode == BACKGROUND_GRADIENT)
		setBackgroundMode(BACKGROUND_SOLID);
}

void OgreWidgetEx::setStereoMode(StereoMode mode)
{
	switch(mode)
	{
	case NONE:
		getRenderWindow()->removeViewport(1);
		mViewportRight = NULL;
		mViewport->setDimensions(0.0f,0.0f,1.0f,1.0f);
		mCamera->setAspectRatio(Ogre::Real(width()) / Ogre::Real(height()));
		break;

	case CROSSED_EYE:
		mViewportRight = getRenderWindow()->addViewport( mCameraRight, 1, 0.5f,0.0f, 0.5f,1.0f);
		mViewportRight->setClearEveryFrame(true);
		mViewport->setDimensions(0.0f,0.0f,0.5f,1.0f);
		setBackgroundMode(mBackgroundMode);
		mCamera->setAspectRatio(0.5f*Ogre::Real(width()) / Ogre::Real(height()));
		mCameraRight->setAspectRatio(0.5f*Ogre::Real(width()) / Ogre::Real(height()));
		break;
	}
}

void OgreWidgetEx::showBoundingBoxes(bool show)
{
	mShowBoundingBoxes = show;
	mSceneManager->showBoundingBoxes(mShowBoundingBoxes);
}

void OgreWidgetEx::setCamera(const Point3f& position, float yaw, float pitch, float roll, float distance)
{
	mCameraSettings.position = position;
	mCameraSettings.yaw = yaw;
	mCameraSettings.pitch = pitch;
	mCameraSettings.roll = roll;
	mCameraSettings.distance = distance;
	updateCamera();
}

void OgreWidgetEx::lookAt(const Point3f& from, const Point3f& to)
{
	Eigen::Vector3f d = to - from;
	float l = d.norm();

	mCameraSettings.pitch = -std::asin( d.z() / l );
	mCameraSettings.yaw   = std::atan2( d.y(), d.x());

	mCameraSettings.position = from;
	mCameraSettings.distance = l;
	updateCamera();
}

void OgreWidgetEx::updateCamera()
{
	Ogre::Vector3 v;

	mCameraSettings.distance = std::max(mCameraSettings.distance, sNearClipping);

	if(mCamera==NULL)
		return;

	switch(mCameraMode)
	{
	case PERSPECTIVE:
		v = Ogre::Vector3(- mCameraSettings.distance,0,0);
		break;
	case BIRDS_EYE:
		mCameraSettings.pitch = 0.49f * pi<float>(); // approx. 90 deg from above (90 deg would be a singularity)
		// dont break, fall through
	case ORTHOGRAPHIC:
		v = Ogre::Vector3(-sFarClipping/2.0f,0,0);
		mCamera->setOrthoWindowWidth(mCameraSettings.distance);
		break;
	}

	assert(mCamera!=NULL);

	mCamera->setPosition(Ogre::Vector3(0,0,0));
	mCamera->lookAt(Ogre::Vector3(1,0,0));
	mCamera->move(Ogre::Vector3(mCameraSettings.position.x(),
								mCameraSettings.position.y(),
								mCameraSettings.position.z()));

	Eigen::Quaternionf q = quaternionFromYawPitchRoll(mCameraSettings.yaw, mCameraSettings.pitch, mCameraSettings.roll);
	Ogre::Quaternion oq(q.w(),
						q.x(),
						q.y(),
						q.z());
	mCamera->rotate(oq);

	mCameraRight->setPosition(Ogre::Vector3(0,0,0));
	mCameraRight->lookAt(Ogre::Vector3(1,0,0));
	mCameraRight->move(Ogre::Vector3(mCameraSettings.position.x(),
								mCameraSettings.position.y(),
								mCameraSettings.position.z()));
	mCameraRight->rotate(oq);

	if(mStereoMode == CROSSED_EYE) { // move eyes to the side
		mCamera->moveRelative(Ogre::Vector3(0.5*mStereoEyeDistance,0,0));
		mCameraRight->moveRelative(Ogre::Vector3(-0.5*mStereoEyeDistance,0,0));
	}
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

