/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file ImgWidget.C
 *    Implementation of ImgWidget.
 *
 * @author Erik Einhorn
 * @date   2012/02/21
 */

#include <QImage>
#include <QScrollArea>
#include <QMenu>
#include <QMouseEvent>
#include <QVBoxLayout>
#include <QPainter>

#include <widgets/ImgWidget.h>

#include <widgets/QtUtils.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class ImgWidget::ImgPainterWidget : public QWidget
{
public:

	ImgPainterWidget(QWidget* parent) :
		QWidget(parent), mListener(NULL)
	{
		setAutoFillBackground(false);
	}

	void setImage(const QImage& img, bool scale, QSize size = QSize())
	{
		if(!scale) {
			mCurrentImage=img;
			size = img.size();
		} else
			mCurrentImage=img.scaled(size, Qt::KeepAspectRatio);

		resize(size);
		setFixedSize(size);
		repaint();
	}

	void setMouseListener(ImgWidget::IMouseListener* listener)
	{
		mListener = listener;
	}

protected:

	void paintEvent (QPaintEvent* e)
	{
		QPainter p(this);

		if(!mCurrentImage.isNull())
			p.drawImage(e->rect(), mCurrentImage, e->rect());
	}

	void mousePressEvent(QMouseEvent* e)
	{
		if(mListener!=NULL)
			mListener->onMousePressed(e);

		QWidget::mousePressEvent(e);
	}

	void mouseReleaseEvent(QMouseEvent* e)
	{
		if(mListener!=NULL)
			mListener->onMouseReleased(e);

		QWidget::mouseReleaseEvent(e);
	}

	void mouseMoveEvent(QMouseEvent* e)
	{
		if(mListener!=NULL)
			mListener->onMouseMoved(e);

		QWidget::mouseMoveEvent(e);
	}

private:

	QImage mCurrentImage;

	ImgWidget::IMouseListener* mListener;
};

///////////////////////////////////////////////////////////////////////////////

ImgWidget::ImgWidget(QWidget* parent, bool scroll) :
		QWidget(parent), mAutoResize(false), mAutoScale(false)
{
	mScrollArea = NULL;
	mLayout = NULL;
	mContentWidget = NULL;
	enableScroll(scroll);
}

ImgWidget::~ImgWidget()
{

}

QWidget * ImgWidget::getContentWidget()
{
	return mContentWidget;
}

void ImgWidget::setAutoResize(bool autoResize)
{
	mAutoResize=autoResize;
}

void ImgWidget::setAutoScale(bool autoScale)
{
	mAutoScale = autoScale;
}

void ImgWidget::enableScroll(bool enable)
{
	delete mContentWidget;
	delete mScrollArea;
	delete mLayout;

	mLayout = new QVBoxLayout(this);
	if(enable) {
		mScrollArea = new QScrollArea(this);
		mLayout->addWidget(mScrollArea);
		mContentWidget = new ImgPainterWidget(mScrollArea->viewport());
		mScrollArea->setWidget(mContentWidget);
	} else {
		mContentWidget = new ImgPainterWidget(this);
		mLayout->addWidget(mContentWidget);
		mScrollArea = NULL;
	}

}

void ImgWidget::contentsMousePressEvent ( QMouseEvent * e )
{
}

void ImgWidget::setMouseListener(IMouseListener* pListener)
{
	mContentWidget->setMouseListener(pListener);
}

void ImgWidget::setImage(const Img<>& image, double scaleMin, double scaleMax)
{
	mCurrentImage = QtUtils::toQImage(image, scaleMin, scaleMax);
	if(!mCurrentImage.isNull()) {
		if(mCurrentImage.format()==QImage::Format_Indexed8 && !mColorTable.isEmpty())
				mCurrentImage.setColorTable(mColorTable);

		if(mAutoResize)
			setFixedSize(mCurrentImage.size());

		if(mAutoScale) {
			QSize s = this->size();
			if (mScrollArea)
				s = mScrollArea->size();
			mContentWidget->setImage(mCurrentImage, true, s - QSize(4, 4));
		} else
			mContentWidget->setImage(mCurrentImage, false);
	} else {
		mContentWidget->setImage(mCurrentImage, false);
	}
}


void ImgWidget::setColormap(const Colormap& map)
{
	mColorTable.resize(map.size());
	for(size_t j=0;j<map.size();++j)
	{
		Color::RGB c = map.get(j);
		mColorTable[j] = QColor(int(c.r*255),int(c.g*255), int(c.b*255)).rgb();
	}
}

void ImgWidget::clearColormap()
{
	mColorTable.clear();
}

void ImgWidget::mousePressEvent(QMouseEvent* pEvent)
{
	// if clicked with right button
	if (pEvent->button() == Qt::RightButton) {
		//mPopupMenu->setMouseTracking(true);
		//mPopupMenu->exec(pEvent->globalPos());
	}

	QWidget::mousePressEvent(pEvent);
}


///////////////////////////////////////////////////////////////////////////////

}
