/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file LineStripObject.C
 *    Implementation of LineStripObject.h.
 *
 * @author Erik Einhorn
 * @date   2012/11/06
 */

#include <visualization/3d/LineStripObject.h>

#include <OGRE/OgreBillboardChain.h>


namespace mira {

///////////////////////////////////////////////////////////////////////////////

LineStripObject::LineStripObject(Ogre::SceneManager* sceneManager, Ogre::SceneNode* parent) :
		VisualizationObject(sceneManager,parent)
{
	std::string materialName = "LineStripObject"+toString(this)+"/Material";
	mMaterial = Ogre::MaterialManager::getSingleton().create(materialName, "MIRAMaterials");
	mMaterial->setReceiveShadows(false);
	mMaterial->getTechnique(0)->setLightingEnabled(false);

	mBillboardLines = mSceneManager->createBillboardChain();
	mBillboardLines->setMaterialName(mMaterial->getName());
	mBillboardLines->setMaxChainElements(2); // lets start with 2 elements per chain
	mBillboardLines->clearAllChains();
	mNode->attachObject(mBillboardLines);

	mPixelLines = mSceneManager->createManualObject("LineStripObjectPixelLines"+ toString(this));
	mNode->attachObject(mPixelLines);

	mColor = Ogre::ColourValue::Black;
	mWidth = 0.1f;
	mIsVisible = true;
}

LineStripObject::~LineStripObject()
{
	mSceneManager->destroyBillboardChain(mBillboardLines);
	mSceneManager->destroyManualObject(mPixelLines);

	if (!mMaterial.isNull())
		Ogre::MaterialManager::getSingletonPtr()->remove(mMaterial->getName());
}

///////////////////////////////////////////////////////////////////////////////

void LineStripObject::setColor(const Ogre::ColourValue& color)
{
	mColor = color;
	foreach(auto s, mStrips)
	{
		foreach(Point& x, s->pts)
			x.c = color;
	}
	regenerateAll();
}

void LineStripObject::setVisible(bool visible, bool cascade)
{
	mIsVisible = visible;
	manageVisibility();
}

void LineStripObject::enableAlphaBlending(bool useAlpha)
{
	if(useAlpha)
		mMaterial->setSceneBlending(Ogre::SBT_TRANSPARENT_ALPHA);
	else
		mMaterial->setSceneBlending(Ogre::SBT_REPLACE);
}


///////////////////////////////////////////////////////////////////////////////

void LineStripObject::setLineWidth(float width)
{
	mWidth = width;
	Ogre::BillboardChain* c = mBillboardLines;
	for(size_t j=0; j<c->getNumberOfChains(); ++j)
	{
		// ogre returns 1 for an empty chain, dammit!
		if(c->getNumChainElements(j)<=1)
			continue;
		for(size_t i=0; i<c->getNumChainElements(j); ++i)
		{
			Ogre::BillboardChain::Element e = c->getChainElement(j, i);
			e.width = mWidth;
			c->updateChainElement(j,i,e);
		}
	}
	manageVisibility();
}

void LineStripObject::clear()
{
	mBillboardLines->clearAllChains();
	mPixelLines->clear();
	mStrips.clear();
}

/// Starts a new line strip.
void LineStripObject::begin()
{
	if(mActiveStrip)
		return; // we are already within a begin() end() block

	mActiveStrip.reset(new Strip);
}

/// Finishes the current line strip.
void LineStripObject::end()
{
	if(!mActiveStrip)
		return; // we have no active strip

	std::size_t numPts = mActiveStrip->pts.size();

	// ignore empty line strip segments
	if(numPts==0) {
		mActiveStrip.reset();
		return;
	}

	// create the billboard line
	mStrips.push_back(mActiveStrip);
	regenerateStrip(mStrips.size()-1);

	mActiveStrip.reset();
}

void LineStripObject::point(const Ogre::Vector3& p)
{
	point(p, mColor);
}

void LineStripObject::point(const Ogre::Vector3& point, const Ogre::ColourValue& color)
{
	if(!mActiveStrip) {
		MIRA_THROW(XLogical, "You can use LineStripObject::point() in between calls to begin() and end() only");
	}

	mActiveStrip->pts.push_back(Point(point,color));
}


void LineStripObject::add(const Ogre::Vector3& point)
{
	add(point, mColor);
}

void LineStripObject::add(const Ogre::Vector3& p, const Ogre::ColourValue& color)
{
	if(mActiveStrip) {
		// if we are in between a a begin()-end() pair, do the same as point()
		point(p, color);
		return;
	}

	if(mStrips.empty()) {
		begin();
		point(p, color);
		end();
	} else {
		// alter last strip
		mStrips.back()->pts.push_back(Point(p, color));
		regenerateStrip(mStrips.size()-1);
	}
}

void LineStripObject::addLine(const Ogre::Vector3& p1,const Ogre::Vector3& p2)
{
	addLine(p1,p1,mColor);
}

void LineStripObject::addLine(const Ogre::Vector3& p1,const Ogre::Vector3& p2, const Ogre::ColourValue& color)
{
	begin();
	point(p1,color);
	point(p2,color);
	end();
}

///////////////////////////////////////////////////////////////////////////////

void LineStripObject::regenerateStrip(std::size_t i)
{
	const boost::shared_ptr<Strip>& s = mStrips[i];
	std::size_t numPts = s->pts.size();

	// prepare strip in billboard line
	if(i>=mBillboardLines->getNumberOfChains()) {
		// the i-th chain does not exist yet, we need to regenerate all
		regenerateAll();
		return;
	} else {
		// chain exists already, but make sure it is cleared
		mBillboardLines->clearChain(i);
	}

	// if chains cannot take enough points, we need to regenerate all
	if(numPts>mBillboardLines->getMaxChainElements()) {
		regenerateAll();
		return;
	}

	// prepare strip in the manual object for the pixel line
	bool useExistingSection = false;
	if(s->pixelLineSection<0) {
		// no section for this strip, begin new one
		mPixelLines->begin("NoLight", Ogre::RenderOperation::OT_LINE_STRIP);
		s->pixelLineSection = -1;
	} else {
		assert(s->pixelLineSection<(int)mPixelLines->getNumSections());
		// otherwise start the existing section of the manual object
		mPixelLines->beginUpdate((std::size_t)s->pixelLineSection);
		useExistingSection = true;
	}

	foreach(const Point& x, s->pts)
	{
		Ogre::BillboardChain::Element e;
		e.position = x.p;
		e.width = mWidth;
		e.colour = x.c;
		mBillboardLines->addChainElement(i, e);

		mPixelLines->position(x.p);
		mPixelLines->colour(x.c);
	}

	if(mPixelLines->end()!=NULL && !useExistingSection)
		s->pixelLineSection = (int)mPixelLines->getNumSections()-1;

	manageVisibility();
}

void LineStripObject::regenerateAll()
{
	// clear renderables
	mBillboardLines->clearAllChains();
	mPixelLines->clear();

	// resize BillboardChain properly
	std::size_t maxNumPts=0;
	foreach(auto s, mStrips)
		maxNumPts=std::max(s->pts.size(), maxNumPts);

	// find new size to grow to
	std::size_t newSize = std::max(mBillboardLines->getMaxChainElements(),std::size_t(2));
	while(maxNumPts>newSize)
		newSize *= 2;

	if(newSize>mBillboardLines->getMaxChainElements())
		mBillboardLines->setMaxChainElements(newSize);

	// find new size to grow to
	std::size_t newChainCount = std::max(mBillboardLines->getNumberOfChains(),std::size_t(1));
	while(mStrips.size()>newChainCount)
		newChainCount *= 2;

	if(newChainCount>mBillboardLines->getNumberOfChains())
		mBillboardLines->setNumberOfChains(newChainCount);

	// regenerate each strip
	for(std::size_t i=0; i<mStrips.size(); ++i)
	{
		mStrips[i]->pixelLineSection = -1; // reset section
		regenerateStrip(i);
	}
}

void LineStripObject::manageVisibility()
{
	bool nullLine = (mWidth<=0.0001f);
	mBillboardLines->setVisible(isVisible() && !nullLine);
	mPixelLines->setVisible(isVisible() && nullLine);
}

///////////////////////////////////////////////////////////////////////////////

} // namespace

