/*
 * Copyright (C) 2014 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file MeasureTool2D.C
 *    Tool to display the coordinates of points in the 2D view and to measure distances
 *    between two points.
 *
 * @author Christian Reuther
 * @date   2014/09/10
 */

#include <visualization/VisualizationTool2D.h>

#include <QMouseEvent>
#include <QGraphicsView>

#include <boost/optional.hpp>
#include <transform/Pose.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

class MIRA_GUI_VISUALIZATION_EXPORT MeasureTool2D : public VisualizationTool2D
{
	MIRA_META_OBJECT(MeasureTool2D,
		("Name", "Measure")
		("Category", "Interact")
		("Description", "Measure distance and angle")
		("Default", "false")
		("Order", "400"))

public:
	MeasureTool2D() : VisualizationTool2D(), mItems(NULL) {
		mTextItem = new QGraphicsSimpleTextItem();
		mTextItem->setFlag(QGraphicsItem::ItemIgnoresTransformations, true);
		mTextItem->setPen(QPen(Qt::black));
		mTextItem->setBrush(Qt::NoBrush);

		mTextBackgroundItem = new QGraphicsRectItem();
		mTextBackgroundItem->setFlag(QGraphicsItem::ItemIgnoresTransformations, true);
		mTextBackgroundItem->setPen(Qt::NoPen);
		mTextBackgroundItem->setBrush(QBrush(Qt::lightGray));

		mLineItem = new QGraphicsLineItem();
		mLineItem->setPen(QPen(Qt::lightGray, 0.0));
	}

	virtual ~MeasureTool2D() {
		auto site = this->getSite();
		if(!site || !site->getSceneManager())
			return;

		site->getSceneManager()->removeItem(mItems);
		if(mTextItem) {
			mItems->removeFromGroup(mTextItem);
			delete mTextItem;
			mTextItem = NULL;
		}
		if(mTextBackgroundItem) {
			mItems->removeFromGroup(mTextBackgroundItem);
			delete mTextBackgroundItem;
			mTextBackgroundItem = NULL;
		}
		if(mLineItem) {
			mItems->removeFromGroup(mLineItem);
			delete mLineItem;
			mLineItem = NULL;
		}
		if(mItems) {
			delete mItems;
			mItems = NULL;
		}
	}

	virtual void init(IVisualizationSite* site) {
		VisualizationTool2D::init(site);

		const float baseZ = getSite()->getToolBaseZValue();
		mTextBackgroundItem->setZValue(baseZ + 0.0f);
		mTextItem->setZValue(baseZ + 0.1f);
		mLineItem->setZValue(baseZ + 0.05f);
	}

public:
	virtual void onMousePressed(QMouseEvent* e) {
		if(e->button() != Qt::LeftButton)
			return;

		QPointF current = getSite()->getViewManager()->mapToScene(e->pos());
		mMouseDownPos = current;
		mItems->setVisible(true);
		updateItems(current, current);
		e->accept();
	}

	virtual void onMouseReleased(QMouseEvent* e) {
		if(e->button() != Qt::LeftButton)
			return;

		mItems->setVisible(false);
		mMouseDownPos.reset();
		e->accept();
	}

	virtual void onMouseMoved(QMouseEvent* e) {
		if(!mMouseDownPos) {
			e->accept();
			return;
		}

		const QPointF current = getSite()->getViewManager()->mapToScene(e->pos());
		const QPointF prev = *mMouseDownPos;
		updateItems(prev, current);
		e->accept();
	}

	virtual QIcon getIcon() {
		return QIcon(QPixmap(":/icons/ToolGoal.png"));
	}

public:
	virtual void activate() {
		VisualizationTool2D::activate();

		if(mItems)
			mItems->setVisible(false);
	}

	virtual void deactivate() {
		VisualizationTool2D::deactivate();

		if(mItems)
			mItems->setVisible(false);
	}

	virtual void update(Duration dt) {
		// Do NOT mess with my positions!
	}

	virtual QGraphicsItem* setupScene(QGraphicsScene* scene) {
		mItems = new QGraphicsItemGroup();
		mItems->addToGroup(mTextItem);
		mItems->addToGroup(mTextBackgroundItem);
		mItems->addToGroup(mLineItem);
		scene->addItem(mItems);
		return NULL;
	}

protected:
	void updateItems(const QPointF& prev, const QPointF& current) {
		if(!mItems)
			return;

		const QPointF diff = QPointF(current - prev);
		const float dist = hypotf(diff.x(), diff.y());

		// Update text and the background. Only display the current pose if the distance is small enough
		if(dist <= 0.1f) {
			mTextItem->setText(QString("     pos: (x: %1 y: %2)").
				arg(QString::number(prev.x(), 'f', 2)).
				arg(QString::number(prev.y(), 'f', 2)));
			mLineItem->setLine(QLineF());
		} else {
			mTextItem->setText(QString("     src: (x: %1 y: %2)\n     dst: (x: %3 y: %4)\n     len: %5m").
				arg(QString::number(prev.x(), 'f', 2)).
				arg(QString::number(prev.y(), 'f', 2)).
				arg(QString::number(current.x(), 'f', 2)).
				arg(QString::number(current.y(), 'f', 2)).
				arg(QString::number(dist, 'f', 4)));
			mLineItem->setLine(QLineF(QPointF(0, 0), -diff));
		}

		// Set the background item so that it has a slight border around the text for contrast and readability
		const QRectF r = mTextItem->boundingRect();
		mTextBackgroundItem->setRect(r.adjusted(r.width() * 0.05, -r.height() * 0.075, r.width() * 0.05, r.height() * 0.15));

		// Position the text and the background next to the mouse
		mItems->setPos(current.x(), current.y());
	}

	boost::optional<QPointF> mMouseDownPos;

	QGraphicsItemGroup* mItems;
	QGraphicsSimpleTextItem* mTextItem;
	QGraphicsRectItem* mTextBackgroundItem;
	QGraphicsLineItem* mLineItem;
};

///////////////////////////////////////////////////////////////////////////////

}

MIRA_CLASS_SERIALIZATION(mira::MeasureTool2D, mira::VisualizationTool2D)
