/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file TitleBar.C
 *    Implements TitleBar.h
 *
 * @author Erik Einhorn
 * @date   2011/09/08
 */

#include <rcp/TitleBar.h>

#include <assert.h>

#include <iostream>

#include <QApplication>
#include <QStyle>
#include <QPainter>
#include <QStylePainter>
#include <QStyleOption>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

/**
 * Reimplements paintEvent to avoid Qt painting its own ugly icon, if a menu
 */
class ToolButtonWithIcon : public QToolButton
{
public:

	explicit ToolButtonWithIcon(QWidget * parent=0) : QToolButton(parent) {}

	void paintEvent(QPaintEvent *)
	{
		QStylePainter p(this);
		QStyleOptionToolButton opt;
		initStyleOption(&opt);
		opt.features = QStyleOptionToolButton::None;
		p.drawComplexControl(QStyle::CC_ToolButton, opt);
	}
};

//////////////////////////////////////////////////////////////////////////////

TitleBar::TitleBar(QWidget *parent) :
	QWidget(parent),
	mBackground(NULL),
	mIsHighlighted(false)
{
	const int s = 14;

	mParent = parentWidget();
	//QStyle* style = QApplication::style();

	mLayout = new QHBoxLayout(this);
	mLayout->setMargin(0);
	mLayout->setSpacing(0);
	this->setFixedHeight(s+4);

	mLeftButtons = new QWidget(this);
	mRightButtons = new QWidget(this);
	mLeftLayout = new QHBoxLayout(mLeftButtons);
	mLeftLayout->setMargin(0);
	mLeftLayout->setSpacing(0);
	mRightLayout = new QHBoxLayout(mRightButtons);
	mRightLayout->setMargin(0);
	mRightLayout->setSpacing(0);

	QMenu* menu = new QMenu("Menu", this);

	mMenuBtn = new ToolButtonWithIcon(mRightButtons);
	mMenuBtn->setMenu(menu);
	mMenuBtn->setPopupMode(QToolButton::InstantPopup);
	mMenuBtn->setIcon(QIcon(QPixmap(":/icons/Menu.png")));
	mMenuBtn->setToolTip(tr("View Menu"));
	mMenuBtn->setFixedSize(s, s);
	mMenuBtn->setAutoRaise(true);
	mMenuBtn->setVisible(false);

	mMinimize = new QToolButton(mRightButtons);
	mMinimize->setIcon(QIcon(QPixmap(":/icons/Minimize.png")));
	mMinimize->setToolTip(tr("Minimize/Shade"));
	mMinimize->setFixedSize(s, s);
	mMinimize->setAutoRaise(true);

	mMaximize = new QToolButton(mRightButtons);
	mMaximize->setIcon(QIcon(QPixmap(":/icons/Maximize.png")));
	mMaximize->setToolTip(tr("Maximize/Tabify"));
	mMaximize->setFixedSize(s, s);
	mMaximize->setAutoRaise(true);

	mRestore = new QToolButton(mRightButtons);
	mRestore->setIcon(QIcon(QPixmap(":/icons/Restore.png")));
	mRestore->setToolTip(tr("Restore"));
	mRestore->setFixedSize(s, s);
	mRestore->setAutoRaise(true);

	mClose = new QToolButton(mRightButtons);
	mClose->setIcon(QIcon(QPixmap(":/icons/Close.png")));
	mClose->setToolTip(tr("Close"));
	mClose->setFixedSize(s, s);
	mClose->setAutoRaise(true);

	mRightLayout->addWidget(mMenuBtn);
	mRightLayout->addWidget(mMinimize);
	mRightLayout->addWidget(mRestore);
	mRightLayout->addWidget(mMaximize);
	mRightLayout->addWidget(mClose);

	mUndock = new QToolButton(mLeftButtons);
	mUndock->setIcon(QIcon(QPixmap(":/icons/Undock.png")));
	mUndock->setToolTip(tr("Floating on Desktop"));
	mUndock->setFixedSize(s, s);
	mUndock->setAutoRaise(true);

	mDock = new QToolButton(mLeftButtons);
	mDock->setIcon(QIcon(QPixmap(":/icons/Dock.png")));
	mDock->setToolTip(tr("Dock to main window"));
	mDock->setFixedSize(s, s);
	mDock->setAutoRaise(true);

	mLeftLayout->addWidget(mDock);
	mLeftLayout->addWidget(mUndock);

	mLayout->addWidget(mLeftButtons);
	mLayout->addStretch();
	mLayout->addWidget(mRightButtons);

	if ( mParent )
		mParent->installEventFilter(this);
}

TitleBar::~TitleBar()
{
	delete mBackground;
}

void TitleBar::setHighlight(bool on)
{
	if(mIsHighlighted != on) {
		mIsHighlighted = on;
		update();
	}
}

void TitleBar::update()
{
	updateBackground();
	repaint();
}

bool TitleBar::eventFilter( QObject* obj, QEvent* event)
{
	if ( obj == parent() && event->type()==QEvent::WindowTitleChange ) {
		updateBackground();
		repaint();
	}
	return QWidget::eventFilter( obj, event );
}

void TitleBar::paintEvent(QPaintEvent* event)
{
	QPainter painter(this);

	painter.drawPixmap(0, 0, *mBackground);
}

void TitleBar::resizeEvent(QResizeEvent* event)
{
	updateBackground();
}

QWidget* TitleBar::takeRightButtons()
{
	mLayout->removeWidget(mRightButtons);
	mRightButtons->setParent(NULL);
	return mRightButtons;
}

void TitleBar::putBackRightButtons()
{
	mRightButtons->setParent(this);
	mLayout->insertWidget(2,mRightButtons);
	mRightButtons->show();
}

QWidget* TitleBar::takeLeftButtons()
{
	mLayout->removeWidget(mLeftButtons);
	mLeftButtons->setParent(NULL);
	return mLeftButtons;
}

/// Puts back the left buttons, that were taken using the above method
void TitleBar::putBackLeftButtons()
{
	mLeftButtons->setParent(this);
	mLayout->insertWidget(0,mLeftButtons);
	mLeftButtons->show();
}

void TitleBar::insertRightButton(QToolButton* button, int index)
{
	if(!button)
		return;
	button->setParent(mRightButtons);
	mRightLayout->insertWidget(index, button);

}

void TitleBar::insertLeftButton(QToolButton* button, int index)
{
	if(!button)
		return;
	button->setParent(mLeftButtons);
	mLeftLayout->insertWidget(index, button);
}

void TitleBar::updateBackground()
{
	if(mBackground)
		delete mBackground;
	mBackground = new QPixmap(size());

	QPainter painter(mBackground);
	QRect rct = rect();

	QLinearGradient grad(QPointF(0, 0), QPointF(0, rct.height()));
	QPalette pal = palette();

	QColor baseColor;
	QColor textColor;
	if(mIsHighlighted) {
		baseColor = pal.color(QPalette::Active, QPalette::Highlight);
		textColor = pal.color(QPalette::Active, QPalette::HighlightedText);
	} else {
		baseColor = pal.color(QPalette::Active, QPalette::Mid);
		textColor = pal.color(QPalette::Active, QPalette::Text);
	}

	grad.setColorAt(0,   baseColor.lighter(120));
	grad.setColorAt(0.4, baseColor);
	grad.setColorAt(0.5, baseColor.darker(120));
	grad.setColorAt(1,   baseColor);

	painter.fillRect(rct, grad);

	int textLeftOffset = 18;
	QFontMetrics fM(painter.font());
	QString text(fM.elidedText(mParent->windowTitle(), Qt::ElideRight,
	                           rct.width() - 65 - textLeftOffset));
	painter.setPen(textColor);
	painter.drawText(rct.left() + 5 + textLeftOffset, rct.top() + 3,
	                 rct.width() - 10 - textLeftOffset,
	                 rct.height() - 6, Qt::AlignVCenter, text);
}

QMenu* TitleBar::getMenu()
{
	mMenuBtn->setVisible(true);
	assert(mMenuBtn->menu());
	return mMenuBtn->menu();
}

//////////////////////////////////////////////////////////////////////////////

}
