/*
 * Copyright (C) 2012 by
 *   MetraLabs GmbH (MLAB), GERMANY
 * and
 *   Neuroinformatics and Cognitive Robotics Labs (NICR) at TU Ilmenau, GERMANY
 * All rights reserved.
 *
 * Contact: info@mira-project.org
 *
 * Commercial Usage:
 *   Licensees holding valid commercial licenses may use this file in
 *   accordance with the commercial license agreement provided with the
 *   software or, alternatively, in accordance with the terms contained in
 *   a written agreement between you and MLAB or NICR.
 *
 * GNU General Public License Usage:
 *   Alternatively, this file may be used under the terms of the GNU
 *   General Public License version 3.0 as published by the Free Software
 *   Foundation and appearing in the file LICENSE.GPL3 included in the
 *   packaging of this file. Please review the following information to
 *   ensure the GNU General Public License version 3.0 requirements will be
 *   met: http://www.gnu.org/copyleft/gpl.html.
 *   Alternatively you may (at your option) use any later version of the GNU
 *   General Public License if such license has been publicly approved by
 *   MLAB and NICR (or its successors, if any).
 *
 * IN NO EVENT SHALL "MLAB" OR "NICR" BE LIABLE TO ANY PARTY FOR DIRECT,
 * INDIRECT, SPECIAL, INCIDENTAL, OR CONSEQUENTIAL DAMAGES ARISING OUT OF
 * THE USE OF THIS SOFTWARE AND ITS DOCUMENTATION, EVEN IF "MLAB" OR
 * "NICR" HAS BEEN ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
 *
 * "MLAB" AND "NICR" SPECIFICALLY DISCLAIM ANY WARRANTIES, INCLUDING,
 * BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND
 * FITNESS FOR A PARTICULAR PURPOSE. THE SOFTWARE PROVIDED HEREUNDER IS
 * ON AN "AS IS" BASIS, AND "MLAB" AND "NICR" HAVE NO OBLIGATION TO
 * PROVIDE MAINTENANCE, SUPPORT, UPDATES, ENHANCEMENTS OR MODIFICATIONS.
 */

/**
 * @file EditorPartWindow.C
 *    Implementation of EditorPartWindow.h.
 *
 * @author Erik Einhorn
 * @date   2011/09/08
 */

#include <rcp/EditorPartWindow.h>

#include <assert.h>

#include <iostream>

#include <QEvent>
#include <QCoreApplication>

#include <QPainter>
#include <QWindowStateChangeEvent>
#include <QAbstractScrollArea>
#include <QScrollBar>

#include <rcp/EditorPartArea.h>
#include <rcp/EditorPartTitleBar.h>

#include <rcp/WidgetResizeHandler.h>

namespace mira {

///////////////////////////////////////////////////////////////////////////////

#define BORDER 3

static const QSize defaultEditorSize(400,300);

EditorPartWindow::EditorPartWindow(EditorPartArea* area, Qt::WindowFlags flags) :
	QFrame(area, flags | Qt::SubWindow | Qt::FramelessWindowHint),
	mArea(area), mWidget(NULL),
	mAutoHideDecorations(true), mDecorationsShown(true),
	mActivated(false), mWindowState(Normal), mDocked(true)//, mMouseOver(false)
{
	setAttribute(Qt::WA_DeleteOnClose, true);
	setFocusPolicy(Qt::StrongFocus);

	setFrameStyle(QFrame::Box);
	setLineWidth(BORDER);
	setAutoFillBackground(true);

	mTitleBar = new EditorPartTitleBar(this);
	mLayout = new QVBoxLayout(this);
	mLayout->setSpacing(0);
	mLayout->setMargin(0);
	mLayout->addWidget(mTitleBar);

	setMinimumHeight(mTitleBar->height());
	setMinimumWidth(32);

	WidgetResizeHandler* resizer = new WidgetResizeHandler(this);
	resizer->setMovingEnabled(true);
	resizer->setActive(true);

	resize(defaultEditorSize);

	// create and add the default content widget
	mEmptyContentWidget = new QWidget(this);
	mLayout->addWidget(mEmptyContentWidget);

	connect(mTitleBar->mMaximize, SIGNAL(clicked()), this, SLOT(showMaximized()));
	connect(mTitleBar->mMinimize, SIGNAL(clicked()), this, SLOT(showMinimized()));
	connect(mTitleBar->mRestore, SIGNAL(clicked()), this, SLOT(showNormal()));
	connect(mTitleBar->mUndock, SIGNAL(clicked()), this, SLOT(onUndock()));
	connect(mTitleBar->mDock, SIGNAL(clicked()), this, SLOT(onDock()));
	connect(mTitleBar->mClose, SIGNAL(clicked()), this, SLOT(close()));

	mTitleBar->mMaximize->show();
	mTitleBar->mMinimize->show();
	mTitleBar->mRestore->hide();
	mTitleBar->mDock->hide();

	if(mArea!=NULL)
		mArea->addSubWindow(this);
	else
		showDecorations(true);

	move(0,0);
}

EditorPartWindow::~EditorPartWindow()
{
	if(mArea)
		mArea->removeSubWindow(this);
}

void EditorPartWindow::setWidget(QWidget* widget)
{
	if(mWidget!=NULL) {
		mLayout->removeWidget(mWidget);
		mWidget->setParent(NULL);
	}

	mWidget = widget;
	if(mWidget!=NULL) {
		mEmptyContentWidget->hide();
		mWidget->setParent(this);
		mLayout->addWidget(mWidget);
	} else {
		mEmptyContentWidget->show();
	}
}

QWidget* EditorPartWindow::widget() const
{
	return mWidget;
}

EditorPartTitleBar* EditorPartWindow::getTitleBar()
{
	return mTitleBar;
}

void EditorPartWindow::showDecorations(bool show)
{
	if(show==mDecorationsShown)
		return;

	mDecorationsShown = show;
	blockSignals(true);
	if(show) {
		QRect r = geometry();
		r.setX(r.x()-BORDER);
		r.setY(r.y()-mTitleBar->height()-BORDER);
		r.setWidth(r.width()+BORDER);
		r.setHeight(r.height()+BORDER);
		mTitleBar->show();
		setGeometry(r);
		setLineWidth(BORDER);
	} else {
		QRect r = geometry();
		mTitleBar->hide();
		r.setX(r.x()+BORDER);
		r.setY(r.y()+mTitleBar->height()+BORDER);
		r.setWidth(r.width()-BORDER);
		r.setHeight(r.height()-BORDER);
		setGeometry(r);
		setLineWidth(0);
	}
	blockSignals(false);
}

void EditorPartWindow::setAutoHideDecorations(bool on)
{
	mAutoHideDecorations = on;
	updateDecorations();
}

void EditorPartWindow::updateDecorations()
{
	if(!mDocked) // always show decorations, if not docked!
		showDecorations(true);
	else {

		if(mWindowState==Maximized) {
			setLineWidth(0);
			mTitleBar->hide();
			mDecorationsShown=false;
		} else {
			setLineWidth(BORDER);
			// otherwise show decorations if we are the active window (if auto hide is enabled)
			// OR if we are minimized ...
			showDecorations(mActivated || !mAutoHideDecorations || isMinimized());
		}
	}
}

void EditorPartWindow::updateStateButtons()
{
	if(mWindowState==Minimized) {
		if(!mDocked)
			mTitleBar->mMaximize->show();
		else
			mTitleBar->mMaximize->hide();
		mTitleBar->mMinimize->hide();
		mTitleBar->mRestore->show();
	} else if(mWindowState==Maximized) {
		mTitleBar->mMaximize->hide();
		if(!mDocked)
			mTitleBar->mMinimize->show();
		else
			mTitleBar->mMinimize->hide();
		mTitleBar->mRestore->show();
	} else {
		mTitleBar->mMinimize->show();
		mTitleBar->mMaximize->show();
		mTitleBar->mRestore->hide();
	}

	if(mDocked) {
		mTitleBar->mDock->hide();
		mTitleBar->mUndock->show();
	} else {
		mTitleBar->mDock->show();
		mTitleBar->mUndock->hide();
	}


}

void EditorPartWindow::paintEvent(QPaintEvent *paintEvent)
{
	QWidget::paintEvent(paintEvent);
}

void EditorPartWindow::changeEvent (QEvent* event)
{
	if(event->type()==QEvent::WindowStateChange) {
		QWindowStateChangeEvent* e = (QWindowStateChangeEvent*)event;
		windowStateChanged(e->oldState(), windowState());
	}
}

void EditorPartWindow::setActive(bool activate)
{
	mActivated = activate;
	mTitleBar->setHighlight(activate);
	updateDecorations();
}

void EditorPartWindow::windowStateChanged(Qt::WindowStates oldState,
                                          Qt::WindowStates newState)
{
	if(newState & Qt::WindowMinimized) {
		mWindowState=Minimized;
		onMinimized();
	} else if(newState & Qt::WindowMaximized) {
		if(mWindowState!=Maximized) {
			mWindowState=Maximized;
			onMaximized();
		}
	} else {
		mWindowState=Normal;
		onRestored();
	}
}

void EditorPartWindow::onMinimized()
{
	// do manual minimizing if we are docked
	if(mDocked) {
		mRestoreGeometry = geometry();
		if(mWidget)
			mWidget->hide();
		QRect mingeometry = mRestoreGeometry;
		mingeometry.setWidth(160);
		mingeometry.setHeight(mTitleBar->height()+5);
		setGeometry(mingeometry);
	}

	updateStateButtons();
}

void EditorPartWindow::onMaximized()
{
	// do manual maximizing if we are docked
	if(mDocked)
		mRestoreGeometry = geometry();

	if(mArea)
		mArea->maximizedSubwindow(this);

	updateStateButtons();
}

void EditorPartWindow::onRestored()
{
	if(mDocked) {
		// make sure to enable decorations, before restoring the geometry
		showDecorations(true);

		if(mWidget)
			mWidget->show();

		if(mRestoreGeometry.isValid())
			setGeometry(mRestoreGeometry);
		else
			setGeometry(0,0, defaultEditorSize.width(), defaultEditorSize.height());
	}

	updateStateButtons();

	if(mArea)
		mArea->restoredSubwindow(this);
}

void EditorPartWindow::onUndock()
{
	if(!mDocked)
		return;

	// save geometry
	mRestoreGeometry = geometry();

	if(mArea)
		mArea->undockSubwindow(this);

	mDocked = false;
	updateStateButtons();
	updateDecorations();
}

void EditorPartWindow::onDock()
{
	if(mDocked)
		return; // already docked

	if(mArea)
		mArea->dockSubwindow(this);

	setActive(true);

	mDocked = true;

	if(isMaximized())
		mTitleBar->mMinimize->hide();

	updateStateButtons();
	updateDecorations();
}

///////////////////////////////////////////////////////////////////////////////

}
